/*
 * Decompiled with CFR 0.152.
 */
package io.dropwizard.servlets.assets;

import com.google.common.base.CharMatcher;
import com.google.common.base.Joiner;
import com.google.common.base.Preconditions;
import com.google.common.base.Splitter;
import com.google.common.collect.ImmutableList;
import com.google.common.hash.Hashing;
import com.google.common.io.Resources;
import com.google.common.net.MediaType;
import io.dropwizard.servlets.assets.ByteRange;
import io.dropwizard.servlets.assets.ResourceURL;
import java.io.IOException;
import java.net.URISyntaxException;
import java.net.URL;
import java.nio.charset.Charset;
import java.util.List;
import javax.servlet.ServletException;
import javax.servlet.ServletOutputStream;
import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

public class AssetServlet
extends HttpServlet {
    private static final long serialVersionUID = 6393345594784987908L;
    private static final CharMatcher SLASHES = CharMatcher.is('/');
    private static final MediaType DEFAULT_MEDIA_TYPE = MediaType.HTML_UTF_8;
    private final String resourcePath;
    private final String uriPath;
    private final String indexFile;
    private final Charset defaultCharset;

    public AssetServlet(String resourcePath, String uriPath, String indexFile, Charset defaultCharset) {
        String trimmedPath = SLASHES.trimFrom(resourcePath);
        this.resourcePath = trimmedPath.isEmpty() ? trimmedPath : trimmedPath + '/';
        String trimmedUri = SLASHES.trimTrailingFrom(uriPath);
        this.uriPath = trimmedUri.isEmpty() ? "/" : trimmedUri;
        this.indexFile = indexFile;
        this.defaultCharset = defaultCharset;
    }

    public URL getResourceURL() {
        return Resources.getResource(this.resourcePath);
    }

    public String getUriPath() {
        return this.uriPath;
    }

    public String getIndexFile() {
        return this.indexFile;
    }

    @Override
    protected void doGet(HttpServletRequest req, HttpServletResponse resp) throws ServletException, IOException {
        try {
            String ifRange;
            CachedAsset cachedAsset;
            StringBuilder builder = new StringBuilder(req.getServletPath());
            if (req.getPathInfo() != null) {
                builder.append(req.getPathInfo());
            }
            if ((cachedAsset = this.loadAsset(builder.toString())) == null) {
                resp.sendError(404);
                return;
            }
            if (this.isCachedClientSide(req, cachedAsset)) {
                resp.sendError(304);
                return;
            }
            String rangeHeader = req.getHeader("Range");
            int resourceLength = cachedAsset.getResource().length;
            ImmutableList<Object> ranges = ImmutableList.of();
            boolean usingRanges = false;
            if (rangeHeader != null && ((ifRange = req.getHeader("If-Range")) == null || cachedAsset.getETag().equals(ifRange))) {
                try {
                    ranges = this.parseRangeHeader(rangeHeader, resourceLength);
                }
                catch (NumberFormatException e) {
                    resp.sendError(416);
                    return;
                }
                if (ranges.isEmpty()) {
                    resp.sendError(416);
                    return;
                }
                resp.setStatus(206);
                usingRanges = true;
                resp.addHeader("Content-Range", "bytes " + Joiner.on(",").join(ranges) + "/" + resourceLength);
            }
            resp.setDateHeader("Last-Modified", cachedAsset.getLastModifiedTime());
            resp.setHeader("ETag", cachedAsset.getETag());
            String mimeTypeOfExtension = req.getServletContext().getMimeType(req.getRequestURI());
            MediaType mediaType = DEFAULT_MEDIA_TYPE;
            if (mimeTypeOfExtension != null) {
                try {
                    mediaType = MediaType.parse(mimeTypeOfExtension);
                    if (this.defaultCharset != null && mediaType.is(MediaType.ANY_TEXT_TYPE)) {
                        mediaType = mediaType.withCharset(this.defaultCharset);
                    }
                }
                catch (IllegalArgumentException illegalArgumentException) {
                    // empty catch block
                }
            }
            if (mediaType.is(MediaType.ANY_VIDEO_TYPE) || mediaType.is(MediaType.ANY_AUDIO_TYPE) || usingRanges) {
                resp.addHeader("Accept-Ranges", "bytes");
            }
            resp.setContentType(mediaType.type() + '/' + mediaType.subtype());
            if (mediaType.charset().isPresent()) {
                resp.setCharacterEncoding(mediaType.charset().get().toString());
            }
            try (ServletOutputStream output = resp.getOutputStream();){
                if (usingRanges) {
                    for (ByteRange byteRange : ranges) {
                        output.write(cachedAsset.getResource(), byteRange.getStart(), byteRange.getEnd() - byteRange.getStart() + 1);
                    }
                } else {
                    output.write(cachedAsset.getResource());
                }
            }
        }
        catch (RuntimeException | URISyntaxException ignored) {
            resp.sendError(404);
        }
    }

    private CachedAsset loadAsset(String key) throws URISyntaxException, IOException {
        long lastModified;
        Preconditions.checkArgument(key.startsWith(this.uriPath));
        String requestedResourcePath = SLASHES.trimFrom(key.substring(this.uriPath.length()));
        String absoluteRequestedResourcePath = SLASHES.trimFrom(this.resourcePath + requestedResourcePath);
        URL requestedResourceURL = this.getResourceUrl(absoluteRequestedResourcePath);
        if (ResourceURL.isDirectory(requestedResourceURL)) {
            if (this.indexFile != null) {
                requestedResourceURL = this.getResourceUrl(absoluteRequestedResourcePath + '/' + this.indexFile);
            } else {
                return null;
            }
        }
        if ((lastModified = ResourceURL.getLastModified(requestedResourceURL)) < 1L) {
            lastModified = System.currentTimeMillis();
        }
        lastModified = lastModified / 1000L * 1000L;
        return new CachedAsset(this.readResource(requestedResourceURL), lastModified);
    }

    protected URL getResourceUrl(String absoluteRequestedResourcePath) {
        return Resources.getResource(absoluteRequestedResourcePath);
    }

    protected byte[] readResource(URL requestedResourceURL) throws IOException {
        return Resources.toByteArray(requestedResourceURL);
    }

    private boolean isCachedClientSide(HttpServletRequest req, CachedAsset cachedAsset) {
        return cachedAsset.getETag().equals(req.getHeader("If-None-Match")) || req.getDateHeader("If-Modified-Since") >= cachedAsset.getLastModifiedTime();
    }

    private ImmutableList<ByteRange> parseRangeHeader(String rangeHeader, int resourceLength) {
        String[] parts;
        ImmutableList.Builder builder = ImmutableList.builder();
        if (rangeHeader.indexOf("=") != -1 && (parts = rangeHeader.split("=")).length > 1) {
            List<String> ranges = Splitter.on(",").trimResults().splitToList(parts[1]);
            for (String range : ranges) {
                builder.add(ByteRange.parse(range, resourceLength));
            }
        }
        return builder.build();
    }

    private static class CachedAsset {
        private final byte[] resource;
        private final String eTag;
        private final long lastModifiedTime;

        private CachedAsset(byte[] resource, long lastModifiedTime) {
            this.resource = resource;
            this.eTag = '\"' + Hashing.murmur3_128().hashBytes(resource).toString() + '\"';
            this.lastModifiedTime = lastModifiedTime;
        }

        public byte[] getResource() {
            return this.resource;
        }

        public String getETag() {
            return this.eTag;
        }

        public long getLastModifiedTime() {
            return this.lastModifiedTime;
        }
    }
}

