/*
 * Decompiled with CFR 0.152.
 */
package org.gluu.oxauth.fido2.service.verifier;

import com.fasterxml.jackson.databind.JsonNode;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.nio.ByteBuffer;
import java.security.InvalidAlgorithmParameterException;
import java.security.InvalidKeyException;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.security.Provider;
import java.security.PublicKey;
import java.security.Signature;
import java.security.SignatureException;
import java.security.cert.Certificate;
import java.security.spec.MGF1ParameterSpec;
import java.security.spec.PSSParameterSpec;
import java.util.Arrays;
import javax.enterprise.context.ApplicationScoped;
import javax.enterprise.inject.Instance;
import javax.inject.Inject;
import org.apache.commons.codec.binary.Hex;
import org.apache.commons.codec.digest.DigestUtils;
import org.apache.commons.lang.StringUtils;
import org.bouncycastle.jce.provider.BouncyCastleProvider;
import org.gluu.oxauth.fido2.ctap.AttestationConveyancePreference;
import org.gluu.oxauth.fido2.ctap.UserVerification;
import org.gluu.oxauth.fido2.exception.Fido2RPRuntimeException;
import org.gluu.oxauth.fido2.model.auth.AuthData;
import org.gluu.oxauth.fido2.service.Base64Service;
import org.gluu.oxauth.fido2.service.DataMapperService;
import org.gluu.oxauth.fido2.service.processors.AttestationFormatProcessor;
import org.slf4j.Logger;
import org.xdi.oxauth.model.util.SecurityProviderUtility;

@ApplicationScoped
public class CommonVerifiers {
    private static final int FLAG_USER_PRESENT = 1;
    private static final int FLAG_ATTESTED_CREDENTIAL_DATA_INCLUDED = 64;
    private static final int FLAG_USER_VERIFIED = 4;
    private static final int FLAG_EXTENSION_DATA_INCLUDED = 128;
    @Inject
    private Logger log;
    @Inject
    private Base64Service base64Service;
    @Inject
    private DataMapperService dataMapperService;
    @Inject
    private Instance<AttestationFormatProcessor> supportedAttestationFormats;

    public void verifyU2FAttestationSignature(AuthData authData, byte[] clientDataHash, String signature, Certificate certificate, int signatureAlgorithm) {
        int bufferSize = 0;
        byte[] reserved = new byte[]{0};
        bufferSize += reserved.length;
        byte[] rpIdHash = authData.getRpIdHash();
        bufferSize += rpIdHash.length;
        bufferSize += clientDataHash.length;
        byte[] credId = authData.getCredId();
        bufferSize += credId.length;
        byte[] publicKey = this.convertCOSEtoPublicKey(authData.getCOSEPublicKey());
        byte[] signatureBase = ByteBuffer.allocate(bufferSize += publicKey.length).put(reserved).put(rpIdHash).put(clientDataHash).put(credId).put(publicKey).array();
        byte[] signatureBytes = this.base64Service.decode(signature.getBytes());
        this.log.info("Signature {}", (Object)Hex.encodeHexString((byte[])signatureBytes));
        this.log.info("Signature Base {}", (Object)Hex.encodeHexString((byte[])signatureBase));
        this.verifySignature(signatureBytes, signatureBase, certificate, signatureAlgorithm);
    }

    void verifyPackedAttestationSignature(AuthData authData, byte[] clientDataHash, String signature, Certificate certificate, int signatureAlgorithm) {
        int bufferSize = 0;
        byte[] rpIdHashBuffer = authData.getRpIdHash();
        bufferSize += rpIdHashBuffer.length;
        byte[] flagsBuffer = authData.getFlags();
        bufferSize += flagsBuffer.length;
        byte[] countersBuffer = authData.getCounters();
        bufferSize += countersBuffer.length;
        byte[] signatureBase = ByteBuffer.allocate(bufferSize += clientDataHash.length).put(rpIdHashBuffer).put(flagsBuffer).put(countersBuffer).put(clientDataHash).array();
        byte[] signatureBytes = this.base64Service.decode(signature.getBytes());
        this.log.info("Signature {}", (Object)Hex.encodeHexString((byte[])signatureBytes));
        this.log.info("Signature Base {}", (Object)Hex.encodeHexString((byte[])signatureBase));
        this.log.info("Signature BaseLen {}", (Object)signatureBase.length);
        this.verifySignature(signatureBytes, signatureBase, certificate, signatureAlgorithm);
    }

    public void verifyPackedAttestationSignature(byte[] authData, byte[] clientDataHash, String signature, PublicKey key, int signatureAlgorithm) {
        int bufferSize = 0;
        bufferSize += authData.length;
        byte[] signatureBase = ByteBuffer.allocate(bufferSize += clientDataHash.length).put(authData).put(clientDataHash).array();
        byte[] signatureBytes = this.base64Service.decode(signature.getBytes());
        this.log.info("Signature {}", (Object)Hex.encodeHexString((byte[])signatureBytes));
        this.log.info("Signature Base {}", (Object)Hex.encodeHexString((byte[])signatureBase));
        this.log.info("Signature BaseLen {}", (Object)signatureBase.length);
        this.verifySignature(signatureBytes, signatureBase, key, signatureAlgorithm);
    }

    public void verifyPackedAttestationSignature(byte[] authData, byte[] clientDataHash, String signature, Certificate certificate, int signatureAlgorithm) {
        this.verifyPackedAttestationSignature(authData, clientDataHash, signature, certificate.getPublicKey(), signatureAlgorithm);
    }

    public void verifyAssertionSignature(AuthData authData, byte[] clientDataHash, String signature, PublicKey publicKey, int signatureAlgorithm) {
        int bufferSize = 0;
        byte[] rpIdHash = authData.getRpIdHash();
        bufferSize += rpIdHash.length;
        byte[] flags = authData.getFlags();
        bufferSize += flags.length;
        byte[] counters = authData.getCounters();
        bufferSize += counters.length;
        this.log.info("Client data hash HEX {}", (Object)Hex.encodeHexString((byte[])clientDataHash));
        byte[] signatureBase = ByteBuffer.allocate(bufferSize += clientDataHash.length).put(rpIdHash).put(flags).put(counters).put(clientDataHash).array();
        byte[] signatureBytes = this.base64Service.urlDecode(signature.getBytes());
        this.log.info("Signature {}", (Object)Hex.encodeHexString((byte[])signatureBytes));
        this.log.info("Signature Base {}", (Object)Hex.encodeHexString((byte[])signatureBase));
        this.verifySignature(signatureBytes, signatureBase, publicKey, signatureAlgorithm);
    }

    public boolean verifyUserPresent(AuthData authData) {
        if ((authData.getFlags()[0] & 1) == 1) {
            return true;
        }
        throw new Fido2RPRuntimeException("User not present");
    }

    public boolean verifyUserVerified(AuthData authData) {
        if ((authData.getFlags()[0] & 4) == 1) {
            return true;
        }
        throw new Fido2RPRuntimeException("User not verified");
    }

    public void verifyRpIdHash(AuthData authData, String domain) {
        try {
            byte[] retrievedRpIdHash = authData.getRpIdHash();
            byte[] calculatedRpIdHash = DigestUtils.getSha256Digest().digest(domain.getBytes("UTF-8"));
            this.log.debug("rpIDHash from Domain    HEX {}", (Object)Hex.encodeHexString((byte[])calculatedRpIdHash));
            this.log.debug("rpIDHash from Assertion HEX {}", (Object)Hex.encodeHexString((byte[])retrievedRpIdHash));
            if (!Arrays.equals(retrievedRpIdHash, calculatedRpIdHash)) {
                this.log.warn("hash from domain doesn't match hash from assertion HEX ");
                throw new Fido2RPRuntimeException("Hashes don't match");
            }
        }
        catch (UnsupportedEncodingException e) {
            throw new Fido2RPRuntimeException("This encoding is not supported");
        }
    }

    public void verifyCounter(int oldCounter, int newCounter) {
        this.log.info("old counter {} new counter {} ", (Object)oldCounter, (Object)newCounter);
        if (newCounter == 0 && oldCounter == 0) {
            return;
        }
        if (newCounter <= oldCounter) {
            throw new Fido2RPRuntimeException("Counter did not increase");
        }
    }

    private byte[] convertCOSEtoPublicKey(byte[] cosePublicKey) {
        try {
            JsonNode cborPublicKey = this.dataMapperService.cborReadTree(cosePublicKey);
            byte[] x = this.base64Service.decode(cborPublicKey.get("-2").asText());
            byte[] y = this.base64Service.decode(cborPublicKey.get("-3").asText());
            byte[] keyBytes = ByteBuffer.allocate(1 + x.length + y.length).put((byte)4).put(x).put(y).array();
            this.log.info("KeyBytes HEX {}", (Object)Hex.encodeHexString((byte[])keyBytes));
            return keyBytes;
        }
        catch (IOException e) {
            throw new Fido2RPRuntimeException("Can't parse public key");
        }
    }

    public void verifySignature(byte[] signature, byte[] signatureBase, Certificate certificate, int signatureAlgorithm) {
        this.verifySignature(signature, signatureBase, certificate.getPublicKey(), signatureAlgorithm);
    }

    private void verifySignature(byte[] signature, byte[] signatureBase, PublicKey publicKey, int signatureAlgorithm) {
        try {
            Signature signatureChecker = this.getSignatureChecker(signatureAlgorithm);
            signatureChecker.initVerify(publicKey);
            signatureChecker.update(signatureBase);
            if (!signatureChecker.verify(signature)) {
                throw new Fido2RPRuntimeException("Unable to verify signature");
            }
        }
        catch (IllegalArgumentException | InvalidKeyException | SignatureException e) {
            this.log.error("Can't verify the signature ", (Throwable)e);
            throw new Fido2RPRuntimeException("Can't verify the signature");
        }
    }

    private Signature getSignatureChecker(int signatureAlgorithm) {
        BouncyCastleProvider provider = SecurityProviderUtility.getInstance();
        try {
            switch (signatureAlgorithm) {
                case -7: {
                    Signature signatureChecker = Signature.getInstance("SHA256withECDSA", (Provider)provider);
                    return signatureChecker;
                }
                case -35: {
                    Signature signatureChecker = Signature.getInstance("SHA384withECDSA", (Provider)provider);
                    return signatureChecker;
                }
                case -36: {
                    Signature signatureChecker = Signature.getInstance("SHA512withECDSA", (Provider)provider);
                    return signatureChecker;
                }
                case -37: {
                    Signature signatureChecker = Signature.getInstance("SHA256withRSA/PSS", (Provider)provider);
                    signatureChecker.setParameter(new PSSParameterSpec("SHA-256", "MGF1", new MGF1ParameterSpec("SHA-256"), 32, 1));
                    return signatureChecker;
                }
                case -38: {
                    Signature signatureChecker = Signature.getInstance("SHA384withRSA/PSS", (Provider)provider);
                    signatureChecker.setParameter(new PSSParameterSpec("SHA-384", "MGF1", new MGF1ParameterSpec("SHA-384"), 32, 1));
                    return signatureChecker;
                }
                case -39: {
                    Signature signatureChecker = Signature.getInstance("SHA512withRSA/PSS", (Provider)provider);
                    signatureChecker.setParameter(new PSSParameterSpec("SHA-512", "MGF1", new MGF1ParameterSpec("SHA-512"), 32, 1));
                    return signatureChecker;
                }
                case -257: {
                    Signature signatureChecker = Signature.getInstance("SHA256withRSA");
                    return signatureChecker;
                }
                case -258: {
                    Signature signatureChecker = Signature.getInstance("SHA384withRSA", (Provider)provider);
                    return signatureChecker;
                }
                case -259: {
                    Signature signatureChecker = Signature.getInstance("SHA512withRSA", (Provider)provider);
                    return signatureChecker;
                }
                case -65535: {
                    Signature signatureChecker = Signature.getInstance("SHA1withRSA", (Provider)provider);
                    return signatureChecker;
                }
            }
            throw new Fido2RPRuntimeException("Unknown mapping");
        }
        catch (InvalidAlgorithmParameterException | NoSuchAlgorithmException e) {
            throw new Fido2RPRuntimeException("Problem with crypto");
        }
    }

    public MessageDigest getDigest(int signatureAlgorithm) {
        switch (signatureAlgorithm) {
            case -257: {
                return DigestUtils.getSha256Digest();
            }
            case -65535: {
                return DigestUtils.getSha1Digest();
            }
        }
        throw new Fido2RPRuntimeException("Unknown mapping ");
    }

    public void verifyOptions(JsonNode params) {
        long count = Arrays.asList(params.hasNonNull("username")).parallelStream().filter(f -> f == false).count();
        if (count != 0L) {
            throw new Fido2RPRuntimeException("Invalid parameters");
        }
    }

    public void verifyBasicPayload(JsonNode params) {
        long count = Arrays.asList(params.hasNonNull("response"), params.hasNonNull("type")).parallelStream().filter(f -> f == false).count();
        if (count != 0L) {
            throw new Fido2RPRuntimeException("Invalid parameters");
        }
    }

    public String verifyBase64UrlString(JsonNode node, String fieldName) {
        String value = this.verifyThatString(node, fieldName);
        try {
            this.base64Service.urlDecode(value);
        }
        catch (IllegalArgumentException e) {
            throw new Fido2RPRuntimeException("Invalid \"" + fieldName + "\"");
        }
        return value;
    }

    public String verifyThatString(JsonNode node) {
        if (!node.isTextual()) {
            if (node.fieldNames().hasNext()) {
                throw new Fido2RPRuntimeException("Invalid field " + (String)node.fieldNames().next());
            }
            throw new Fido2RPRuntimeException("Field hasn't sub fields");
        }
        return node.asText();
    }

    public String verifyThatString(JsonNode node, String fieldName) {
        JsonNode fieldNode = node.get(fieldName);
        if (fieldNode == null) {
            throw new Fido2RPRuntimeException("Invalid \"" + fieldName + "\"");
        }
        return this.verifyThatString(fieldNode);
    }

    public String verifyThatNonEmptyString(JsonNode node) {
        String value = this.verifyThatString(node);
        if (StringUtils.isEmpty((String)value)) {
            throw new Fido2RPRuntimeException("Invalid field " + node);
        }
        return value;
    }

    public String verifyAuthData(JsonNode node) {
        if (node.isNull()) {
            throw new Fido2RPRuntimeException("Empty auth data");
        }
        String data = this.verifyThatBinary(node);
        if (data.isEmpty()) {
            throw new Fido2RPRuntimeException("Invalid field " + node);
        }
        return data;
    }

    public String verifyThatBinary(JsonNode node) {
        if (!node.isBinary()) {
            throw new Fido2RPRuntimeException("Invalid field " + node);
        }
        return node.asText();
    }

    public void verifyCounter(int counter) {
        if (counter < 0) {
            throw new Fido2RPRuntimeException("Invalid field : counter");
        }
    }

    public boolean verifyAtFlag(byte[] flags) {
        return (flags[0] & 0x40) == 64;
    }

    public void verifyAttestationBuffer(boolean hasAtFlag, byte[] attestationBuffer) {
        if (!hasAtFlag && attestationBuffer.length > 0) {
            throw new Fido2RPRuntimeException("Invalid attestation data buffer");
        }
        if (hasAtFlag && attestationBuffer.length == 0) {
            throw new Fido2RPRuntimeException("Invalid attestation data buffer");
        }
    }

    public void verifyNoLeftovers(byte[] leftovers) {
        if (leftovers.length > 0) {
            throw new Fido2RPRuntimeException("Invalid attestation data buffer: leftovers");
        }
    }

    public int verifyAlgorithm(JsonNode alg, int registeredAlgorithmType) {
        if (alg.isNull()) {
            throw new Fido2RPRuntimeException("Wrong algorithm");
        }
        int algorithmType = Integer.parseInt(alg.asText());
        if (algorithmType != registeredAlgorithmType) {
            throw new Fido2RPRuntimeException("Wrong algorithm");
        }
        return algorithmType;
    }

    public String verifyBase64String(JsonNode node) {
        if (node.isNull()) {
            throw new Fido2RPRuntimeException("Invalid data");
        }
        String value = this.verifyThatBinary(node);
        if (value.isEmpty()) {
            throw new Fido2RPRuntimeException("Invalid data");
        }
        try {
            this.base64Service.decode(value.getBytes("UTF-8"));
        }
        catch (UnsupportedEncodingException e) {
            throw new Fido2RPRuntimeException("Invalid data");
        }
        catch (IllegalArgumentException e) {
            throw new Fido2RPRuntimeException("Invalid data");
        }
        return value;
    }

    public void verifyPackedSurrogateAttestationSignature(byte[] authData, byte[] clientDataHash, String signature, PublicKey publicKey, int signatureAlgorithm) {
        int bufferSize = 0;
        bufferSize += authData.length;
        byte[] signatureBase = ByteBuffer.allocate(bufferSize += clientDataHash.length).put(authData).put(clientDataHash).array();
        byte[] signatureBytes = this.base64Service.decode(signature.getBytes());
        this.log.info("Signature {}", (Object)Hex.encodeHexString((byte[])signatureBytes));
        this.log.info("Signature Base {}", (Object)Hex.encodeHexString((byte[])signatureBase));
        this.log.info("Signature BaseLen {}", (Object)signatureBase.length);
        this.verifySignature(signatureBytes, signatureBase, publicKey, signatureAlgorithm);
    }

    public String verifyFmt(JsonNode fmtNode, String fieldName) {
        String fmt = this.verifyThatString(fmtNode, fieldName);
        this.supportedAttestationFormats.stream().filter(f -> f.getAttestationFormat().getFmt().equals(fmt)).findAny().orElseThrow(() -> new Fido2RPRuntimeException("Unsupported attestation format " + fmt));
        return fmt;
    }

    public void verifyAAGUIDZeroed(AuthData authData) {
        byte[] buf = authData.getAaguid();
        for (int i = 0; i < buf.length; ++i) {
            if (buf[i] == 0) continue;
            throw new Fido2RPRuntimeException("Invalid AAGUID");
        }
    }

    public void verifyTPMVersion(JsonNode ver) {
        if (!"2.0".equals(ver.asText())) {
            throw new Fido2RPRuntimeException("Invalid TPM Attestation version");
        }
    }

    public AttestationConveyancePreference verifyAttestationConveyanceType(JsonNode params) {
        if (params.has("attestation")) {
            String type = this.verifyThatString(params.get("attestation"));
            return AttestationConveyancePreference.valueOf(type);
        }
        return AttestationConveyancePreference.direct;
    }

    public void verifyClientJSONTypeIsGet(JsonNode clientJsonNode) {
        this.verifyClientJSONType(clientJsonNode, "webauthn.get");
    }

    void verifyClientJSONType(JsonNode clientJsonNode, String type) {
        if (clientJsonNode.has("type") && !type.equals(clientJsonNode.get("type").asText())) {
            throw new Fido2RPRuntimeException("Invalid client json parameters");
        }
    }

    public void verifyClientJSONTypeIsCreate(JsonNode clientJsonNode) {
        this.verifyClientJSONType(clientJsonNode, "webauthn.create");
    }

    public void verifyClientJSON(JsonNode clientJsonNode) {
        String origin;
        long count = Arrays.asList(clientJsonNode.hasNonNull("challenge"), clientJsonNode.hasNonNull("origin"), clientJsonNode.hasNonNull("type")).parallelStream().filter(f -> f == false).count();
        if (count != 0L) {
            throw new Fido2RPRuntimeException("Invalid client json parameters");
        }
        this.verifyBase64UrlString(clientJsonNode, "challenge");
        if (clientJsonNode.hasNonNull("tokenBinding")) {
            this.verifyThatString(clientJsonNode.get("tokenBinding"));
        }
        if ((origin = this.verifyThatString(clientJsonNode.get("origin"))).isEmpty()) {
            throw new Fido2RPRuntimeException("Invalid client json parameters");
        }
    }

    public String verifyUserVerification(JsonNode userVerification) {
        try {
            return UserVerification.valueOf(userVerification.asText()).name();
        }
        catch (Exception e) {
            throw new Fido2RPRuntimeException("Wrong user verification parameter " + e.getMessage());
        }
    }

    public void verifyAttestationSignature(AuthData authData, byte[] clientDataHash, String signature, Certificate certificate, int signatureAlgorithm) {
        int bufferSize = 0;
        byte[] authDataBuffer = authData.getAttestationBuffer();
        bufferSize += authDataBuffer.length;
        byte[] signatureBase = ByteBuffer.allocate(bufferSize += clientDataHash.length).put(authDataBuffer).put(clientDataHash).array();
        byte[] signatureBytes = this.base64Service.decode(signature.getBytes());
        this.log.info("Signature {}", (Object)Hex.encodeHexString((byte[])signatureBytes));
        this.log.info("Signature Base {}", (Object)Hex.encodeHexString((byte[])signatureBase));
        this.verifySignature(signatureBytes, signatureBase, certificate, signatureAlgorithm);
    }

    public String verifyAssertionType(JsonNode typeNode) {
        String type = this.verifyThatString(typeNode);
        if (!"public-key".equals(type)) {
            throw new Fido2RPRuntimeException("Invalid type");
        }
        return type;
    }

    public void verifyRequiredUserPresent(AuthData authData) {
        this.log.info("required user present {}", (Object)Hex.encodeHexString((byte[])authData.getFlags()));
        byte flags = authData.getFlags()[0];
        if (!this.isUserPresent(flags) && !this.hasUserVerified(flags)) {
            throw new Fido2RPRuntimeException("User required not present");
        }
    }

    public void verifyPreferredUserPresent(AuthData authData) {
        this.log.info("preferred user present {}", (Object)Hex.encodeHexString((byte[])authData.getFlags()));
        byte flags = authData.getFlags()[0];
        if (this.isUserPresent(flags) && this.hasUserVerified(flags)) {
            throw new Fido2RPRuntimeException("User required not present");
        }
    }

    public void verifyDiscouragedUserPresent(AuthData authData) {
        this.log.info("discouraged user present {}", (Object)Hex.encodeHexString((byte[])authData.getFlags()));
        byte flags = authData.getFlags()[0];
        if (this.hasUserVerified(flags) && this.isUserPresent(flags)) {
            throw new Fido2RPRuntimeException("User discouraged is present present");
        }
    }

    private boolean hasUserVerified(byte flags) {
        boolean uv = (flags & 4) != 0;
        this.log.info("UV = {}", (Object)uv);
        return uv;
    }

    private boolean isUserPresent(byte flags) {
        boolean up = (flags & 1) != 0;
        this.log.info("UP = {}", (Object)up);
        return up;
    }

    public void verifyThatMetadataIsValid(JsonNode metadata) {
        long count = Arrays.asList(metadata.hasNonNull("aaguid"), metadata.hasNonNull("assertionScheme"), metadata.hasNonNull("attestationTypes"), metadata.hasNonNull("description")).parallelStream().filter(f -> f == false).count();
        if (count != 0L) {
            throw new Fido2RPRuntimeException("Invalid parameters in metadata");
        }
    }
}

