package org.gluu.oxtrust.ws.rs;

import java.net.URI;
import java.util.ArrayList;
import java.util.List;

import javax.servlet.http.HttpServletRequest;
import javax.ws.rs.PathParam;
import javax.ws.rs.QueryParam;
import javax.ws.rs.core.Context;
import javax.ws.rs.core.Response;

import org.gluu.oxtrust.ldap.service.GroupService;
import org.gluu.oxtrust.ldap.service.OrganizationService;
import org.gluu.oxtrust.ldap.service.SecurityService;
import org.gluu.oxtrust.model.GluuCustomPerson;
import org.gluu.oxtrust.model.GluuGroup;
import org.gluu.oxtrust.model.GluuGroupList;
import org.gluu.oxtrust.model.GluuOrganization;
import org.gluu.oxtrust.model.GluuUserRole;
import org.gluu.oxtrust.model.scim.Error;
import org.gluu.oxtrust.model.scim.Errors;
import org.gluu.oxtrust.model.scim.ScimGroup;
import org.gluu.oxtrust.util.OxTrustConstants;
import org.gluu.oxtrust.util.CopyUtils;
import org.gluu.oxtrust.util.Utils;
import org.gluu.site.ldap.persistence.exception.EntryPersistenceException;
import org.jboss.seam.annotations.In;
import org.jboss.seam.annotations.Logger;
import org.jboss.seam.annotations.Name;
import org.jboss.seam.contexts.Contexts;
import org.jboss.seam.log.Log;
import org.xdi.ldap.model.GluuStatus;

/**
 * @author Reda Zerrad Date: 04.13.2012
 */
@Name("GroupWebService")
public class GroupWebServiceImpl implements GroupWebService {

	@Logger
	private Log log;

	@In
	private GroupService groupService;

	@Override
	public Response listGroups(@Context HttpServletRequest request,
			@QueryParam(OxTrustConstants.QUERY_PARAMETER_FILTER) final String filterString,
			@QueryParam(OxTrustConstants.QUERY_PARAMETER_SORT_BY) final String sortBy,
			@QueryParam(OxTrustConstants.QUERY_PARAMETER_SORT_ORDER) final String sortOrder) throws Exception {

		groupService = GroupService.instance();

		boolean authorized = getAuthorizedUser();

		if (!authorized) {
			return getErrorResponse("User isn't authorized", Response.Status.FORBIDDEN.getStatusCode());
		}
		try {
			List<GluuGroup> groupList = groupService.getAllGroupsList();
			GluuGroupList allGroupList = new GluuGroupList();
			if (groupList != null) {
				for (GluuGroup gluuGroup : groupList) {
					ScimGroup group = CopyUtils.copy(gluuGroup, null);
					allGroupList.getResources().add(group);
				}

			}
			List<String> schema = new ArrayList<String>();
			schema.add("urn:scim:schemas:core:1.0");
			allGroupList.setSchemas(schema);
			List<ScimGroup> resources = allGroupList.getResources();
			allGroupList.setTotalResults((long) resources.size());

			URI location = new URI("/Groups/");
			return Response.ok(allGroupList).location(location).build();
		} catch (Exception ex) {
			log.error("Exception: ", ex);
			return getErrorResponse("Unexpected processing error, please check the input parameters",
					Response.Status.INTERNAL_SERVER_ERROR.getStatusCode());
		}
	}

	@Override
	public Response getGroupById(@Context HttpServletRequest request, @PathParam("id") String id) throws Exception {

		groupService = GroupService.instance();

		boolean authorized = getAuthorizedUser();

		if (!authorized) {
			return getErrorResponse("User isn't authorized", Response.Status.FORBIDDEN.getStatusCode());
		}
		try {
			GluuGroup gluuGroup = groupService.getGroupByInum(id);
			if (gluuGroup == null) {
				// sets HTTP status code 404 Not Found
				return getErrorResponse("Resource " + id + " not found", Response.Status.NOT_FOUND.getStatusCode());
			}

			ScimGroup group = CopyUtils.copy(gluuGroup, null);

			URI location = new URI("/Groups/" + id);

			return Response.ok(group).location(location).build();
		} catch (EntryPersistenceException ex) {
			log.error("Exception: ", ex);
			return getErrorResponse("Resource " + id + " not found", Response.Status.NOT_FOUND.getStatusCode());
		} catch (Exception ex) {
			log.error("Exception: ", ex);
			return getErrorResponse("Unexpected processing error, please check the input parameters",
					Response.Status.INTERNAL_SERVER_ERROR.getStatusCode());
		}

	}

	@Override
	public Response createGroup(@Context HttpServletRequest request, ScimGroup group) throws Exception {
		groupService = GroupService.instance();
		boolean authorized = getAuthorizedUser();

		if (!authorized) {
			return getErrorResponse("User isn't authorized", Response.Status.FORBIDDEN.getStatusCode());
		}
		// Return HTTP response with status code 201 Created

		log.debug(" copying gluuGroup ");
		GluuGroup gluuGroup = CopyUtils.copy(group, null, false);
		if (gluuGroup == null) {
			return getErrorResponse("Failed to create group", Response.Status.BAD_REQUEST.getStatusCode());
		}

		try {
			log.debug(" generating inum ");
			String inum = groupService.generateInumForNewGroup();
			log.debug(" getting DN ");
			String dn = groupService.getDnForGroup(inum);
			log.debug(" getting iname ");
			String iname = groupService.generateInameForNewGroup(group.getDisplayName().replaceAll(" ", ""));
			log.debug(" setting dn ");
			gluuGroup.setDn(dn);
			log.debug(" setting inum ");
			gluuGroup.setInum(inum);
			log.debug(" setting iname ");
			gluuGroup.setIname(iname);

			log.info("group.getMembers().size() : " + group.getMembers().size());
			if (group.getMembers().size() > 0) {
				Utils.personMemebersAdder(gluuGroup, dn);
			}

			log.debug("adding new GluuGroup");
			groupService.addGroup(gluuGroup);
			ScimGroup newGroup = CopyUtils.copy(gluuGroup, null);
			String uri = "/Groups/" + newGroup.getId();
			return Response.created(URI.create(uri)).entity(newGroup).build();
		} catch (Exception ex) {
			log.error("Failed to add user", ex);
			return getErrorResponse("Unexpected processing error, please check the input parameters",
					Response.Status.INTERNAL_SERVER_ERROR.getStatusCode());
		}
	}

	@Override
	public Response updateGroup(@Context HttpServletRequest request, @PathParam("id") String id, ScimGroup group) throws Exception {

		groupService = GroupService.instance();

		boolean authorized = getAuthorizedUser();

		if (!authorized) {
			return getErrorResponse("User isn't authorized", Response.Status.FORBIDDEN.getStatusCode());
		}

		try {
			GluuGroup gluuGroup = groupService.getGroupByInum(id);
			if (gluuGroup == null) {
				return getErrorResponse("Resource " + id + " not found", Response.Status.NOT_FOUND.getStatusCode());
			}
			GluuGroup newGluuGroup = CopyUtils.copy(group, gluuGroup, true);

			if (group.getMembers().size() > 0) {
				Utils.personMemebersAdder(newGluuGroup, groupService.getDnForGroup(id));
			}

			groupService.updateGroup(newGluuGroup);
			log.debug(" group updated ");
			ScimGroup newGroup = CopyUtils.copy(newGluuGroup, null);

			URI location = new URI("/Groups/" + id);
			return Response.ok(newGroup).location(location).build();
		} catch (EntryPersistenceException ex) {
			return getErrorResponse("Resource " + id + " not found", Response.Status.NOT_FOUND.getStatusCode());
		} catch (Exception ex) {
			log.error("Exception: ", ex);
			ex.printStackTrace();
			return getErrorResponse("Unexpected processing error, please check the input parameters",
					Response.Status.INTERNAL_SERVER_ERROR.getStatusCode());
		}

	}

	@Override
	public Response deleteGroup(@Context HttpServletRequest request, @PathParam("id") String id) throws Exception {
		groupService = GroupService.instance();
		boolean authorized = getAuthorizedUser();

		if (!authorized) {
			return getErrorResponse("User isn't authorized", Response.Status.FORBIDDEN.getStatusCode());
		}
		try {
			log.info(" Checking if the group exists ");
			log.info(" id : " + id);
			GluuGroup group = groupService.getGroupByInum(id);
			if (group == null) {
				log.info(" the group is null ");
				return getErrorResponse("Resource " + id + " not found", Response.Status.NOT_FOUND.getStatusCode());
			} else {
				log.info(" getting started to delete members from groups ");
				if (group.getMembers() != null) {
					if (group.getMembers().size() > 0) {
						log.info(" getting dn for group ");
						String dn = groupService.getDnForGroup(id);
						log.info(" DN : " + dn);
						Utils.deleteGroupFromPerson(group, dn);
					}
				}
				log.info(" removing the group ");
				groupService.removeGroup(group);
			}
			return Response.ok().build();
		} catch (EntryPersistenceException ex) {
			log.error("Exception: ", ex);
			return getErrorResponse("Resource " + id + " not found", Response.Status.NOT_FOUND.getStatusCode());
		} catch (Exception ex) {
			log.error("Exception: ", ex);
			return getErrorResponse("Unexpected processing error, please check the input parameters",
					Response.Status.INTERNAL_SERVER_ERROR.getStatusCode());
		}
	}

	private boolean getAuthorizedUser() {
		try {
			GluuCustomPerson authUser = (GluuCustomPerson) Contexts.getSessionContext().get(OxTrustConstants.CURRENT_PERSON);
			SecurityService securityService = SecurityService.instance();

			OrganizationService organizationService = OrganizationService.instance();
			GluuOrganization org = organizationService.getOrganization();
			if (!GluuStatus.ACTIVE.equals(org.getScimStatus())) {
				return false;
			}

			GluuUserRole[] userRoles = securityService.getUserRoles(authUser);
			for (GluuUserRole role : userRoles) {

				if (role.getRoleName().equalsIgnoreCase("MANAGER") || role.getRoleName().equalsIgnoreCase("OWNER")) {

					if (Utils.isScimGroupMemberOrOwner(authUser)) {
						return true;
					}
				}
			}
			return false;
		} catch (Exception ex) {
			log.error("Exception: ", ex);
			return false;
		}

	}

	private Response getErrorResponse(String errMsg, int statusCode) {
		Errors errors = new Errors();
		Error error = new org.gluu.oxtrust.model.scim.Error(errMsg, statusCode, "");
		errors.getErrors().add(error);
		return Response.status(statusCode).entity(errors).build();
	}

}
