package org.gluu.oxtrust.ws.rs;

import java.io.StringReader;
import java.net.URI;
import java.util.ArrayList;
import java.util.List;

import javax.servlet.http.HttpServletRequest;
import javax.ws.rs.PathParam;
import javax.ws.rs.QueryParam;
import javax.ws.rs.core.Context;
import javax.ws.rs.core.Response;
import javax.xml.bind.JAXBContext;
import javax.xml.bind.Unmarshaller;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;

import org.dom4j.Document;
import org.dom4j.io.DOMReader;
import org.dom4j.io.DocumentSource;
import org.gluu.oxtrust.ldap.service.OrganizationService;
import org.gluu.oxtrust.ldap.service.PersonService;
import org.gluu.oxtrust.ldap.service.SecurityService;
import org.gluu.oxtrust.model.GluuCustomPerson;
import org.gluu.oxtrust.model.GluuCustomPersonList;
import org.gluu.oxtrust.model.GluuOrganization;
import org.gluu.oxtrust.model.GluuUserRole;
import org.gluu.oxtrust.model.scim.Error;
import org.gluu.oxtrust.model.scim.Errors;
import org.gluu.oxtrust.model.scim.ScimPerson;
import org.gluu.oxtrust.model.scim.ScimPersonPatch;
import org.gluu.oxtrust.model.scim.ScimPersonSearch;
import org.gluu.oxtrust.util.OxTrustConstants;
import org.gluu.oxtrust.util.CopyUtils;
import org.gluu.oxtrust.util.Utils;
import org.gluu.site.ldap.persistence.exception.EntryPersistenceException;
import org.jboss.seam.annotations.In;
import org.jboss.seam.annotations.Logger;
import org.jboss.seam.annotations.Name;
import org.jboss.seam.contexts.Contexts;
import org.jboss.seam.log.Log;
import org.xml.sax.InputSource;

/**
 * SCIM UserWebService Implementation
 * 
 * @author Reda Zerrad Date: 04.03.2012
 */
@Name("userWebService")
public class UserWebServiceImpl implements UserWebService {

	@Logger
	private Log log;

	@In
	private PersonService personService;

	@Override
	public Response listUsers(@Context HttpServletRequest request,
			@QueryParam(OxTrustConstants.QUERY_PARAMETER_FILTER) final String filterString,
			@QueryParam(OxTrustConstants.QUERY_PARAMETER_SORT_BY) final String sortBy,
			@QueryParam(OxTrustConstants.QUERY_PARAMETER_SORT_ORDER) final String sortOrder) throws Exception {

		personService = PersonService.instance();

		boolean authorized = getAuthorizedUser();

		if (!authorized) {
			return getErrorResponse("User isn't authorized", Response.Status.FORBIDDEN.getStatusCode());
		}

		try {
			log.info(" getting a list of all users from LDAP ");
			List<GluuCustomPerson> personList = personService.findAllPersons(null);
			GluuCustomPersonList allPersonList = new GluuCustomPersonList();
			if (personList != null) {
				log.info(" LDAP person list is not empty ");
				for (GluuCustomPerson gluuPerson : personList) {
					log.info(" copying person from GluuPerson to ScimPerson ");
					ScimPerson person = CopyUtils.copy(gluuPerson, null);
					log.info(" adding ScimPerson to the AllPersonList ");
					log.info(" person to be added userid : " + person.getUserName());
					allPersonList.getResources().add(person);
					log.info(" person added? : " + allPersonList.getResources().contains(person));
				}

			}
			List<String> schema = new ArrayList<String>();
			schema.add("urn:scim:schemas:core:1.0");
			log.info(" setting schema ");
			allPersonList.setSchemas(schema);
			List<ScimPerson> resources = allPersonList.getResources();
			allPersonList.setTotalResults((long) resources.size());

			URI location = new URI("/Users/");
			return Response.ok(allPersonList).location(location).build();
		} catch (Exception ex) {
			log.error("Exception: ", ex);
			return getErrorResponse("Unexpected processing error, please check the input parameters",
					Response.Status.INTERNAL_SERVER_ERROR.getStatusCode());
		}
	}

	@Override
	public Response getUserByUid(@Context HttpServletRequest request, @PathParam("uid") String uid) throws Exception {
		personService = PersonService.instance();

		boolean authorized = getAuthorizedUser();

		if (!authorized) {
			return getErrorResponse("User isn't authorized", Response.Status.FORBIDDEN.getStatusCode());
		}

		try {
			GluuCustomPerson gluuPerson = personService.getPersonByInum(uid);
			if (gluuPerson == null) {
				// sets HTTP status code 404 Not Found
				return getErrorResponse("Resource " + uid + " not found", Response.Status.NOT_FOUND.getStatusCode());
			}

			ScimPerson person = CopyUtils.copy(gluuPerson, null);

			URI location = new URI("/Users/" + uid);

			return Response.ok(person).location(location).build();
		} catch (EntryPersistenceException ex) {
			log.error("Exception: ", ex);
			return getErrorResponse("Resource " + uid + " not found", Response.Status.NOT_FOUND.getStatusCode());
		} catch (Exception ex) {
			log.error("Exception: ", ex);
			return getErrorResponse("Unexpected processing error, please check the input parameters",
					Response.Status.INTERNAL_SERVER_ERROR.getStatusCode());
		}
	}

	@Override
	public Response createUser(@Context HttpServletRequest request, ScimPerson person) throws Exception {

		personService = PersonService.instance();

		boolean authorized = getAuthorizedUser();

		if (!authorized) {
			return getErrorResponse("User isn't authorized", Response.Status.FORBIDDEN.getStatusCode());
		}

		// Return HTTP response with status code 201 Created

		log.debug(" copying gluuperson ");
		GluuCustomPerson gluuPerson = CopyUtils.copy(person, null, false);
		if (gluuPerson == null) {
			return getErrorResponse("Failed to create user", Response.Status.BAD_REQUEST.getStatusCode());
		}

		try {
			log.debug(" generating inum ");
			String inum = personService.generateInumForNewPerson(); // inumService.generateInums(Configuration.INUM_TYPE_PEOPLE_SLUG);
																	// //personService.generateInumForNewPerson();
			log.debug(" getting DN ");
			String dn = personService.getDnForPerson(inum);
			log.debug(" getting iname ");
			String iname = personService.generateInameForNewPerson(person.getUserName());
			log.debug(" setting dn ");
			gluuPerson.setDn(dn);
			log.debug(" setting inum ");
			gluuPerson.setInum(inum);
			log.debug(" setting iname ");
			gluuPerson.setIname(iname);
			log.debug(" setting commonName ");
			gluuPerson.setCommonName(gluuPerson.getGivenName() + " " + gluuPerson.getSurname());
			log.info("gluuPerson.getMemberOf().size() : " + gluuPerson.getMemberOf().size());
			if (person.getGroups().size() > 0) {
				log.info(" jumping to groupMemebersAdder ");
				log.info("gluuPerson.getDn() : " + gluuPerson.getDn());

				Utils.groupMemebersAdder(gluuPerson, gluuPerson.getDn());
			}

			log.debug("adding new GluuPerson");

			personService.addPerson(gluuPerson);
			ScimPerson newPerson = CopyUtils.copy(gluuPerson, null);
			String uri = "/Users/" + newPerson.getId();
			return Response.created(URI.create(uri)).entity(newPerson).build();
		} catch (Exception ex) {
			log.error("Failed to add user", ex);
			return getErrorResponse("Unexpected processing error, please check the input parameters",
					Response.Status.INTERNAL_SERVER_ERROR.getStatusCode());
		}
	}

	@Override
	public Response updateUser(@Context HttpServletRequest request, @PathParam("uid") String uid, ScimPerson person_update)
			throws Exception {
		personService = PersonService.instance();

		boolean authorized = getAuthorizedUser();

		if (!authorized) {
			return getErrorResponse("User isn't authorized", Response.Status.FORBIDDEN.getStatusCode());
		}

		try {
			GluuCustomPerson gluuPerson = personService.getPersonByInum(uid);
			if (gluuPerson == null) {
				return getErrorResponse("Resource " + uid + " not found", Response.Status.NOT_FOUND.getStatusCode());
			}
			GluuCustomPerson newGluuPesron = CopyUtils.copy(person_update, gluuPerson, true);

			if (person_update.getGroups().size() > 0) {
				Utils.groupMemebersAdder(newGluuPesron, personService.getDnForPerson(uid));
			}

			personService.updatePerson(newGluuPesron);
			log.debug(" person updated ");
			ScimPerson newPerson = CopyUtils.copy(newGluuPesron, null);

			// person_update = CopyUtils.copy(gluuPerson, null, attributes);
			URI location = new URI("/Users/" + uid);
			return Response.ok(newPerson).location(location).build();
		} catch (EntryPersistenceException ex) {
			return getErrorResponse("Resource " + uid + " not found", Response.Status.NOT_FOUND.getStatusCode());
		} catch (Exception ex) {
			log.error("Exception: ", ex);
			ex.printStackTrace();
			return getErrorResponse("Unexpected processing error, please check the input parameters",
					Response.Status.INTERNAL_SERVER_ERROR.getStatusCode());
		}
	}

	@Override
	public Response deleteUser(@Context HttpServletRequest request, @PathParam("uid") String uid) throws Exception {
		personService = PersonService.instance();

		boolean authorized = getAuthorizedUser();

		if (!authorized) {
			return getErrorResponse("User isn't authorized", Response.Status.FORBIDDEN.getStatusCode());
		}

		try {
			GluuCustomPerson person = personService.getPersonByInum(uid);
			if (person == null) {
				return getErrorResponse("Resource " + uid + " not found", Response.Status.NOT_FOUND.getStatusCode());
			} else {
				log.info("person.getMemberOf().size() : " + person.getMemberOf().size());
				if (person.getMemberOf() != null) {
					if (person.getMemberOf().size() > 0) {
						String dn = personService.getDnForPerson(uid);
						log.info("DN : " + dn);

						Utils.deleteUserFromGroup(person, dn);
					}
				}
				personService.removePerson(person);
			}
			return Response.ok().build();
		} catch (EntryPersistenceException ex) {
			log.error("Exception: ", ex);
			return getErrorResponse("Resource " + uid + " not found", Response.Status.NOT_FOUND.getStatusCode());
		} catch (Exception ex) {
			log.error("Exception: ", ex);
			return getErrorResponse("Unexpected processing error, please check the input parameters",
					Response.Status.INTERNAL_SERVER_ERROR.getStatusCode());
		}
	}

	private Response getErrorResponse(String errMsg, int statusCode) {
		Errors errors = new Errors();
		Error error = new org.gluu.oxtrust.model.scim.Error(errMsg, statusCode, "");
		errors.getErrors().add(error);
		return Response.status(statusCode).entity(errors).build();
	}

	private boolean getAuthorizedUser() {
		try {
			GluuCustomPerson authUser = (GluuCustomPerson) Contexts.getSessionContext().get(OxTrustConstants.CURRENT_PERSON);
			SecurityService securityService = SecurityService.instance();

			OrganizationService organizationService = OrganizationService.instance();
			GluuOrganization org = organizationService.getOrganization();
			if (org.getScimStatus().equalsIgnoreCase("deactivated")) {
				return false;
			}

			GluuUserRole[] userRoles = securityService.getUserRoles(authUser);
			for (GluuUserRole role : userRoles) {

				if (role.getRoleName().equalsIgnoreCase("MANAGER") || role.getRoleName().equalsIgnoreCase("OWNER")) {
					if (Utils.isScimGroupMemberOrOwner(authUser)) {
						return true;
					}
				}
			}
			return false;
		} catch (Exception ex) {
			log.error("Exception: ", ex);
			return false;
		}

	}

	public Response createUserTestHelper(@Context HttpServletRequest request, String person_data) throws Exception {
		try {
			ScimPerson person = (ScimPerson) xmlToObject(person_data);
			return createUser(request, person);
		} catch (Exception ex) {
			log.error("Exception: ", ex);
			ex.printStackTrace();
			return getErrorResponse("Data parsing error.", Response.Status.INTERNAL_SERVER_ERROR.getStatusCode());
		}
	}

	public Response updateUserTestHelper(@Context HttpServletRequest request, @PathParam("inum") String inum, String person_data)
			throws Exception {
		try {
			ScimPerson person_update = (ScimPerson) xmlToObject(person_data);
			return updateUser(request, inum, person_update);
		} catch (Exception ex) {
			log.error("Exception: ", ex);
			return getErrorResponse("Data parsing error.", Response.Status.INTERNAL_SERVER_ERROR.getStatusCode());
		}
	}

	private Object xmlToObject(String xml) throws Exception {
		DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
		DocumentBuilder builder = factory.newDocumentBuilder();
		org.w3c.dom.Document document = builder.parse(new InputSource(new StringReader(xml)));
		DOMReader reader = new DOMReader();
		Document doc = reader.read(document);

		JAXBContext context = JAXBContext.newInstance(ScimPerson.class);
		DocumentSource source = new DocumentSource(doc);
		context = JAXBContext.newInstance(ScimPerson.class);
		Unmarshaller unmarshaller = context.createUnmarshaller();
		return unmarshaller.unmarshal(source);
	}

	@Override
	public Response updateUserPatch(HttpServletRequest request, String uid, ScimPersonPatch person) {
		// TODO Auto-generated method stub
		return null;
	}

	@Override
	public Response personSearch(HttpServletRequest request, ScimPersonSearch searchPattern) throws Exception {
		personService = PersonService.instance();
		boolean authorized = getAuthorizedUser();
		if (!authorized) {
			return getErrorResponse("User isn't authorized", Response.Status.FORBIDDEN.getStatusCode());
		}

		try {
			GluuCustomPerson gluuPerson = personService.getPersonByAttribute(searchPattern.getAttribute(), searchPattern.getValue());
			if (gluuPerson == null) {
				// sets HTTP status code 404 Not Found
				return getErrorResponse(
						"No result found for search pattern '" + searchPattern.getAttribute() + " = " + searchPattern.getValue()
								+ "' please try again or use another pattern.", Response.Status.NOT_FOUND.getStatusCode());
			}
			ScimPerson person = CopyUtils.copy(gluuPerson, null);
			URI location = new URI("/Users/" + gluuPerson.getInum());
			return Response.ok(person).location(location).build();
		} catch (EntryPersistenceException ex) {
			log.error("Exception: ", ex);
			return getErrorResponse("Resource not found", Response.Status.NOT_FOUND.getStatusCode());
		} catch (Exception ex) {
			log.error("Exception: ", ex);
			return getErrorResponse("Unexpected processing error, please check the input parameters",
					Response.Status.INTERNAL_SERVER_ERROR.getStatusCode());
		}
	}
}
