package org.gluu.oxtrust.model;

import java.io.Serializable;
import java.util.Arrays;

import javax.persistence.Transient;
import javax.validation.constraints.NotNull;
import javax.validation.constraints.Pattern;
import javax.validation.constraints.Size;

import org.gluu.site.ldap.persistence.annotation.LdapAttribute;
import org.gluu.site.ldap.persistence.annotation.LdapEntry;
import org.gluu.site.ldap.persistence.annotation.LdapObjectClass;
import org.xdi.ldap.model.Entry;
import org.xdi.ldap.model.GluuStatus;

/**
 * Attribute Metadata
 * 
 * @author Yuriy Movchan Date: 10.07.2010
 */
@LdapEntry(sortBy = { "displayName" })
@LdapObjectClass(values = { "top", "gluuAttribute" })
public class GluuAttribute extends Entry implements Serializable {

	private static final long serialVersionUID = 4817004894646725606L;

	private transient boolean selected;

	@LdapAttribute(ignoreDuringUpdate = true)
	private String inum;

	@NotNull
	@Pattern(regexp = "^[a-zA-Z_]+$", message = "Name should contain only letters and underscores")
	@Size(min = 1, max = 30, message = "Length of the Name should be between 1 and 30")
	@LdapAttribute(name = "gluuAttributeName")
	private String name;

	@NotNull
	@Size(min = 0, max = 60, message = "Length of the Display Name should not exceed 60")
	@LdapAttribute
	private String displayName;

	@NotNull
	@Size(min = 0, max = 4000, message = "Length of the Description should not exceed 4000")
	@LdapAttribute
	private String description;

	@LdapAttribute(name = "gluuAttributeOrigin")
	private String origin;

	@NotNull
	@LdapAttribute(name = "gluuAttributeType")
	private GluuAttributeDataType dataType;

	@NotNull
	@LdapAttribute(name = "gluuAttributeEditType")
	private GluuUserRole[] editType;

	@NotNull
	@LdapAttribute(name = "gluuAttributeViewType")
	private GluuUserRole[] viewType;

	@NotNull
	@LdapAttribute(name = "gluuAttributePrivacyLevel")
	private GluuAttributePrivacyLevel privacyLevel;

	@LdapAttribute(name = "seeAlso")
	private String seeAlso;

	@LdapAttribute(name = "gluuStatus")
	private GluuStatus status;

	@LdapAttribute(name = "gluuSAML1URI")
	private String saml1Uri;

	@LdapAttribute(name = "gluuSAML2URI")
	private String saml2Uri;

	@LdapAttribute(ignoreDuringUpdate = true)
	private String urn;

	@LdapAttribute(name = "oxSCIMCustomAttribute")
	private ScimCustomAtribute oxSCIMCustomAttribute;

	@LdapAttribute(name = "oxMultivaluedAttribute")
	private OxMultivalued oxMultivaluedAttribute;

	@Transient
	private boolean custom;

	@Transient
	private boolean requred;

	public String getInum() {
		return inum;
	}

	public void setInum(String inum) {
		this.inum = inum;
	}

	public String getName() {
		return name;
	}

	public void setName(String name) {
		this.name = name;
	}

	public String getDisplayName() {
		return displayName;
	}

	public void setDisplayName(String displayName) {
		this.displayName = displayName;
	}

	public String getDescription() {
		return description;
	}

	public void setDescription(String description) {
		this.description = description;
	}

	public String getOrigin() {
		return origin;
	}

	public void setOrigin(String origin) {
		this.origin = origin;
	}

	public GluuAttributeDataType getDataType() {
		return dataType;
	}

	public void setDataType(GluuAttributeDataType dataType) {
		this.dataType = dataType;
	}

	public GluuUserRole[] getEditType() {
		return editType;
	}

	public void setEditType(GluuUserRole[] editType) {
		this.editType = editType;
	}

	public GluuUserRole[] getViewType() {
		return viewType;
	}

	public void setViewType(GluuUserRole[] viewType) {
		this.viewType = viewType;
	}

	public GluuAttributePrivacyLevel getPrivacyLevel() {
		return privacyLevel;
	}

	public void setPrivacyLevel(GluuAttributePrivacyLevel privacyLevel) {
		this.privacyLevel = privacyLevel;
	}

	public String getSeeAlso() {
		return seeAlso;
	}

	public void setSeeAlso(String seeAlso) {
		this.seeAlso = seeAlso;
	}

	public GluuStatus getStatus() {
		return status;
	}

	public void setStatus(GluuStatus status) {
		this.status = status;
	}

	public boolean isCustom() {
		return custom;
	}

	public void setCustom(boolean custom) {
		this.custom = custom;
	}

	public boolean allowEditBy(GluuUserRole role) {
		return GluuUserRole.containsRole(editType, role);
	}

	public boolean allowViewBy(GluuUserRole role) {
		return GluuUserRole.containsRole(viewType, role);
	}

	public boolean isAdminCanAccess() {
		return isAdminCanView() | isAdminCanEdit();
	}

	public boolean isAdminCanView() {
		return allowViewBy(GluuUserRole.ADMIN);
	}

	public boolean isAdminCanEdit() {
		return allowEditBy(GluuUserRole.ADMIN);
	}

	public boolean isUserCanAccess() {
		return isUserCanView() | isUserCanEdit();
	}

	public boolean isUserCanView() {
		return allowViewBy(GluuUserRole.USER);
	}

	public boolean isWhitePagesCanView() {
		return allowViewBy(GluuUserRole.WHITEPAGES);
	}

	public boolean isUserCanEdit() {
		return allowEditBy(GluuUserRole.USER);
	}

	public String getUrn() {
		return urn;
	}

	public void setUrn(String urn) {
		this.urn = urn;
	}

	public ScimCustomAtribute getOxSCIMCustomAttribute() {
		return this.oxSCIMCustomAttribute;
	}

	public void setOxSCIMCustomAttribute(ScimCustomAtribute oxSCIMCustomAttribute) {
		this.oxSCIMCustomAttribute = oxSCIMCustomAttribute;
	}

	public OxMultivalued getOxMultivaluedAttribute() {
		return this.oxMultivaluedAttribute;
	}

	public void setOxMultivaluedAttribute(OxMultivalued oxMultivaluedAttribute) {
		this.oxMultivaluedAttribute = oxMultivaluedAttribute;
	}

	@Override
	public String toString() {
		return "GluuAttribute [inum=" + inum + ", name=" + name + ", displayName=" + displayName + ", description=" + description
				+ ", origin=" + origin + ", dataType=" + dataType + ", editType=" + Arrays.toString(editType) + ", viewType="
				+ Arrays.toString(viewType) + ", privacyLevel=" + privacyLevel + ", seeAlso=" + seeAlso + ", status=" + status
				+ ", saml1Uri=" + saml1Uri + ", saml2Uri=" + saml2Uri + ", urn=" + urn + ", custom=" + custom + ", requred=" + requred
				+ ", oxSCIMCustomAttribute=" + oxSCIMCustomAttribute + ",oxMultivaluedAttribute=" + oxMultivaluedAttribute
				+ ", toString()=" + super.toString() + "]";
	}

	public void setSaml1Uri(String saml1Uri) {
		this.saml1Uri = saml1Uri;
	}

	public String getSaml1Uri() {
		return saml1Uri;
	}

	public void setSaml2Uri(String saml2Uri) {
		this.saml2Uri = saml2Uri;
	}

	public String getSaml2Uri() {
		return saml2Uri;
	}

	public boolean isRequred() {
		return requred;
	}

	public void setRequred(boolean requred) {
		this.requred = requred;
	}

	public boolean isSelected() {
		return selected;
	}

	public void setSelected(boolean selected) {
		this.selected = selected;
	}

}
