package org.gluu.oxtrust.ws.rs;

import java.net.URI;

import javax.persistence.EntityManager;
import javax.servlet.http.HttpServletRequest;
import javax.ws.rs.core.Response;

import org.codehaus.jackson.map.ObjectMapper;
import org.gluu.oxtrust.ldap.service.InumService;
import org.gluu.oxtrust.ldap.service.OrganizationService;
import org.gluu.oxtrust.ldap.service.SecurityService;
import org.gluu.oxtrust.ldap.service.intercept.InumGeneratorInterceptorService;
import org.gluu.oxtrust.model.GluuCustomPerson;
import org.gluu.oxtrust.model.GluuOrganization;
import org.gluu.oxtrust.model.GluuUserRole;
import org.gluu.oxtrust.model.InumConf;
import org.gluu.oxtrust.model.InumResponse;
import org.gluu.oxtrust.model.python.InumGeneratorType;
import org.gluu.oxtrust.model.scim.Error;
import org.gluu.oxtrust.model.scim.Errors;
import org.gluu.oxtrust.model.sql.InumSqlEntry;
import org.gluu.oxtrust.util.OxTrustConstants;
import org.gluu.oxtrust.util.Utils;
import org.jboss.seam.Component;
import org.jboss.seam.annotations.In;
import org.jboss.seam.annotations.Logger;
import org.jboss.seam.annotations.Name;
import org.jboss.seam.contexts.Contexts;
import org.jboss.seam.log.Log;

/**
 * Inum Generation WebService Implementation
 * 
 * @author Reda Zerrad Date: 08.22.2012
 */
@Name("inumGenerationWebService")
public class InumGenerationWebServiceImpl implements InumGenerationWebService {
	@Logger
	Log log;

	InumGeneratorInterceptorService inumInterService;

	@In
	OrganizationService organizationService;

	@In
	InumService inumService;

	@Override
	public Response getInum(HttpServletRequest request, String prefix) throws Exception {

		boolean authorized = getAuthorizedUser();

		if (!authorized) {
			return getErrorResponse("User isn't authorized", Response.Status.FORBIDDEN.getStatusCode());
		}

		try {
			organizationService = OrganizationService.instance();
			GluuOrganization org = organizationService.getOrganization();
			inumInterService = InumGeneratorInterceptorService.instance();
			InumGeneratorType inumGeneratorType = inumInterService.createInumGenerator();
			InumConf conf = null;

			conf = (InumConf) jsonToObject(org.getOxInumConfig(), InumConf.class);

			String EntityPrefix = null;

			if (prefix.equalsIgnoreCase("person")) {
				EntityPrefix = conf.getPersonPrefix();
			} else if (prefix.equalsIgnoreCase("group")) {
				EntityPrefix = conf.getGroupPrefix();
			} else if (prefix.equalsIgnoreCase("client")) {
				EntityPrefix = conf.getClientPrefix();
			} else if (prefix.equalsIgnoreCase("scope")) {
				EntityPrefix = conf.getScopePrefix();
			} else {
				EntityPrefix = prefix;
			}

			String inum = inumGeneratorType.generateInum(org.getInum(), EntityPrefix);

			EntityManager inumEntryManager = (EntityManager) Component.getInstance("InumEntryManager");
			inumService.addInum(inumEntryManager, inum, prefix);

			InumSqlEntry foundInum = inumService.findInum(inumEntryManager, "@!1111!0000!D4E7");
			System.out.println("foundInum : " + foundInum.getInum());

			URI location = new URI("/InumGenerator/" + inum);

			InumResponse inumObject = new InumResponse();
			inumObject.setGeneratedInum(inum);

			return Response.ok(inumObject).location(location).build();
		} catch (Exception ex) {
			log.error("Exception: ", ex);
			ex.printStackTrace();
			return getErrorResponse("Unexpected processing error, please check the input parameters",
					Response.Status.INTERNAL_SERVER_ERROR.getStatusCode());
		}

	}

	private Object jsonToObject(String json, Class<?> clazz) throws Exception {

		ObjectMapper mapper = new ObjectMapper();
		Object clazzObject = mapper.readValue(json, clazz);
		return clazzObject;
	}

	private Response getErrorResponse(String errMsg, int statusCode) {
		Errors errors = new Errors();
		Error error = new org.gluu.oxtrust.model.scim.Error(errMsg, statusCode, "");
		errors.getErrors().add(error);
		return Response.status(statusCode).entity(errors).build();
	}

	private boolean getAuthorizedUser() {
		try {
			GluuCustomPerson authUser = (GluuCustomPerson) Contexts.getSessionContext().get(OxTrustConstants.CURRENT_PERSON);
			SecurityService securityService = SecurityService.instance();

			OrganizationService organizationService = OrganizationService.instance();
			GluuOrganization org = organizationService.getOrganization();
			if (org.getScimStatus().equalsIgnoreCase("deactivated")) {
				return false;
			}

			GluuUserRole[] userRoles = securityService.getUserRoles(authUser);
			for (GluuUserRole role : userRoles) {

				if (role.getRoleName().equalsIgnoreCase("MANAGER") || role.getRoleName().equalsIgnoreCase("OWNER")) {
					if (Utils.isScimGroupMemberOrOwner(authUser)) {
						return true;
					}
				}
			}
			return false;
		} catch (Exception ex) {
			log.error("Exception: ", ex);
			return false;
		}

	}

}
