package org.gluu.oxtrust.ldap.service.uma;

import java.net.URI;
import java.net.URISyntaxException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;

import javax.faces.application.ViewHandler;
import javax.faces.context.ExternalContext;
import javax.faces.context.FacesContext;

import org.gluu.oxtrust.ldap.service.HostService;
import org.gluu.oxtrust.ldap.service.ViewHandlerService;
import org.gluu.oxtrust.model.uma.HostToken;
import org.gluu.oxtrust.model.uma.ResourceSet;
import org.jboss.resteasy.client.ClientResponseFailure;
import org.jboss.seam.ScopeType;
import org.jboss.seam.annotations.AutoCreate;
import org.jboss.seam.annotations.In;
import org.jboss.seam.annotations.Logger;
import org.jboss.seam.annotations.Name;
import org.jboss.seam.annotations.Scope;
import org.jboss.seam.log.Log;
import org.xdi.oxauth.client.uma.MetaDataConfigurationService;
import org.xdi.oxauth.client.uma.ResourceSetRegistrationService;
import org.xdi.oxauth.client.uma.UmaClientFactory;
import org.xdi.oxauth.model.uma.MetadataConfiguration;
import org.xdi.oxauth.model.uma.persistence.ScopeDescription;
import org.xdi.util.StringHelper;

/**
 * Provides operations with UMA host AM protection endpoint
 * 
 * @author Yuriy Movchan
 * @version 0.1, 12/08/2012
 */
@Name("clientUmaWebService")
@Scope(ScopeType.STATELESS)
@AutoCreate
public class ClientUmaWebService {

	@Logger
	private Log log;

	@In
	private FacesContext facesContext;

	@In(value = "#{facesContext.externalContext}")
	private ExternalContext externalContext;

	@In(value = "#{facesContext.application.viewHandler}")
	private ViewHandler viewHandler;

	@In
	private HostTokenService hostTokenService;

	@In
	private HostService hostService;

	@In
	private ScopeDescriptionService scopeDescriptionService;

	@In
	private ViewHandlerService viewHandlerService;

	public MetadataConfiguration obtainUmaMetaDataConfigurationByIssuer(String issuer) {
		if (StringHelper.isEmpty(issuer)) {
			log.error("Failed to obtain metadata due to invalid UMA AM meta data configuration from '{0}'", issuer);
			return null;
		}

		MetadataConfiguration metadataConfiguration = null;
		try {
			URI uri = new URI(issuer);
			if (StringHelper.isEmpty(uri.getPath())) {
				// Attempt to read default metadata configuration
				metadataConfiguration = obtainUmaMetaDataConfigurationByUri(issuer + "/.well-known/uma-configuration");
				if (metadataConfiguration == null) {
					// Attempt to read oxAuth specific metadata configuration
					metadataConfiguration = obtainUmaMetaDataConfigurationByUri(issuer
							+ "/oxauth/seam/resource/restv1/oxauth/uma-configuration");
				}
			}
		} catch (URISyntaxException ex) {
			log.error("Failed to obtain metadata due to invalid UMA AM meta data configuration from '{0}'", ex, issuer);
			return null;
		}

		return metadataConfiguration;
	}

	@SuppressWarnings("unchecked")
	public MetadataConfiguration obtainUmaMetaDataConfigurationByUri(String metaDataConfigurationUri) {
		MetaDataConfigurationService metaDataConfigurationService = UmaClientFactory.instance().createMetaDataConfigurationService(
				metaDataConfigurationUri);

		// Get meta data configuration
		try {
			return metaDataConfigurationService.getMetadataConfiguration();
		} catch (ClientResponseFailure ex) {
			log.error("Get invalid responce from UMA server: '{0}'", ex, ex.getResponse().getEntity(String.class));
		} catch (Exception ex) {
			log.error("Failed to load meta data configuration", ex);
		}

		return null;
	}

	@SuppressWarnings("unchecked")
	public org.xdi.oxauth.model.uma.VersionedResourceSet getUmaResourceSet(HostToken hostToken, String resourceSetId) {
		// Get resource
		ResourceSetRegistrationService resourceSetRegistrationService = UmaClientFactory.instance().createResourceSetRegistrationService(
				hostToken.getMetadataConfiguration());
		try {
			org.xdi.oxauth.model.uma.VersionedResourceSet umaResourceSet = resourceSetRegistrationService.getResourceSet("Bearer "
					+ hostToken.getToken(), resourceSetId);

			return umaResourceSet;
		} catch (ClientResponseFailure ex) {
			log.error("Get invalid responce from UMA server: '{0}'", ex, ex.getResponse().getEntity(String.class));
		}

		return null;
	}

	@SuppressWarnings("unchecked")
	public boolean addUmaResourceSet(HostToken hostToken, ResourceSet resourceSet) {
		org.xdi.oxauth.model.uma.ResourceSet umaResourceSet = convertToUmaResourceSet(hostToken, resourceSet);
		// Add resource set description

		ResourceSetRegistrationService resourceSetRegistrationService = UmaClientFactory.instance().createResourceSetRegistrationService(
				hostToken.getMetadataConfiguration());
		try {
			org.xdi.oxauth.model.uma.ResourceSetStatus resourceSetStatus = resourceSetRegistrationService.addResourceSet("Bearer "
					+ hostToken.getToken(), resourceSet.getId(), umaResourceSet);

			return StringHelper.equals(resourceSetStatus.getStatus(), "created");
		} catch (ClientResponseFailure ex) {
			log.error("Get invalid responce from UMA server: '{0}'", ex, ex.getResponse().getEntity(String.class));
		}

		return false;
	}

	@SuppressWarnings("unchecked")
	public boolean updateUmaResourceSet(HostToken hostToken, ResourceSet resourceSet) {
		org.xdi.oxauth.model.uma.ResourceSet umaResourceSet = convertToUmaResourceSet(hostToken, resourceSet);
		org.xdi.oxauth.model.uma.VersionedResourceSet loadedUmaResourceSet = getUmaResourceSet(hostToken, resourceSet.getId());

		ResourceSetRegistrationService resourceSetRegistrationService = UmaClientFactory.instance().createResourceSetRegistrationService(
				hostToken.getMetadataConfiguration());
		try {
			org.xdi.oxauth.model.uma.ResourceSetStatus resourceSetStatus = resourceSetRegistrationService.updateResourceSet("Bearer "
					+ hostToken.getToken(), loadedUmaResourceSet.getRev(), loadedUmaResourceSet.getId(), umaResourceSet);

			return StringHelper.equals(resourceSetStatus.getStatus(), "updated");
		} catch (ClientResponseFailure ex) {
			log.error("Get invalid responce from UMA server: '{0}'", ex, ex.getResponse().getEntity(String.class));
		}

		return false;
	}

	@SuppressWarnings("unchecked")
	public boolean deleteUmaResourceSet(HostToken hostToken, ResourceSet resourceSet) {
		org.xdi.oxauth.model.uma.VersionedResourceSet loadedUmaResourceSet = getUmaResourceSet(hostToken, resourceSet.getId());

		ResourceSetRegistrationService resourceSetRegistrationService = UmaClientFactory.instance().createResourceSetRegistrationService(
				hostToken.getMetadataConfiguration());
		try {
			resourceSetRegistrationService.deleteResourceSet("Bearer " + hostToken.getToken(), loadedUmaResourceSet.getRev(),
					loadedUmaResourceSet.getId());

			return true;
		} catch (ClientResponseFailure ex) {
			log.error("Get invalid responce from UMA server: '{0}'", ex, ex.getResponse().getEntity(String.class));
		}

		return false;
	}

	private org.xdi.oxauth.model.uma.ResourceSet convertToUmaResourceSet(HostToken hostToken, ResourceSet resourceSet) {
		List<ScopeDescription> hostScopeDescriptions = scopeDescriptionService.getAllScopeDescriptions(
				hostService.getDnForHost(hostToken.getHostInum()), "oxId");

		HashMap<String, List<String>> pageParams = new HashMap<String, List<String>>();
		pageParams.put("host", Arrays.asList(hostToken.getHostInum()));

		// Prepare UMA resource set
		org.xdi.oxauth.model.uma.ResourceSet umaResourceSet = new org.xdi.oxauth.model.uma.ResourceSet();
		umaResourceSet.setName(resourceSet.getId());

		List<String> associatedScopes = resourceSet.getAssociatedScopes();
		if ((associatedScopes != null) && (associatedScopes.size() > 0)) {
			List<String> umaScopes = new ArrayList<String>();

			for (ScopeDescription hostScopeDescription : hostScopeDescriptions) {
				if (associatedScopes.contains(hostScopeDescription.getDn())) {
					pageParams.put("scope", Arrays.asList(hostScopeDescription.getId()));

					String umaScope = viewHandlerService.getBookmarkableURL("/admin/uma/scope/scopeDescriptionFile.xhtml", pageParams);
					umaScopes.add(umaScope);
				}
			}
			umaResourceSet.setScopes(umaScopes);
		}

		// TODO: UMA fix
		umaResourceSet.setIconUri("icon");

		return umaResourceSet;
	}

}