package org.gluu.oxtrust.action.uma;

import java.io.Serializable;
import java.util.List;

import javax.validation.constraints.NotNull;
import javax.validation.constraints.Size;

import org.gluu.oxtrust.ldap.service.ClientService;
import org.gluu.oxtrust.ldap.service.HostService;
import org.gluu.oxtrust.ldap.service.ImageService;
import org.gluu.oxtrust.ldap.service.uma.ResourceSetService;
import org.gluu.oxtrust.model.GluuImage;
import org.gluu.oxtrust.model.uma.ResourceSet;
import org.gluu.oxtrust.util.Configuration;
import org.jboss.seam.ScopeType;
import org.jboss.seam.annotations.In;
import org.jboss.seam.annotations.Logger;
import org.jboss.seam.annotations.Name;
import org.jboss.seam.annotations.Scope;
import org.jboss.seam.annotations.security.Restrict;
import org.jboss.seam.log.Log;
import org.xdi.util.StringHelper;
import org.xdi.util.Util;

/**
 * Action class for search resource sets
 * 
 * @author Yuriy Movchan Date: 12/06/2012
 */
@Name("searchResourceSetAction")
@Scope(ScopeType.CONVERSATION)
@Restrict("#{identity.loggedIn}")
public class SearchResourceSetAction implements Serializable {

	private static final long serialVersionUID = 1361095046179474395L;

	@Logger
	private Log log;

	@In
	protected HostService hostService;

	@In
	private ResourceSetService resourceSetService;

	@In
	protected ClientService clientService;

	@In
	protected ImageService imageService;

	private String hostInum, hostDn;

	@NotNull
	@Size(min = 0, max = 30, message = "Length of search string should be less than 30")
	private String searchPattern;

	private String oldSearchPattern;

	private List<ResourceSet> resourceSetList;

	@Restrict("#{s:hasPermission('uma', 'access')}")
	public String start() {
		try {
			this.hostDn = hostService.getDnForHost(this.hostInum);
			if (!hostService.contains(this.hostDn)) {
				return Configuration.RESULT_FAILURE;
			}

			resourceSetService.prepareResourceSetBranch(this.hostDn);
		} catch (Exception ex) {
			log.error("Failed to initialize form", ex);
			return Configuration.RESULT_FAILURE;
		}

		if (StringHelper.isEmpty(this.searchPattern)) {
			searchPattern = "";
		}

		search();

		return Configuration.RESULT_SUCCESS;
	}

	@Restrict("#{s:hasPermission('uma', 'access')}")
	public String search() {
		if (Util.equals(this.oldSearchPattern, this.searchPattern)) {
			return Configuration.RESULT_SUCCESS;
		}

		try {
			this.resourceSetList = resourceSetService.searchResourceSets(this.hostDn, this.searchPattern, 100);
			this.oldSearchPattern = this.searchPattern;
		} catch (Exception ex) {
			log.error("Failed to find resource sets", ex);

			return Configuration.RESULT_FAILURE;
		}

		return Configuration.RESULT_SUCCESS;
	}

	public byte[] getIcon(ResourceSet resourceSet) {
		GluuImage gluuImage = imageService.getGluuImageFromXML(resourceSet.getIcon());

		if (gluuImage != null) {
			try {
				return imageService.getThumImageData(gluuImage);
			} catch (Exception ex) {
				log.error("Failed to load thumnail data for image '{0}'", ex, gluuImage);
			}
		}

		return imageService.getBlankIconData();
	}

	public String getSearchPattern() {
		return searchPattern;
	}

	public void setSearchPattern(String searchPattern) {
		this.searchPattern = searchPattern;
	}

	public List<ResourceSet> getResourceSetList() {
		return resourceSetList;
	}

	public String getHostInum() {
		return hostInum;
	}

	public void setHostInum(String hostInum) {
		this.hostInum = hostInum;
	}

}
