package org.gluu.oxtrust.action.host;

import java.io.Serializable;
import java.util.Date;

import org.gluu.oxtrust.ldap.service.ApplianceService;
import org.gluu.oxtrust.ldap.service.HostService;
import org.gluu.oxtrust.model.GluuCustomPerson;
import org.gluu.oxtrust.model.OxAuthHost;
import org.gluu.oxtrust.util.Configuration;
import org.gluu.site.ldap.persistence.exception.LdapMappingException;
import org.jboss.seam.ScopeType;
import org.jboss.seam.annotations.Destroy;
import org.jboss.seam.annotations.In;
import org.jboss.seam.annotations.Logger;
import org.jboss.seam.annotations.Name;
import org.jboss.seam.annotations.Scope;
import org.jboss.seam.annotations.security.Restrict;
import org.jboss.seam.log.Log;
import org.xdi.util.StringHelper;

/**
 * Action class for view and update host
 * 
 * @author Yuriy Movchan Date: 12/11/2012
 */
@Name("updateHostAction")
@Scope(ScopeType.CONVERSATION)
@Restrict("#{identity.loggedIn}")
public class UpdateHostAction implements Serializable {

	private static final long serialVersionUID = 6180729281938167478L;

	@Logger
	private Log log;

	@In
	protected GluuCustomPerson currentPerson;

	@In
	protected ApplianceService applianceService;

	@In
	protected HostService hostService;

	private String inum;

	private OxAuthHost host;

	private boolean update;

	@Restrict("#{s:hasPermission('uma', 'access')}")
	public String modify() {
		this.update = StringHelper.isNotEmpty(this.inum);

		try {
			hostService.prepareHostBranch();
		} catch (Exception ex) {
			log.error("Failed to initialize form", ex);
			return Configuration.RESULT_FAILURE;
		}

		if (update) {
			return update();
		} else {
			return add();
		}
	}

	private String add() {
		if (this.host != null) {
			return Configuration.RESULT_SUCCESS;
		}

		this.host = new OxAuthHost();

		return Configuration.RESULT_SUCCESS;
	}

	private String update() {
		if (this.host != null) {
			return Configuration.RESULT_SUCCESS;
		}

		log.debug("Loading host '{0}'", this.inum);
		try {
			this.host = hostService.getHostByDn(hostService.getDnForHost(this.inum));
		} catch (LdapMappingException ex) {
			log.error("Failed to find host '{0}'", ex, this.inum);
			return Configuration.RESULT_FAILURE;
		}

		if (this.host == null) {
			log.error("Host description is null");
			return Configuration.RESULT_FAILURE;
		}

		return Configuration.RESULT_SUCCESS;
	}

	@Restrict("#{s:hasPermission('uma', 'access')}")
	public void cancel() {
	}

	@Restrict("#{s:hasPermission('uma', 'access')}")
	public String save() {
		if (this.update) {
			host.setRev(String.valueOf(StringHelper.toInteger(host.getRev(), 0) + 1));
			// Update host
			try {
				hostService.updateHost(this.host);
			} catch (LdapMappingException ex) {
				log.error("Failed to update host '{0}'", ex, this.host.getId());
				return Configuration.RESULT_FAILURE;
			}
		} else {
			// TODO: Validate id

			// Prepare score description
			this.host.setRev(String.valueOf(0));

			String inum = hostService.generateInumForNewHost();
			String hostDn = hostService.getDnForHost(inum);

			this.host.setInum(inum);
			this.host.setDn(hostDn);
			this.host.setOwner(currentPerson.getDn());
			this.host.setRegistrationDate(new Date());

			// Check if host with this name already exist
			OxAuthHost exampleHost = new OxAuthHost();
			exampleHost.setDn(hostDn);
			exampleHost.setId(host.getId());
			if (hostService.containsHost(exampleHost)) {
				return Configuration.RESULT_DUPLICATE;
			}

			// Save host
			try {
				hostService.addHost(this.host);
			} catch (LdapMappingException ex) {
				log.error("Failed to add new host '{0}'", ex, this.host.getId());
				return Configuration.RESULT_FAILURE;
			}

			this.update = true;
		}

		log.debug("Host description were {0} successfully", (this.update ? "added" : "updated"));
		return Configuration.RESULT_SUCCESS;
	}

	@Restrict("#{s:hasPermission('uma', 'access')}")
	public String delete() {
		if (update) {
			// Remove host
			try {
				// TODO: Remove all data from UMA OAuth server
				// TODO: Remove scopes, clients, ...
				hostService.removeHost(this.host);
				return Configuration.RESULT_SUCCESS;
			} catch (LdapMappingException ex) {
				log.error("Failed to remove host {0}", ex, this.host.getId());
			}
		}

		return Configuration.RESULT_FAILURE;
	}

	@Destroy
	public void destroy() throws Exception {
		cancel();
	}

	public boolean isUpdate() {
		return update;
	}

	public OxAuthHost getHost() {
		return this.host;
	}

	public String getInum() {
		return inum;
	}

	public void setInum(String inum) {
		this.inum = inum;
	}

}
