package org.gluu.oxtrust.action.host;

import java.io.Serializable;
import java.util.List;

import javax.validation.constraints.NotNull;
import javax.validation.constraints.Size;

import org.gluu.oxtrust.ldap.service.ApplianceService;
import org.gluu.oxtrust.ldap.service.ClientService;
import org.gluu.oxtrust.ldap.service.HostService;
import org.gluu.oxtrust.ldap.service.ImageService;
import org.gluu.oxtrust.model.OxAuthHost;
import org.gluu.oxtrust.util.Configuration;
import org.jboss.seam.ScopeType;
import org.jboss.seam.annotations.In;
import org.jboss.seam.annotations.Logger;
import org.jboss.seam.annotations.Name;
import org.jboss.seam.annotations.Scope;
import org.jboss.seam.annotations.security.Restrict;
import org.jboss.seam.log.Log;
import org.xdi.util.StringHelper;
import org.xdi.util.Util;

/**
 * Action class for search hosts
 * 
 * @author Yuriy Movchan Date: 12/11/2012
 */
@Name("searchHostAction")
@Scope(ScopeType.CONVERSATION)
@Restrict("#{identity.loggedIn}")
public class SearchHostAction implements Serializable {

	private static final long serialVersionUID = 1361095046179474395L;

	@Logger
	private Log log;

	@In
	protected ApplianceService applianceService;

	@In
	private HostService hostService;

	@In
	protected ClientService clientService;

	@In
	protected ImageService imageService;

	private String applianceInum, applianceDn;

	@NotNull
	@Size(min = 0, max = 30, message = "Length of search string should be less than 30")
	private String searchPattern;

	private String oldSearchPattern;

	private List<OxAuthHost> hostList;

	@Restrict("#{s:hasPermission('uma', 'access')}")
	public String start() {
		this.applianceInum = applianceService.getApplianceInum();
		try {
			this.applianceDn = applianceService.getDnForAppliance(this.applianceInum);
			if (!applianceService.contains(this.applianceDn)) {
				return Configuration.RESULT_FAILURE;
			}

			hostService.prepareHostBranch();
		} catch (Exception ex) {
			log.error("Failed to initialize form", ex);
			return Configuration.RESULT_FAILURE;
		}

		if (StringHelper.isEmpty(this.searchPattern)) {
			searchPattern = "";
		}

		search();

		return Configuration.RESULT_SUCCESS;
	}

	@Restrict("#{s:hasPermission('uma', 'access')}")
	public String search() {
		if (Util.equals(this.oldSearchPattern, this.searchPattern)) {
			return Configuration.RESULT_SUCCESS;
		}

		try {
			this.hostList = hostService.searchHosts(this.searchPattern, 100);
			this.oldSearchPattern = this.searchPattern;
		} catch (Exception ex) {
			log.error("Failed to find hosts", ex);

			return Configuration.RESULT_FAILURE;
		}

		return Configuration.RESULT_SUCCESS;
	}

	public String getSearchPattern() {
		return searchPattern;
	}

	public void setSearchPattern(String searchPattern) {
		this.searchPattern = searchPattern;
	}

	public List<OxAuthHost> getHostList() {
		return hostList;
	}

}
