package org.gluu.oxtrust.action;

import java.io.IOException;
import java.io.Serializable;
import java.io.StringWriter;
import java.text.DateFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Date;
import java.util.Iterator;
import java.util.List;
import java.util.Properties;

import javax.mail.AuthenticationFailedException;
import javax.mail.MessagingException;

import org.apache.commons.beanutils.PropertyUtils;
import org.codehaus.jackson.JsonGenerationException;
import org.codehaus.jackson.JsonParseException;
import org.codehaus.jackson.map.JsonMappingException;
import org.codehaus.jackson.map.ObjectMapper;
import org.codehaus.jackson.type.TypeReference;
import org.gluu.oxtrust.ldap.cache.service.CacheRefreshConfiguration;
import org.gluu.oxtrust.ldap.service.ApplianceService;
import org.gluu.oxtrust.ldap.service.ImageService;
import org.gluu.oxtrust.ldap.service.OrganizationService;
import org.gluu.oxtrust.model.CustomAuthenticationConfig;
import org.gluu.oxtrust.model.GluuAppliance;
import org.gluu.oxtrust.model.GluuCustomPerson;
import org.gluu.oxtrust.model.GluuImage;
import org.gluu.oxtrust.model.GluuOrganization;
import org.gluu.oxtrust.model.LdapConfigurationModel;
import org.gluu.oxtrust.model.OxIDPAuthConf;
import org.gluu.oxtrust.model.SimpleCustomPropertiesListModel;
import org.gluu.oxtrust.model.SimpleCustomProperty;
import org.gluu.oxtrust.model.SimplePropertiesListModel;
import org.gluu.oxtrust.model.scim.ScimCustomAttributes;
import org.gluu.oxtrust.util.Configuration;
import org.gluu.oxtrust.util.MailUtils;
import org.gluu.oxtrust.util.Version;
import org.gluu.site.ldap.LDAPConnectionProvider;
import org.gluu.site.ldap.persistence.exception.LdapMappingException;
import org.jboss.seam.ScopeType;
import org.jboss.seam.annotations.Destroy;
import org.jboss.seam.annotations.In;
import org.jboss.seam.annotations.Logger;
import org.jboss.seam.annotations.Name;
import org.jboss.seam.annotations.Scope;
import org.jboss.seam.annotations.security.Restrict;
import org.jboss.seam.core.Events;
import org.jboss.seam.faces.FacesMessages;
import org.jboss.seam.international.StatusMessage.Severity;
import org.jboss.seam.international.StatusMessages;
import org.jboss.seam.log.Log;
import org.richfaces.event.FileUploadEvent;
import org.richfaces.model.UploadedFile;
import org.xdi.model.SimpleProperty;
import org.xdi.model.ldap.GluuLdapConfiguration;
import org.xdi.util.FileUtil;
import org.xdi.util.StringHelper;
import org.xdi.util.Util;
import org.xdi.util.properties.FileConfiguration;
import org.xdi.util.security.PropertiesDecrypter;
import org.xdi.util.security.StringEncrypter;
import org.xdi.util.security.StringEncrypter.EncryptionException;

/**
 * Action class for configuring application
 * 
 * @author Yuriy Movchan Date: 11.16.2010
 */
@Name("updateOrganizationAction")
@Scope(ScopeType.CONVERSATION)
@Restrict("#{identity.loggedIn}")
public class UpdateOrganizationAction implements SimplePropertiesListModel, SimpleCustomPropertiesListModel, LdapConfigurationModel, Serializable {

	private static final long serialVersionUID = -4470460481895022468L;

	private static final String CUSTOM_AUTHENTICATION_SCRIPT_PROPERTY_NAME = "script.__$__customAuthenticationScript__$__";
	private static final String CUSTOM_AUTHENTICATION_PROPERTY_PREFIX = "property.";

	@Logger
	private Log log;

	@In
	private StatusMessages statusMessages;

	@In
	private GluuCustomPerson currentPerson;

	@In
	private ImageService imageService;

	@In
	private OrganizationService organizationService;

	@In
	private ApplianceService applianceService;

	@In(required = false)
	private CacheRefreshConfiguration cacheRefreshConfiguration;

	@In
	private FacesMessages facesMessages;

	private GluuOrganization organization;

	protected GluuImage oldLogoImage, curLogoImage;
	protected String loginPageCustomMessage;
	protected String welcomePageCustomMessage;
	protected String welcomeTitleText;

	private String buildDate;
	private String buildNumber;
	private GluuLdapConfiguration ldapConfig;

	private GluuImage curFaviconImage, oldFaviconImage;

	private boolean certsMmanagePossible;

	private GluuAppliance appliance;

	private boolean cacheRefreshEnabled;

	private List<CustomAuthenticationConfig> customAuthenticationConfigs;

	private List<String> customAuthenticationConfigNames;

	private List<String> customAuthenticationConfigLevels;

	@Restrict("#{s:hasPermission('configuration', 'access')}")
	public String modify()  {
		String resultOrganization = modifyOrganization();
		String resultApplliance = modifyApplliance();

		if (StringHelper.equals(Configuration.RESULT_SUCCESS, resultOrganization)
				&& StringHelper.equals(Configuration.RESULT_SUCCESS, resultApplliance)) {
			return Configuration.RESULT_SUCCESS;
		} else {
			return Configuration.RESULT_FAILURE;
		}
	}

	private String modifyOrganization()  {
		if (this.organization != null) {
			return Configuration.RESULT_SUCCESS;
		}

		try {
			GluuOrganization tmpOrganization = organizationService.getOrganization();
			this.organization = new GluuOrganization();

			// Clone shared instance
			try {
				PropertyUtils.copyProperties(this.organization, tmpOrganization);
			} catch (Exception ex) {
				log.error("Failed to load organization", ex);
				this.organization = null;
			}
		} catch (LdapMappingException ex) {
			log.error("Failed to load organization", ex);
		}

		if (this.organization == null) {
			return Configuration.RESULT_FAILURE;
		}

		initLogoImage();
		initFaviconImage();

		this.loginPageCustomMessage = organizationService.getOrganizationCustomMessage(Configuration.CUSTOM_MESSAGE_LOGIN_PAGE);
		this.welcomePageCustomMessage = organizationService.getOrganizationCustomMessage(Configuration.CUSTOM_MESSAGE_WELCOME_PAGE);
		this.welcomeTitleText = organizationService.getOrganizationCustomMessage(Configuration.CUSTOM_MESSAGE_TITLE_TEXT);

		setCertsManagePossible(UpdateCertificatesAction.instance().prepareTempWorkspace());

		Configuration gluuConfig = Configuration.instance();
		this.cacheRefreshEnabled = gluuConfig.isCacheRefreshEnabled() && (cacheRefreshConfiguration != null);

		return Configuration.RESULT_SUCCESS;
	}

	private void initLogoImage() {
		this.oldLogoImage = imageService.getGluuImageFromXML(this.organization.getLogoImage());
		if (this.oldLogoImage != null) {
			this.oldLogoImage.setLogo(true);
		}
		this.curLogoImage = this.oldLogoImage;
	}

	private void initFaviconImage() {
		this.oldFaviconImage = imageService.getGluuImageFromXML(this.organization.getFaviconImage());
		this.curFaviconImage = this.oldFaviconImage;
	}

	@Restrict("#{s:hasPermission('configuration', 'access')}")
	public String save() {
		// Update organization
		try {
			saveLogoImage();
			saveFaviconImage();

			setCustomMessages();
			// saveLDAPConf();
			saveAuthConf();
			organizationService.updateOrganization(this.organization);
			applianceService.updateAppliance(appliance);

			if (isCertsManagePossible() && UpdateCertificatesAction.instance().updateCerts()) {
				facesMessages.add(Severity.WARN,
						"Certificates were updated and appliance service will be restarted. Please log in again in 5 minutes.");
			}
			Events.instance().raiseEvent(Configuration.EVENT_CLEAR_ORGANIZATION);
			/* Resolv.conf update */
			// saveDnsInformation(); // This will be handled by puppet.
			/* Resolv.conf update */
		} catch (LdapMappingException ex) {
			log.error("Failed to update organization", ex);
			facesMessages.add(Severity.ERROR, "Failed to update organization");
			return Configuration.RESULT_FAILURE;
		}

		reset();

		return modify();
	}

	public void reset() {
		this.customAuthenticationConfigNames = null;
		this.customAuthenticationConfigLevels = null;
	}

	// Resolv.conf file update will be handled by puppet.
	private void saveDnsInformation() {
		if (Util.empty(appliance.getApplianceDnsServer())) {
			appliance.setApplianceDnsServer("");
		}
		String filePath = "/etc/resolv.conf";
		FileUtil fileUtil = new FileUtil();
		String startTag = "##CONFIGURED-FROM-APPLIANCE-START##";
		String endTag = "##CONFIGURED-FROM-APPLIANCE-END##";
		String fileContent = FileUtil.readFile(filePath).toString();

		int startTagPositon = fileContent.indexOf(startTag);
		int endTagPositon = fileContent.indexOf(endTag);

		if (startTagPositon >= 0 && endTagPositon >= 0) {
			String startContent = fileContent.substring(0, startTagPositon - 1);
			String endContent = fileContent.substring(endTagPositon + endTag.length());
			fileContent = startContent + endContent;
		}

		fileContent = fileContent.replace(startTag, "");
		fileContent = fileContent.replace(endTag, "");

		String[] entries = appliance.getApplianceDnsServer().split(",");
		String servers = "";
		if (entries != null && entries.length > 0) {
			for (int i = 0; i < entries.length; i++) {
				entries[i] = entries[i].trim();
				servers += "\n" + entries[i];
			}
		}

		fileContent += "\n" + startTag + servers + "\n" + endTag;

		boolean success = fileUtil.writeLargeFile(filePath, 0, fileContent);
		if (!success) {
			log.error("Failed to update resolv.conf");
			facesMessages
					.add(Severity.ERROR,
							"Failed to update DNS Server information in /etc/resolv.conf. Please make sure that you have the permission to modify that file.");
		}
	}

	@Restrict("#{s:hasPermission('configuration', 'access')}")
	public String verifySmtpConfiguration() {
		log.info("HostName: " + appliance.getSmtpHost() + " Port: " + appliance.getSmtpPort() + " RequireSSL: " + appliance.isRequiresSsl()
				+ " RequireSSL: " + appliance.isRequiresAuthentication());
		log.info("UserName: " + appliance.getSmtpUserName() + " Password: " + appliance.getSmtpPasswordStr());

		try {
			MailUtils mail = new MailUtils(appliance.getSmtpHost(), appliance.getSmtpPort(), appliance.isRequiresSsl(),
					appliance.isRequiresAuthentication(), appliance.getSmtpUserName(), appliance.getSmtpPasswordStr());
			mail.sendMail(appliance.getSmtpFromName() + " <" + appliance.getSmtpFromEmailAddress() + ">",
					appliance.getSmtpFromEmailAddress(), "SMTP Server Configuration Verification",
					"SMTP Server Configuration Verification Successful.");
		} catch (AuthenticationFailedException ex) {
			log.error("SMTP Authentication Error: ", ex);
			return Configuration.RESULT_NO_PERMISSIONS;
		} catch (MessagingException ex) {
			log.error("SMTP Host Connection Error", ex);
			return Configuration.RESULT_FAILURE;
		}
		log.info("Connection Successful");
		return Configuration.RESULT_SUCCESS;
	}

	private String modifyApplliance() {
		if (this.appliance != null) {
			return Configuration.RESULT_SUCCESS;
		}
		try {
			this.appliance = applianceService.getAppliance();

			if (this.appliance == null) {
				return Configuration.RESULT_FAILURE;
			}

			this.customAuthenticationConfigs = new ArrayList<CustomAuthenticationConfig>();

			if (this.appliance.getOxIDPAuthentication() != null) {

				List<OxIDPAuthConf> idpConfs = new ArrayList<OxIDPAuthConf>();

				for (String oneConf : this.appliance.getOxIDPAuthentication()) {
					OxIDPAuthConf aConf = (OxIDPAuthConf) jsonToObject(oneConf, OxIDPAuthConf.class);
					idpConfs.add(aConf);
				}

				for (OxIDPAuthConf oneConf : idpConfs) {
					if (oneConf.getType().equalsIgnoreCase("ldap")) {
						this.ldapConfig = mapOldLdapConfig(oneConf);
					} else if (oneConf.getType().equalsIgnoreCase("auth")) {
						this.ldapConfig = mapLdapConfig(oneConf.getConfig());
					} else if (oneConf.getType().equalsIgnoreCase("customAuthentication")) {
						CustomAuthenticationConfig customAuthenticationConfig = mapCustomAuthentication(oneConf);
						this.customAuthenticationConfigs.add(customAuthenticationConfig);
					}
				}
			}

			if (this.ldapConfig == null) {
				this.ldapConfig = new GluuLdapConfiguration();
			}

			return Configuration.RESULT_SUCCESS;

		} catch (Exception ex) {
			log.error("an error occured", ex);

			return Configuration.RESULT_FAILURE;
		}
	}

	private void setCustomMessages() {
		String[][] customMessages = { { Configuration.CUSTOM_MESSAGE_LOGIN_PAGE, loginPageCustomMessage },
				{ Configuration.CUSTOM_MESSAGE_WELCOME_PAGE, welcomePageCustomMessage },
				{ Configuration.CUSTOM_MESSAGE_TITLE_TEXT, welcomeTitleText } };
		String[] customMessagesArray = organizationService.buildOrganizationCustomMessages(customMessages);
		if (customMessagesArray.length > 0) {
			this.organization.setCustomMessages(customMessagesArray);
		} else {
			this.organization.setCustomMessages(null);
		}
	}

	public boolean isCacheRefreshEnabled() {
		return cacheRefreshEnabled;
	}

	@Deprecated
	// Remove it after 2013/10/01
	private GluuLdapConfiguration mapOldLdapConfig(OxIDPAuthConf oneConf) {
		GluuLdapConfiguration ldapConfig = new GluuLdapConfiguration();
		ldapConfig.setServers(Arrays.asList(
				new SimpleProperty(oneConf.getFields().get(0).getValues().get(0) + ":" + oneConf.getFields().get(1).getValues().get(0))));
		ldapConfig.setBindDn(oneConf.getFields().get(2).getValues().get(0));
		ldapConfig.setBindPassword(oneConf.getFields().get(3).getValues().get(0));
		ldapConfig.setUseSSL(Boolean.valueOf(oneConf.getFields().get(4).getValues().get(0)));
		ldapConfig.setMaxConnections(3);
		ldapConfig.setConfigId("auth_ldap_server");
		ldapConfig.setEnabled(oneConf.getEnabled());
		return ldapConfig;
	}

	private GluuLdapConfiguration mapLdapConfig(String config) throws JsonParseException, JsonMappingException, IOException {
		return (GluuLdapConfiguration) jsonToObject(config, GluuLdapConfiguration.class);
	}

	private CustomAuthenticationConfig mapCustomAuthentication(OxIDPAuthConf oneConf) {
		CustomAuthenticationConfig customAuthenticationConfig = new CustomAuthenticationConfig();
		customAuthenticationConfig.setName(oneConf.getName());
		customAuthenticationConfig.setLevel(oneConf.getLevel());
		customAuthenticationConfig.setPriority(oneConf.getPriority());
		customAuthenticationConfig.setEnabled(oneConf.getEnabled());
		customAuthenticationConfig.setVersion(oneConf.getVersion());

		for (ScimCustomAttributes scimCustomAttributes : oneConf.getFields()) {
			if ((scimCustomAttributes.getValues() == null) || (scimCustomAttributes.getValues().size() == 0)) {
				continue;
			}

			String attrName = StringHelper.toLowerCase(scimCustomAttributes.getName());

			if (StringHelper.isEmpty(attrName)) {
				continue;
			}

			String value = scimCustomAttributes.getValues().get(0);

			if (attrName.startsWith(CUSTOM_AUTHENTICATION_PROPERTY_PREFIX)) {
				String key = scimCustomAttributes.getName().substring(CUSTOM_AUTHENTICATION_PROPERTY_PREFIX.length());
				SimpleCustomProperty property = new SimpleCustomProperty(key, value);
				customAuthenticationConfig.getCustomAuthenticationAttributes().add(property);
			} else if (StringHelper.equalsIgnoreCase(attrName, CUSTOM_AUTHENTICATION_SCRIPT_PROPERTY_NAME)) {
				customAuthenticationConfig.setCustomAuthenticationScript(value);
			}
		}

		return customAuthenticationConfig;
	}

	@Restrict("#{s:hasPermission('configuration', 'access')}")
	public String testLdapConnection() {

		try {
			FileConfiguration configuration = new FileConfiguration(Configuration.CONFIGURATION_FILE_LOCAL_LDAP_PROPERTIES_FILE);
			Properties properties = configuration.getProperties();
			properties.setProperty("bindDN", this.ldapConfig.getBindDn());
			properties.setProperty("bindPassword", this.ldapConfig.getBindPassword());
			properties.setProperty("servers", buildServersString(this.ldapConfig.getServers()));
			properties.setProperty("useSSL", Boolean.toString(this.ldapConfig.isUseSSL()));
			LDAPConnectionProvider connectionProvider = new LDAPConnectionProvider(PropertiesDecrypter.decryptProperties(properties));
			if (connectionProvider.isConnected()) {
				connectionProvider.closeConnectionPool();
				return Configuration.RESULT_SUCCESS;

			}
			connectionProvider.closeConnectionPool();
			return Configuration.RESULT_FAILURE;

		} catch (Exception ex) {
			log.error("Could not connect to LDAP", ex);
			return Configuration.RESULT_FAILURE;
		}
	}

	private String buildServersString(List<SimpleProperty> servers) {
		StringBuilder sb = new StringBuilder();

		if (servers == null) {
			return sb.toString();
		}
		
		boolean first = true;
		for (SimpleProperty server : servers) {
			if (first) {
				first = false;
			} else {
				sb.append(",");
			}

			sb.append(server.getValue());
		}

		return sb.toString();
	}

	@Restrict("#{s:hasPermission('configuration', 'access')}")
	public String getBuildDate() {
		if (this.buildDate != null) {
			return this.buildDate;
		}

		DateFormat formatter = new SimpleDateFormat("yyyyMMddhhmm");
		try {
			final Date date = formatter.parse(Version.GLUU_BUILD_DATE);
			this.buildDate = new SimpleDateFormat("hh:mm MMM dd yyyy").format(date) + " UTC";
		} catch (ParseException e) {
			log.error("Error formating date. Build process is invalid.", e);

		}
		return this.buildDate;
	}

	@Restrict("#{s:hasPermission('configuration', 'access')}")
	public String getBuildNumber() {
		if (this.buildNumber != null) {
			return this.buildNumber;
		}

		this.buildNumber = Version.GLUU_HUDSON_BUILDNO;
		return this.buildNumber;
	}

	@Restrict("#{s:hasPermission('configuration', 'access')}")
	public void cancel() throws Exception {
		cancelLogoImage();
		cancelFaviconImage();
	}

	public void setCustLogoImage(FileUploadEvent event) {
		UploadedFile uploadedFile = event.getUploadedFile();
		try {
			setCustLogoImageImpl(uploadedFile);
		} finally {
			try {
				uploadedFile.delete();
			} catch (IOException ex) {
				log.error("Failed to remove temporary image", ex);
			}
		}
	}

	private void setCustLogoImageImpl(UploadedFile uploadedFile) {
		removeLogoImage();

		GluuImage newLogoImage = imageService.constructImage(currentPerson, uploadedFile);
		newLogoImage.setStoreTemporary(true);
		newLogoImage.setLogo(true);
		try {
			if (imageService.createImageFiles(newLogoImage)) {
				this.curLogoImage = newLogoImage;
			}

			this.organization.setLogoImage(imageService.getXMLFromGluuImage(newLogoImage));
		} catch (Exception ex) {
			log.error("Failed to store icon image: '{0}'", ex, newLogoImage);
		}
	}

	public boolean isCustLogoImageExist() {
		return this.curLogoImage != null;
	}

	public void removeLogoImage() {
		cancelLogoImage();

		this.curLogoImage = null;
		this.organization.setLogoImage(null);
	}

	public void cancelLogoImage() {
		if ((this.curLogoImage != null) && this.curLogoImage.isStoreTemporary()) {
			try {
				imageService.deleteImage(this.curLogoImage);
			} catch (Exception ex) {
				log.error("Failed to delete temporary icon image: '{0}'", ex, this.curLogoImage);
			}
		}
	}

	public byte[] getLogoImageThumbData() throws Exception {
		if (this.curLogoImage != null) {
			return imageService.getThumImageData(this.curLogoImage);
		}

		return imageService.getBlankImageData();
	}

	public String getLogoImageSourceName() {
		if (this.curLogoImage != null) {
			return this.curLogoImage.getSourceName();
		}

		return null;
	}

	public void saveLogoImage() {
		// Remove old logo image if user upload new logo
		if ((this.oldLogoImage != null)
				&& ((this.curLogoImage == null) || !this.oldLogoImage.getUuid().equals(this.curLogoImage.getUuid()))) {
			try {
				imageService.deleteImage(this.oldLogoImage);
			} catch (Exception ex) {
				log.error("Failed to remove old icon image: '{0}'", ex, this.oldLogoImage);
			}
		}

		// Move added photo to persistent location
		if ((this.curLogoImage != null) && this.curLogoImage.isStoreTemporary()) {
			try {
				imageService.moveLogoImageToPersistentStore(this.curLogoImage);
				this.organization.setLogoImage(imageService.getXMLFromGluuImage(curLogoImage));
			} catch (Exception ex) {
				log.error("Failed to move new icon image to persistence store: '{0}'", ex, this.curLogoImage);
			}
		}

		this.oldLogoImage = this.curLogoImage;
	}

	public void setFaviconImage(FileUploadEvent event) {
		UploadedFile uploadedFile = event.getUploadedFile();
		try {
			setFaviconImageImpl(uploadedFile);
		} finally {
			try {
				uploadedFile.delete();
			} catch (IOException ex) {
				log.error("Failed to remove temporary image", ex);
			}
		}
	}

	public void setFaviconImageImpl(UploadedFile uploadedFile) {
		removeFaviconImage();

		GluuImage newFaviconImage = imageService.constructImage(currentPerson, uploadedFile);
		newFaviconImage.setStoreTemporary(true);
		newFaviconImage.setLogo(false);
		try {
			if (imageService.createFaviconImageFiles(newFaviconImage)) {
				this.curFaviconImage = newFaviconImage;
			}

			this.organization.setFaviconImage(imageService.getXMLFromGluuImage(newFaviconImage));
		} catch (Exception ex) {
			log.error("Failed to store favicon image: '{0}'", ex, newFaviconImage);
		}
	}

	public boolean isFaviconImageExist() {
		return this.curFaviconImage != null;
	}

	public void removeFaviconImage() {
		cancelFaviconImage();

		this.curFaviconImage = null;
		this.organization.setFaviconImage(null);
	}

	public void cancelFaviconImage() {
		if ((this.curFaviconImage != null) && this.curFaviconImage.isStoreTemporary()) {
			try {
				imageService.deleteImage(this.curFaviconImage);
			} catch (Exception ex) {
				log.error("Failed to delete temporary favicon image: '{0}'", ex, this.curFaviconImage);
			}
		}
	}

	public byte[] getFaviconImage() throws Exception {
		if (this.curFaviconImage != null) {
			return imageService.getThumImageData(this.curFaviconImage);
		}

		return imageService.getBlankImageData();
	}

	public String getFaviconImageSourceName() {
		if (this.curFaviconImage != null) {
			return this.curFaviconImage.getSourceName();
		}

		return null;
	}

	public void saveFaviconImage() {
		// Remove old favicon image if user upload new image
		if ((this.oldFaviconImage != null)
				&& ((this.curFaviconImage == null) || !this.oldFaviconImage.getUuid().equals(this.curFaviconImage.getUuid()))) {
			try {
				imageService.deleteImage(this.oldFaviconImage);
			} catch (Exception ex) {
				log.error("Failed to remove old favicon image: '{0}'", ex, this.oldFaviconImage);
			}
		}

		// Move added photo to persistent location
		if ((this.curFaviconImage != null) && this.curFaviconImage.isStoreTemporary()) {
			try {
				imageService.moveImageToPersistentStore(this.curFaviconImage);
			} catch (Exception ex) {
				log.error("Failed to move new favicon image to persistence store: '{0}'", ex, this.curFaviconImage);
			}
		}

		this.oldFaviconImage = this.curFaviconImage;
	}

	public void removeThemeColor() {
		this.organization.setThemeColor(null);
	}

	public GluuOrganization getOrganization() {
		return organization;
	}

	@Destroy
	public void destroy() throws Exception {
		// When user decided to leave form without saving we must remove added
		// logo image from disk
		cancel();
	}

	public void setCertsManagePossible(boolean managePossible) {
		this.certsMmanagePossible = managePossible;
	}

	public boolean isCertsManagePossible() {
		return certsMmanagePossible;
	}

	public String getLoginPageCustomMessage() {
		return loginPageCustomMessage;
	}

	public void setLoginPageCustomMessage(String loginPageCustomMessage) {
		this.loginPageCustomMessage = loginPageCustomMessage;
	}

	public String getWelcomePageCustomMessage() {
		return welcomePageCustomMessage;
	}

	public void setWelcomePageCustomMessage(String welcomePageCustomMessage) {
		this.welcomePageCustomMessage = welcomePageCustomMessage;
	}

	public String getWelcomeTitleText() {
		return welcomeTitleText;
	}

	public void setWelcomeTitleText(String welcomeTitleText) {
		this.welcomeTitleText = welcomeTitleText;
	}

	public GluuAppliance getAppliance() {
		return this.appliance;
	}

	public void setAppliance(GluuAppliance appliance) {
		this.appliance = appliance;
	}

	public GluuLdapConfiguration getLdapConfig() {
		return this.ldapConfig;
	}

	private Object jsonToObject(String json, Class<?> clazz) throws JsonParseException, JsonMappingException, IOException {
		ObjectMapper mapper = new ObjectMapper();
		Object clazzObject = mapper.readValue(json, clazz);
		return clazzObject;
	}

	private String objectToJson(Object obj) throws JsonGenerationException, JsonMappingException, IOException {
		ObjectMapper mapper = new ObjectMapper();
		return mapper.writeValueAsString(obj);
	}

	private List<OxIDPAuthConf> jsonTooxIDPAuthConfList(String json) throws Exception {

		ObjectMapper mapper = new ObjectMapper();
		List<OxIDPAuthConf> clazzObject = mapper.readValue(json, new TypeReference<List<OxIDPAuthConf>>() {
		});
		return clazzObject;
	}

	private String getJSONString(Object ob) throws JsonGenerationException, JsonMappingException, IOException {
		StringWriter sw = new StringWriter();
		ObjectMapper mapper = new ObjectMapper();
		mapper.writeValue(sw, ob);
		return sw.toString();
	}

	/*
	 * public void saveLDAPConf(){ try{
	 * this.props.saveLDAPConfig(this.ldapConfig);
	 * 
	 * }catch(Exception ex){ log.error("An Error occured ", ex);
	 * 
	 * return;
	 * 
	 * } }
	 */

	public void saveAuthConf() {
		try {
			List<OxIDPAuthConf> cleanIdpConfs = new ArrayList<OxIDPAuthConf>();

			for (CustomAuthenticationConfig customAuthenticationConfig : this.customAuthenticationConfigs) {
				customAuthenticationConfig.setVersion(customAuthenticationConfig.getVersion() + 1);
				OxIDPAuthConf oxIDPAuthConf = customAuthenticationToIdp(customAuthenticationConfig);
				oxIDPAuthConf.setVersion(oxIDPAuthConf.getVersion());
				cleanIdpConfs.add(oxIDPAuthConf);
			}

			List<String> idpAuthStrings = new ArrayList<String>();
			for (OxIDPAuthConf oneConf : cleanIdpConfs) {
				String oneString = getJSONString(oneConf);
				idpAuthStrings.add(oneString);
			}

			OxIDPAuthConf authConfig = new OxIDPAuthConf();
			authConfig.setType("auth");
			authConfig.setName(this.ldapConfig.getConfigId());
			authConfig.setEnabled(this.ldapConfig.isEnabled());
			authConfig.setConfig(objectToJson(this.ldapConfig));

			idpAuthStrings.add(getJSONString(authConfig));

			this.appliance.setOxIDPAuthentication(idpAuthStrings);
			this.applianceService.updateAppliance(this.appliance);
		} catch (Exception ex) {
			log.error("An Error occured ", ex);

			return;
		}
	}

	private OxIDPAuthConf customAuthenticationToIdp(CustomAuthenticationConfig customAuthenticationConfig) {
		OxIDPAuthConf oxIDP = new OxIDPAuthConf();
		oxIDP.setEnabled(customAuthenticationConfig.isEnabled());

		oxIDP.setType("customAuthentication");
		oxIDP.setName(customAuthenticationConfig.getName());
		oxIDP.setLevel(customAuthenticationConfig.getLevel());
		oxIDP.setPriority(customAuthenticationConfig.getPriority());
		oxIDP.setVersion(customAuthenticationConfig.getVersion());

		List<ScimCustomAttributes> properties = new ArrayList<ScimCustomAttributes>();

		for (SimpleCustomProperty customProperty : customAuthenticationConfig.getCustomAuthenticationAttributes()) {
			ScimCustomAttributes property = new ScimCustomAttributes();

			property.setName(CUSTOM_AUTHENTICATION_PROPERTY_PREFIX + customProperty.getValue1());
			property.getValues().add(customProperty.getValue2());

			properties.add(property);
		}

		ScimCustomAttributes property = new ScimCustomAttributes();
		property.setName(CUSTOM_AUTHENTICATION_SCRIPT_PROPERTY_NAME);
		property.getValues().add(customAuthenticationConfig.getCustomAuthenticationScript());

		properties.add(property);

		oxIDP.setFields(properties);

		return oxIDP;
	}

	public List<CustomAuthenticationConfig> getCustomAuthenticationConfigs() {
		return customAuthenticationConfigs;
	}

	public void addCustomAuthenticationConfig() {
		this.customAuthenticationConfigs.add(new CustomAuthenticationConfig());
	}

	public void removeCustomAuthenticationConfig(CustomAuthenticationConfig removeCustomAuthenticationConfig) {
		for (Iterator<CustomAuthenticationConfig> iterator = this.customAuthenticationConfigs.iterator(); iterator.hasNext();) {
			CustomAuthenticationConfig customAuthenticationConfig = iterator.next();
			if (System.identityHashCode(removeCustomAuthenticationConfig) == System.identityHashCode(customAuthenticationConfig)) {
				iterator.remove();
				return;
			}
		}
	}

	public List<String> getCustomAuthenticationConfigNames() {
		if (this.customAuthenticationConfigNames == null) {
			this.customAuthenticationConfigNames = new ArrayList<String>();
			for (CustomAuthenticationConfig customAuthenticationConfig : this.customAuthenticationConfigs) {
				String name = customAuthenticationConfig.getName();
				if (StringHelper.isEmpty(name)) {
					continue;
				}

				this.customAuthenticationConfigNames.add(customAuthenticationConfig.getName());
			}
		}

		return this.customAuthenticationConfigNames;
	}

	public List<String> getCustomAuthenticationConfigLevels() {
		if (this.customAuthenticationConfigLevels == null) {
			this.customAuthenticationConfigLevels = new ArrayList<String>();
			for (CustomAuthenticationConfig customAuthenticationConfig : this.customAuthenticationConfigs) {
				String level = Integer.toString(customAuthenticationConfig.getLevel());

				if (!this.customAuthenticationConfigLevels.contains(level)) {
					this.customAuthenticationConfigLevels.add(level);
				}
			}
		}

		return this.customAuthenticationConfigLevels;
	}

	@Override
	public void setActiveLdapConfig(GluuLdapConfiguration activeLdapConfig) {
	}

	@Override
	public void addLdapConfig(List<GluuLdapConfiguration> ldapConfigList) {
	}

	@Override
	public void removeLdapConfig(List<GluuLdapConfiguration> ldapConfigList, GluuLdapConfiguration removeLdapConfig) {
	}

	@Override
	public void addItemToSimpleProperties(List<SimpleProperty> simpleProperties) {
		if (simpleProperties != null) {
			simpleProperties.add(new SimpleProperty(""));
		}
	}

	@Override
	public void removeItemFromSimpleProperties(List<SimpleProperty> simpleProperties, SimpleProperty simpleProperty) {
		if (simpleProperties != null) {
			simpleProperties.remove(simpleProperty);
		}
	}

	@Override
	public void addItemToSimpleCustomProperties(List<SimpleCustomProperty> simpleCustomProperties) {
		if (simpleCustomProperties != null) {
			simpleCustomProperties.add(new SimpleCustomProperty("", ""));
		}
	}

	@Override
	public void removeItemFromSimpleCustomProperties(List<SimpleCustomProperty> simpleCustomProperties,
			SimpleCustomProperty simpleCustomProperty) {
		if (simpleCustomProperties != null) {
			simpleCustomProperties.remove(simpleCustomProperty);
		}
	}

	public void updateLdapBindPassword() {
		String encryptedLdapBindPassword = null;
		try {
			encryptedLdapBindPassword = StringEncrypter.defaultInstance().encrypt(this.ldapConfig.getBindPassword());
		} catch (EncryptionException ex) {
			log.error("Failed to encrypt LDAP bind password", ex);
		}

		this.ldapConfig.setBindPassword(encryptedLdapBindPassword);
	}
}
