package org.gluu.oxtrust.ws.rs;

import java.net.URI;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import javax.servlet.http.HttpServletRequest;
import javax.ws.rs.core.Response;

import org.gluu.oxtrust.ldap.service.ClientService;
import org.gluu.oxtrust.ldap.service.PersonService;
import org.gluu.oxtrust.ldap.service.SecurityService;
import org.gluu.oxtrust.model.GluuCustomPerson;
import org.gluu.oxtrust.model.GluuUserRole;
import org.gluu.oxtrust.model.OxAuthClient;
import org.gluu.oxtrust.model.OxAuthCustomClient;
import org.gluu.oxtrust.model.association.ClientAssociation;
import org.gluu.oxtrust.model.association.PersonAssociation;
import org.gluu.oxtrust.model.scim.Error;
import org.gluu.oxtrust.model.scim.Errors;
import org.gluu.oxtrust.util.Configuration;
import org.gluu.oxtrust.util.MapperUtil;
import org.gluu.site.ldap.persistence.exception.EntryPersistenceException;
import org.jboss.seam.annotations.In;
import org.jboss.seam.annotations.Logger;
import org.jboss.seam.annotations.Name;
import org.jboss.seam.contexts.Contexts;
import org.jboss.seam.log.Log;

@Name("clientAssociationWebService")
public class ClientAssociationWebServiceImpl implements ClientAssociationWebService {

	@Logger
	private Log log;

	@In
	private PersonService personService;

	@In
	private ClientService clientService;

	@Override
	public Response getAssociatedClients(HttpServletRequest request, String uid) throws Exception {

		personService = PersonService.instance();

		boolean authorized = getAuthorizedUser();

		if (!authorized) {
			return getErrorResponse("User isn't authorized", Response.Status.FORBIDDEN.getStatusCode());
		}

		try {

			GluuCustomPerson gluuPerson = personService.getPersonByInum(uid);
			if (gluuPerson == null) {
				// sets HTTP status code 404 Not Found
				return getErrorResponse("Resource " + uid + " not found", Response.Status.NOT_FOUND.getStatusCode());
			}

			PersonAssociation personAssociation = MapperUtil.map(gluuPerson, null);

			URI location = new URI("/ClientAssociation/User/" + uid);

			return Response.ok(personAssociation).location(location).build();
		} catch (EntryPersistenceException ex) {
			log.error("Exception: ", ex);
			return getErrorResponse("Resource " + uid + " not found", Response.Status.NOT_FOUND.getStatusCode());
		} catch (Exception ex) {
			log.error("Exception: ", ex);
			return getErrorResponse("Unexpected processing error, please check the input parameters",
					Response.Status.INTERNAL_SERVER_ERROR.getStatusCode());
		}
	}

	@Override
	public Response getAssociatedPersons(HttpServletRequest request, String cid) throws Exception {
		clientService = ClientService.instance();

		boolean authorized = getAuthorizedUser();

		if (!authorized) {
			return getErrorResponse("User isn't authorized", Response.Status.FORBIDDEN.getStatusCode());
		}

		try {

			log.info("getting the client");
			OxAuthClient client = clientService.getClientByInum(cid);

			if (client == null) {
				// sets HTTP status code 404 Not Found
				return getErrorResponse("Resource " + cid + " not found", Response.Status.NOT_FOUND.getStatusCode());
			}

			log.info("mapping client attributes");
			ClientAssociation clientAssociation = MapperUtil.map(client, null);
			log.info("getting URL");
			URI location = new URI("/ClientAssociation/Client/" + cid);
			log.info("returning response");
			return Response.ok(clientAssociation).location(location).build();

		} catch (EntryPersistenceException ex) {
			log.error("Exception: ", ex);
			return getErrorResponse("Resource " + cid + " not found", Response.Status.NOT_FOUND.getStatusCode());
		} catch (Exception ex) {
			log.error("Exception: ", ex);
			return getErrorResponse("Unexpected processing error, please check the input parameters",
					Response.Status.INTERNAL_SERVER_ERROR.getStatusCode());
		}

	}

	@Override
	public Response createAssociation(HttpServletRequest request, PersonAssociation personAssociation) throws Exception {

		personService = PersonService.instance();
		clientService = ClientService.instance();

		boolean authorized = getAuthorizedUser();

		if (!authorized) {
			return getErrorResponse("User isn't authorized", Response.Status.FORBIDDEN.getStatusCode());
		}

		try {
			log.info("creating an instance of gluuCustomperson");
			GluuCustomPerson person = personService.getPersonByInum(personAssociation.getUserAssociation().replaceAll(" ", ""));

			log.info("setting AssociatedClientDNs");
			List<String> cleanCDNList = new ArrayList<String>();
			for (String dn : personAssociation.getEntryAssociations()) {
				cleanCDNList.add(dn.replaceAll(" ", ""));
			}

			person.setAssociatedClient(cleanCDNList);

			log.info("updating person");

			personService.updatePerson(person);

			log.info("setting user in clients");
			for (String clientDn : personAssociation.getEntryAssociations()) {
				log.info("getting a client");
				OxAuthCustomClient client = clientService.getClientByAttributeCustom(Configuration.instance()
						.getClientAssociationAttribute(), clientDn.replaceAll(" ", ""));

				log.info("the inum of the client ", client.getInum());

				log.info("checking if the list is empty");
				boolean isAPDNsEmpty = client.getAttributes("associatedPerson") == null;

				log.info("instantiating a new arraylist");

				List<String> listOfpersons = new ArrayList<String>();
				log.info("getting AssociatedPersonDN");
				if (!isAPDNsEmpty) {
					listOfpersons = new ArrayList(Arrays.asList(client.getAttributes("associatedPerson")));
					/*
					 * for(String dn :
					 * client.getAttributes("associatedPersonDN")){ if(dn !=
					 * null && !dn.equalsIgnoreCase("")){listOfpersons.add(dn);}
					 * }
					 */

				}
				log.info("getting persons dn");
				String personInum = personAssociation.getUserAssociation().replaceAll(" ", "");

				if (isAPDNsEmpty || !listOfpersons.contains(personInum)) {
					log.info("adding person");
					listOfpersons.add(personInum);
				}

				String[] arrayOfpersons = new String[listOfpersons.size()];
				for (int i = 0; i < listOfpersons.size(); i++) {
					arrayOfpersons[i] = listOfpersons.get(i);
				}
				log.info("setting list of AssociatedPersonDns");
				client.setAttribute("associatedPerson", arrayOfpersons);
				log.info("Updating client");
				clientService.updateCustomClient(client);
			}

			String uri = "/ClientAssociation/Associate/" + person.getInum();
			log.info("returning response");
			return Response.created(URI.create(uri)).entity(personAssociation).build();

		} catch (Exception ex) {
			log.error("Failed to add Association", ex);
			// log.info("Failed to add Association" , ex);
			return getErrorResponse("Unexpected processing error, please check the input parameters",
					Response.Status.INTERNAL_SERVER_ERROR.getStatusCode());
		}

	}

	@Override
	public Response deleteAssociation(HttpServletRequest request, PersonAssociation personAssociation) throws Exception {

		personService = PersonService.instance();
		clientService = ClientService.instance();

		boolean authorized = getAuthorizedUser();

		if (!authorized) {
			return getErrorResponse("User isn't authorized", Response.Status.FORBIDDEN.getStatusCode());
		}

		try {
			log.info("Creating an instance of GluuCustomPerson");
			GluuCustomPerson person = personService.getPersonByInum(personAssociation.getUserAssociation().replaceAll(" ", ""));
			log.info("getting a list of clientDNs");
			List<String> listClientDNs = new ArrayList<String>();

			boolean isACDNsEmpty = person.getAssociatedClient() == null;

			if (!isACDNsEmpty) {
				for (String dn : person.getAssociatedClient()) {
					log.info("isACDNsEmpty = false");
					if (dn != null && !dn.equalsIgnoreCase("")) {
						listClientDNs.add(dn.replaceAll(" ", ""));
					}

				}
			}

			log.info("getting a list of clean clients");

			List<String> cleanPACDNs = new ArrayList<String>();
			for (String dn : personAssociation.getEntryAssociations()) {
				if (dn != null && !dn.equalsIgnoreCase("")) {
					cleanPACDNs.add(dn);
				}
			}

			log.info("removing clientdns");

			for (String clientdn : cleanPACDNs) {

				if (listClientDNs.contains(clientdn)) {
					listClientDNs.remove(clientdn);
				}

			}

			log.info("geting a cleanlist");

			List<String> cleanList = new ArrayList<String>();
			for (String cDn : listClientDNs) {
				if (cDn != null && !cDn.equalsIgnoreCase("")) {
					cleanList.add(cDn);
				}
			}
			log.info("setting AssociatedClientDNs");
			if (cleanList.size() < 1) {
				person.setAssociatedClient(null);
			} else {
				person.setAssociatedClient(cleanList);
			}

			log.info("Updating person");

			personService.updatePerson(person);

			log.info("deleting user dn from clients");

			List<String> EntryAssociations = new ArrayList<String>();

			for (String dn : personAssociation.getEntryAssociations()) {
				if (dn != null && !dn.equalsIgnoreCase("")) {
					EntryAssociations.add(dn.replaceAll(" ", ""));
				}
			}

			for (String clientDn : EntryAssociations) {
				log.info("getting a client");

				OxAuthCustomClient client = clientService.getClientByAttributeCustom(Configuration.instance()
						.getClientAssociationAttribute(), clientDn.replaceAll(" ", ""));
				// String[] personDNS =
				// client.getAttributes("associatedPersonDN");
				log.info("checking if the associatedPerson is empty");
				log.info("client dn : ", client.getDn());
				boolean isAPDNsEmpty = client.getAttributes("associatedPerson") == null;
				log.info("new ArrayList");
				List<String> list = new ArrayList<String>();
				if (!isAPDNsEmpty) {
					log.info("!isAPDNsEmpty");
					// list =
					// Arrays.asList(client.getAttributes("associatedPersonDN"));
					for (int i = 0; i < client.getAttributes("associatedPerson").length; i++) {
						if (client.getAttributes("associatedPerson")[i] != null
								&& !client.getAttributes("associatedPerson")[i].equalsIgnoreCase("")) {
							list.add(client.getAttributes("associatedPerson")[i]);
						}
					}
					/*
					 * for(String dn : client.getAssociatedPersonDNs()){ if(dn
					 * != null && !dn.equalsIgnoreCase("")){list.add(dn);} }
					 */
				}
				log.info("getting personDN");
				String personInum = personAssociation.getUserAssociation().replaceAll(" ", "");

				if (list.contains(personInum)) {
					log.info("removing person's dn");
					list.remove(personInum);
				}

				log.info("Creating a clean list");

				List<String> cleanPersonList = new ArrayList<String>();
				for (String cDn : list) {
					if (cDn != null && cDn.equalsIgnoreCase("")) {
						cleanPersonList.add(cDn);
					}
				}
				log.info("Setting AssociatedPersonDNs");
				if (cleanPersonList.size() < 1) {
					String[] nullArray = null;
					client.setAttribute("associatedPerson", nullArray);
				} else {
					String[] arrayPersonDns = new String[cleanPersonList.size()];
					for (int i = 0; i < cleanPersonList.size(); i++) {
						arrayPersonDns[i] = cleanPersonList.get(i);
					}
					client.setAttribute("associatedPerson", arrayPersonDns);
				}
				clientService.updateCustomClient(client);

			}
			log.info("returning result;");

			return Response.ok().build();
		} catch (Exception ex) {
			log.info("Exception: ", ex);
			log.error("Exception: ", ex);
			return getErrorResponse("Unexpected processing error, please check the input parameters",
					Response.Status.INTERNAL_SERVER_ERROR.getStatusCode());
		}
	}

	private boolean getAuthorizedUser() {
		try {
			GluuCustomPerson authUser = (GluuCustomPerson) Contexts.getSessionContext().get(Configuration.CURRENT_PERSON);
			SecurityService securityService = SecurityService.instance();

			GluuUserRole[] userRoles = securityService.getUserRoles(authUser);
			for (GluuUserRole role : userRoles) {

				if (role.getRoleName().equalsIgnoreCase("MANAGER") || role.getRoleName().equalsIgnoreCase("OWNER")) {

					return true;

				}
			}
			return false;
		} catch (Exception ex) {
			log.error("Exception: ", ex);
			return false;
		}

	}

	private Response getErrorResponse(String errMsg, int statusCode) {
		Errors errors = new Errors();
		Error error = new org.gluu.oxtrust.model.scim.Error(errMsg, statusCode, "");
		errors.getErrors().add(error);
		return Response.status(statusCode).entity(errors).build();
	}

}
