package org.gluu.oxtrust.ldap.service.uma;

import java.io.Serializable;
import java.util.List;

import org.gluu.oxtrust.model.uma.ResourceSet;
import org.gluu.oxtrust.util.Configuration;
import org.gluu.site.ldap.persistence.LdapEntryManager;
import org.hibernate.annotations.common.util.StringHelper;
import org.jboss.seam.Component;
import org.jboss.seam.ScopeType;
import org.jboss.seam.annotations.AutoCreate;
import org.jboss.seam.annotations.In;
import org.jboss.seam.annotations.Logger;
import org.jboss.seam.annotations.Name;
import org.jboss.seam.annotations.Scope;
import org.jboss.seam.log.Log;
import org.xdi.ldap.model.SimpleBranch;
import org.xdi.util.INumGenerator;

import com.unboundid.ldap.sdk.Filter;

/**
 * Provides operations with resource sets
 * 
 * @author Yuriy Movchan Date: 12/06/2012
 */
@Scope(ScopeType.STATELESS)
@Name("resourceSetService")
@AutoCreate
public class ResourceSetService implements Serializable {

	private static final long serialVersionUID = -1537567020929600777L;

	@In
	private LdapEntryManager ldapEntryManager;

	@Logger
	private Log log;

	public void addBranch(String hostDn) {
		SimpleBranch branch = new SimpleBranch();
		branch.setOrganizationalUnitName("resource_sets");
		branch.setDn(getDnForResourceSet(hostDn, null));

		ldapEntryManager.persist(branch);
	}

	/**
	 * Add new resource set entry
	 * 
	 * @param group
	 *            Group
	 */
	public void addResourceSet(ResourceSet resourceSet) {
		ldapEntryManager.persist(resourceSet);
	}

	/**
	 * Update resource set entry
	 * 
	 * @param group
	 *            Group
	 */
	public void updateResourceSet(ResourceSet resourceSet) {
		ldapEntryManager.merge(resourceSet);
	}

	/**
	 * Remove resource set entry
	 * 
	 * @param group
	 *            Group
	 */
	public void removeResourceSet(ResourceSet resourceSet) {
		ldapEntryManager.remove(resourceSet);
	}

	/**
	 * Get all resource sets
	 * 
	 * @return List of resource sets
	 */
	public List<ResourceSet> getAllResourceSets(String hostDn, String... ldapReturnAttributes) {
		return ldapEntryManager.findEntries(getDnForResourceSet(hostDn, null), ResourceSet.class, ldapReturnAttributes, null);
	}

	/**
	 * Search resource sets by pattern
	 * 
	 * @param pattern
	 *            Pattern
	 * @param sizeLimit
	 *            Maximum count of results
	 * @return List of resource sets
	 */
	public List<ResourceSet> searchResourceSets(String hostDn, String pattern, int sizeLimit) {
		String[] targetArray = new String[] { pattern };
		Filter oxIdFilter = Filter.createSubstringFilter("oxId", null, targetArray, null);
		Filter displayNameFilter = Filter.createSubstringFilter(Configuration.displayName, null, targetArray, null);
		Filter searchFilter = Filter.createORFilter(oxIdFilter, displayNameFilter);

		List<ResourceSet> result = ldapEntryManager.findEntries(getDnForResourceSet(hostDn, null), ResourceSet.class, searchFilter,
				sizeLimit);

		return result;
	}

	/**
	 * Get resource sets by example
	 * 
	 * @param resourceSet
	 *            ResourceSet
	 * @return List of ResourceSets which conform example
	 */
	public List<ResourceSet> findResourceSets(ResourceSet resourceSet) {
		return ldapEntryManager.findEntries(resourceSet);
	}

	/**
	 * Get resource sets by Id
	 * 
	 * @param id
	 *            Id
	 * @return List of ResourceSets which specified id
	 */
	public List<ResourceSet> findResourceSetsById(String hostDn, String id) {
		return ldapEntryManager.findEntries(getDnForResourceSet(hostDn, null), ResourceSet.class, Filter.createEqualityFilter("oxId", id));
	}

	public boolean containsBranch(String hostDn) {
		return ldapEntryManager.contains(SimpleBranch.class, getDnForResourceSet(hostDn, null));
	}

	/**
	 * Check if LDAP server contains resource set with specified attributes
	 * 
	 * @return True if resource set with specified attributes exist
	 */
	public boolean containsResourceSet(ResourceSet resourceSet) {
		return ldapEntryManager.contains(resourceSet);
	}

	public void prepareResourceSetBranch(String hostDn) {
		// Create resource set branch if needed
		if (!containsBranch(hostDn)) {
			addBranch(hostDn);
		}
	}

	/**
	 * Get resource set by DN
	 * 
	 * @param DN
	 *            Resource set DN
	 * @return Resource set
	 */
	public ResourceSet getResourceSetByDn(String dn) {
		return ldapEntryManager.find(ResourceSet.class, dn);
	}

	/**
	 * Generate new inum for resource set
	 * 
	 * @return New inum for resource set
	 * @throws Exception
	 */
	public String generateInumForNewResourceSet(String hostDn, String baseInum) {
		ResourceSet resourceSet = new ResourceSet();
		String newInum = null;
		do {
			newInum = generateInumForNewGroupImpl(hostDn, baseInum);
			String newDn = getDnForResourceSet(hostDn, newInum);
			resourceSet.setDn(newDn);
		} while (ldapEntryManager.contains(resourceSet));

		return newInum;
	}

	/**
	 * Generate new inum for resource set
	 * 
	 * @return New inum for resource set
	 */

	private String generateInumForNewGroupImpl(String hostDn, String baseInum) {
		return baseInum + Configuration.inumDelimiter + INumGenerator.generate(2);
	}

	/**
	 * Build DN string for resource set
	 */
	public String getDnForResourceSet(String hostDn, String inum) {
		if (StringHelper.isEmpty(inum)) {
			return String.format("ou=resource_sets,%s", hostDn);
		}

		return String.format("inum=%s,ou=resource_sets,%s", inum, hostDn);
	}

	/**
	 * Get ResourceSetService instance
	 * 
	 * @return ResourceSetService instance
	 */
	public static ResourceSetService instance() {
		return (ResourceSetService) Component.getInstance(ResourceSetService.class);
	}

}
