package org.gluu.oxtrust.ldap.service.uma;

import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

import javax.ws.rs.core.Response;

import org.gluu.oxtrust.ldap.service.HostService;
import org.gluu.oxtrust.model.uma.HostClient;
import org.gluu.oxtrust.model.uma.HostToken;
import org.jboss.seam.ScopeType;
import org.jboss.seam.annotations.AutoCreate;
import org.jboss.seam.annotations.Create;
import org.jboss.seam.annotations.In;
import org.jboss.seam.annotations.Logger;
import org.jboss.seam.annotations.Name;
import org.jboss.seam.annotations.Scope;
import org.jboss.seam.annotations.Startup;
import org.jboss.seam.log.Log;
import org.xdi.oxauth.client.TokenClient;
import org.xdi.oxauth.client.TokenResponse;
import org.xdi.oxauth.model.uma.MetadataConfiguration;
import org.xdi.oxauth.model.uma.UmaConstants;
import org.xdi.oxauth.model.uma.UmaScopeType;
import org.xdi.util.StringHelper;
import org.xdi.util.security.StringEncrypter;
import org.xdi.util.security.StringEncrypter.EncryptionException;

/**
 * Get and hold host tokens
 * 
 * @author Yuriy Movchan
 * @version 0.1, 12/08/2012
 */
@Name("hostTokenService")
@AutoCreate
@Startup
@Scope(ScopeType.APPLICATION)
public class HostTokenService {

	private final static int REMAIN_TOKEN_LIFE_TIME = 1 * 60 * 1000; // 5
																		// minutes

	private Map<String, HostToken> umaHostClients;

	@Logger
	private Log log;

	@In
	private ClientUmaWebService clientUmaWebService;

	@In
	private HostService hostService;

	@In
	private HostClientService hostClientService;

	@Create
	public void init() {
		this.umaHostClients = new ConcurrentHashMap<String, HostToken>();
	}

	public HostToken getHostToken(String hostInum, String hostClientInum) {
		String hostDn = hostService.getDnForHost(hostInum);
		HostClient hostClient = hostClientService.getHostClientByDn(hostClientService.getDnForHostClient(hostDn, hostClientInum));
		if (hostClient == null) {
			log.error("Failed to load hostClient entry by inum: '{0}'", hostClientInum);
			return null;
		}

		String issuer = hostClient.getId();

		if (umaHostClients.containsKey(issuer)) {
			// Return cached host token
			HostToken hostToken = umaHostClients.get(issuer);
			if (hostToken.getExpiredAt() + REMAIN_TOKEN_LIFE_TIME < System.currentTimeMillis()) {
				return hostToken;
			}
		}

		MetadataConfiguration metadataConfiguration = clientUmaWebService.obtainUmaMetaDataConfigurationByIssuer(issuer);
		if (metadataConfiguration == null) {
			log.error("Failed to load UMA AM meta data configuration");
			return null;
		}

		// Obtain PAT
		String encodedClientSecret = hostClient.getEncodedClientSecret();
		String clientPassword = null;
		if (StringHelper.isNotEmpty(encodedClientSecret)) {
			try {
				clientPassword = StringEncrypter.defaultInstance().decrypt(encodedClientSecret);
			} catch (EncryptionException ex) {
				log.error("Failed to decrypt client password '{0}'", encodedClientSecret);
				return null;
			}
		}

		TokenClient tokenClient = new TokenClient(metadataConfiguration.getTokenEndpoint());
		TokenResponse response = tokenClient.execClientCredentialsGrant(UmaScopeType.PROTECTION.getValue(), hostClient.getClientId(),
				clientPassword);
		if (response == null) {
			log.error("UMA Host '{0}' returns invalid response", metadataConfiguration.getIssuer());
			return null;
		}

		if (Response.Status.OK.getStatusCode() != response.getStatus()) {
			log.error("UMA Host returns invalid token response: status = '{0}', token = '{1}', expiresIn = '{2}', scope = '{3}'",
					response.getStatus(), response.getAccessToken(), response.getExpiresIn(), response.getScope());
			return null;
		}

		if (!UmaScopeType.PROTECTION.getValue().equalsIgnoreCase(response.getScope())) {
			log.warn("According to allowed scopes this client can access UMA AM protection API");

			return null;
		}

		// Store PAT token
		HostToken hostToken = new HostToken(hostInum, hostClientInum, metadataConfiguration, response.getAccessToken(),
				System.currentTimeMillis() + response.getExpiresIn());
		this.umaHostClients.put(issuer, hostToken);

		return hostToken;
	}

}