package org.gluu.oxtrust.ldap.service;

import java.io.File;
import java.io.Serializable;
import java.util.Properties;

import org.gluu.oxtrust.exception.PythonException;
import org.gluu.site.ldap.persistence.util.ReflectHelper;
import org.jboss.seam.Component;
import org.jboss.seam.ScopeType;
import org.jboss.seam.annotations.AutoCreate;
import org.jboss.seam.annotations.Logger;
import org.jboss.seam.annotations.Name;
import org.jboss.seam.annotations.Scope;
import org.jboss.seam.log.Log;
import org.python.core.PyObject;
import org.python.util.PythonInterpreter;
import org.xdi.util.StringHelper;

/**
 * Provides operations with python module
 * 
 * @author Yuriy Movchan Date: 12.15.2010
 */
@Scope(ScopeType.APPLICATION)
@Name("pythonService")
@AutoCreate
public class PythonService implements Serializable {

	private static final long serialVersionUID = 3398422090669045605L;

	@Logger
	private Log log;

	/*
	 * Initialize singleton instance during startup
	 */
	public void initPythonInterpreter() {
		try {
			PythonInterpreter.initialize(getPreProperties(), getPostProperties(), null);
		} catch (Exception ex) {
			log.error("Failed to initialize PythonInterpreter", ex);
		}
	}

	private Properties getPreProperties() {
		Properties props = System.getProperties();
		Properties clonedProps = (Properties) props.clone();
		clonedProps.setProperty("java.class.path", ".");
		clonedProps.setProperty("java.library.path", "");

		return clonedProps;
	}

	private Properties getPostProperties() {
		Properties props = getPreProperties();

		String catalinaTmpFolder = System.getProperty("java.io.tmpdir") + File.separator + "python" + File.separator + "cachedir";
		props.setProperty("python.cachedir", catalinaTmpFolder);

		return props;
	}

	@SuppressWarnings("unchecked")
	public <T> T loadPythonScript(String scriptName, String scriptPythonType, Class<T> scriptJavaType, PyObject[] constructorArgs)
			throws PythonException {
		if (StringHelper.isEmpty(scriptName)) {
			return null;
		}

		PythonInterpreter interpret = new PythonInterpreter();
		try {
			interpret.execfile(scriptName);
		} catch (Exception ex) {
			throw new PythonException(String.format("Failed to load python file '%s'", scriptName), ex);
		}

		PyObject scriptPythonTypeObject = interpret.get(scriptPythonType);
		if (scriptPythonTypeObject == null) {
			return null;
		}

		PyObject scriptPythonTypeClass;
		try {
			scriptPythonTypeClass = scriptPythonTypeObject.__call__(constructorArgs);
		} catch (Exception ex) {
			throw new PythonException(String.format("Failed to initialize python class '%s'", scriptPythonType), ex);
		}

		Object scriptJavaClass = scriptPythonTypeClass.__tojava__(scriptJavaType);
		if (!ReflectHelper.assignableFrom(scriptJavaClass.getClass(), scriptJavaType)) {
			return null;

		}

		return (T) scriptJavaClass;
	}

	/**
	 * Get pythonService instance
	 * 
	 * @return PythonService instance
	 */
	public static PythonService instance() {
		return (PythonService) Component.getInstance(PythonService.class);
	}

}
