package org.gluu.oxtrust.ldap.service;

import java.io.Serializable;
import java.util.List;

import org.gluu.oxtrust.model.OxAuthHost;
import org.gluu.oxtrust.util.Configuration;
import org.gluu.site.ldap.persistence.LdapEntryManager;
import org.hibernate.annotations.common.util.StringHelper;
import org.jboss.seam.Component;
import org.jboss.seam.ScopeType;
import org.jboss.seam.annotations.AutoCreate;
import org.jboss.seam.annotations.In;
import org.jboss.seam.annotations.Logger;
import org.jboss.seam.annotations.Name;
import org.jboss.seam.annotations.Scope;
import org.jboss.seam.log.Log;
import org.xdi.ldap.model.SimpleBranch;
import org.xdi.util.INumGenerator;

import com.unboundid.ldap.sdk.Filter;

/**
 * Provides operations with host registrations
 * 
 * @author Yuriy Movchan Date: 12/07/2012
 */
@Scope(ScopeType.STATELESS)
@Name("hostService")
@AutoCreate
public class HostService implements Serializable {

	private static final long serialVersionUID = -3537567020929600777L;

	@In
	private LdapEntryManager ldapEntryManager;

	@In
	private OrganizationService organizationService;

	@Logger
	private Log log;

	public boolean contains(String hostDn) {
		return ldapEntryManager.contains(OxAuthHost.class, hostDn);
	}

	public void addBranch() {
		SimpleBranch branch = new SimpleBranch();
		branch.setOrganizationalUnitName("hosts");
		branch.setDn(getDnForHost(null));

		ldapEntryManager.persist(branch);
	}

	/**
	 * Add new host registration entry
	 * 
	 * @param group
	 *            Group
	 */
	public void addHost(OxAuthHost host) {
		ldapEntryManager.persist(host);
	}

	/**
	 * Update host registration entry
	 * 
	 * @param group
	 *            Group
	 */
	public void updateHost(OxAuthHost host) {
		ldapEntryManager.merge(host);
	}

	/**
	 * Remove host registration entry
	 * 
	 * @param group
	 *            Group
	 */
	public void removeHost(OxAuthHost host) {
		ldapEntryManager.remove(host);
	}

	/**
	 * Get all host registrations
	 * 
	 * @return List of host registrations
	 */
	public List<OxAuthHost> getAllHosts(String... ldapReturnAttributes) {
		return ldapEntryManager.findEntries(getDnForHost(null), OxAuthHost.class, ldapReturnAttributes, null);
	}

	/**
	 * Search host registrations by pattern
	 * 
	 * @param pattern
	 *            Pattern
	 * @param sizeLimit
	 *            Maximum count of results
	 * @return List of host registrations
	 */
	public List<OxAuthHost> searchHosts(String pattern, int sizeLimit) {
		String[] targetArray = new String[] { pattern };
		Filter oxIdFilter = Filter.createSubstringFilter("oxId", null, targetArray, null);
		Filter displayNameFilter = Filter.createSubstringFilter(Configuration.displayName, null, targetArray, null);
		Filter searchFilter = Filter.createORFilter(oxIdFilter, displayNameFilter);

		List<OxAuthHost> result = ldapEntryManager.findEntries(getDnForHost(null), OxAuthHost.class, searchFilter, sizeLimit);

		return result;
	}

	/**
	 * Get host registrations by example
	 * 
	 * @param host
	 *            Host
	 * @return List of Host which conform example
	 */
	public List<OxAuthHost> findHosts(OxAuthHost host) {
		return ldapEntryManager.findEntries(host);
	}

	public boolean containsBranch() {
		return ldapEntryManager.contains(SimpleBranch.class, getDnForHost(null));
	}

	/**
	 * Check if LDAP server contains host registration with specified attributes
	 * 
	 * @return True if host registration with specified attributes exist
	 */
	public boolean containsHost(OxAuthHost host) {
		return ldapEntryManager.contains(host);
	}

	public void prepareHostBranch() {
		// Create host registration branch if needed
		if (!containsBranch()) {
			addBranch();
		}
	}

	/**
	 * Get host registration by DN
	 * 
	 * @param DN
	 *            Scope description DN
	 * @return Scope description
	 */
	public OxAuthHost getHostByDn(String dn) {
		return ldapEntryManager.find(OxAuthHost.class, dn);
	}

	/**
	 * Generate new inum for host registration
	 * 
	 * @return New inum for host registration
	 * @throws Exception
	 */
	public String generateInumForNewHost() {
		String orgInum = organizationService.getInumForOrganization();
		OxAuthHost host = new OxAuthHost();
		String newInum = null;
		do {
			newInum = generateInumForNewGroupImpl(orgInum);
			String newDn = getDnForHost(newInum);
			host.setDn(newDn);
		} while (ldapEntryManager.contains(host));

		return newInum;
	}

	/**
	 * Generate new inum for host registration
	 * 
	 * @return New inum for host registration
	 */

	private String generateInumForNewGroupImpl(String baseInum) {
		return baseInum + Configuration.inumDelimiter + INumGenerator.generate(2);
	}

	/**
	 * Build DN string for host registration
	 */
	public String getDnForHost(String inum) {
		String baseDn = organizationService.getDnForOrganization();
		if (StringHelper.isEmpty(inum)) {
			return String.format("ou=hosts,%s", baseDn);
		}

		return String.format("inum=%s,ou=hosts,%s", inum, baseDn);
	}

	/**
	 * Get HostService instance
	 * 
	 * @return HostService instance
	 */
	public static HostService instance() {
		return (HostService) Component.getInstance(HostService.class);
	}

}
