package org.gluu.oxtrust.action.uma;

import java.io.IOException;
import java.io.Serializable;
import java.util.List;

import org.gluu.oxtrust.ldap.service.HostService;
import org.gluu.oxtrust.ldap.service.ImageService;
import org.gluu.oxtrust.ldap.service.uma.ScopeDescriptionService;
import org.gluu.oxtrust.model.GluuCustomPerson;
import org.gluu.oxtrust.model.GluuImage;
import org.gluu.oxtrust.model.uma.ScopeDescription;
import org.gluu.oxtrust.util.Configuration;
import org.gluu.site.ldap.persistence.exception.LdapMappingException;
import org.jboss.seam.ScopeType;
import org.jboss.seam.annotations.Destroy;
import org.jboss.seam.annotations.In;
import org.jboss.seam.annotations.Logger;
import org.jboss.seam.annotations.Name;
import org.jboss.seam.annotations.Scope;
import org.jboss.seam.annotations.security.Restrict;
import org.jboss.seam.log.Log;
import org.richfaces.event.FileUploadEvent;
import org.richfaces.model.UploadedFile;
import org.xdi.util.StringHelper;

/**
 * Action class for view and update scope description
 * 
 * @author Yuriy Movchan Date: 11/21/2012
 */
@Name("updateScopeDescriptionAction")
@Scope(ScopeType.CONVERSATION)
@Restrict("#{identity.loggedIn}")
public class UpdateScopeDescriptionAction implements Serializable {

	private static final long serialVersionUID = 6180729281938167478L;

	@Logger
	private Log log;

	@In
	protected GluuCustomPerson currentPerson;

	@In
	protected HostService hostService;

	@In
	protected ScopeDescriptionService scopeDescriptionService;

	@In
	protected ImageService imageService;

	private String scopeId;
	private String hostInum, hostDn;

	private ScopeDescription scopeDescription;

	private GluuImage oldIconImage, curIconImage;

	private boolean update;

	@Restrict("#{s:hasPermission('uma', 'access')}")
	public String modify() {
		this.update = StringHelper.isNotEmpty(this.scopeId);

		try {
			this.hostDn = hostService.getDnForHost(this.hostInum);
			if (!hostService.contains(this.hostDn)) {
				return Configuration.RESULT_FAILURE;
			}

			scopeDescriptionService.prepareScopeDescriptionBranch(this.hostDn);
		} catch (Exception ex) {
			log.error("Failed to initialize form", ex);
			return Configuration.RESULT_FAILURE;
		}

		if (update) {
			return update();
		} else {
			return add();
		}
	}

	private String add() {
		if (this.scopeDescription != null) {
			return Configuration.RESULT_SUCCESS;
		}

		this.scopeDescription = new ScopeDescription();

		return Configuration.RESULT_SUCCESS;
	}

	private String update() {
		if (this.scopeDescription != null) {
			return Configuration.RESULT_SUCCESS;
		}

		log.debug("Loading UMA scope description '{0}' for host '{1}'", this.scopeId, this.hostInum);
		try {
			List<ScopeDescription> scopeDescriptions = scopeDescriptionService.findScopeDescriptionsById(this.hostDn, this.scopeId);
			if (scopeDescriptions.size() != 1) {
				log.error("Failed to find scope description '{0}'. Found: '{1}'", this.scopeId, scopeDescriptions.size());
				return Configuration.RESULT_FAILURE;
			}

			this.scopeDescription = scopeDescriptions.get(0);
		} catch (LdapMappingException ex) {
			log.error("Failed to find scope description '{0}'", ex, this.scopeId);
			return Configuration.RESULT_FAILURE;
		}

		if (this.scopeDescription == null) {
			log.error("Scope description is null");
			return Configuration.RESULT_FAILURE;
		}

		initIconImage();

		return Configuration.RESULT_SUCCESS;
	}

	@Restrict("#{s:hasPermission('uma', 'access')}")
	public void cancel() {
		cancelIconImage();
	}

	@Restrict("#{s:hasPermission('uma', 'access')}")
	public String save() {
		if (this.update) {
			scopeDescription.setRev(String.valueOf(StringHelper.toInteger(scopeDescription.getRev(), 0) + 1));
			// Update scope description
			try {
				scopeDescriptionService.updateScopeDescription(this.scopeDescription);
			} catch (LdapMappingException ex) {
				log.error("Failed to update scope description '{0}'", ex, this.scopeDescription.getId());
				return Configuration.RESULT_FAILURE;
			}
		} else {
			// Check if scope description with this name already exist
			ScopeDescription exampleScopeDescription = new ScopeDescription();
			exampleScopeDescription.setDn(scopeDescriptionService.getDnForScopeDescription(this.hostDn, null));
			exampleScopeDescription.setId(scopeDescription.getId());
			if (scopeDescriptionService.containsScopeDescription(exampleScopeDescription)) {
				return Configuration.RESULT_DUPLICATE;
			}

			// Prepare score description
			this.scopeDescription.setRev(String.valueOf(0));

			String inum = scopeDescriptionService.generateInumForNewScopeDescription(this.hostDn, this.hostInum);
			String scopeDescriptionDn = scopeDescriptionService.getDnForScopeDescription(this.hostDn, inum);

			this.scopeDescription.setInum(inum);
			this.scopeDescription.setDn(scopeDescriptionDn);
			this.scopeDescription.setOwner(currentPerson.getDn());

			// Save scope description
			try {
				scopeDescriptionService.addScopeDescription(this.scopeDescription);
			} catch (LdapMappingException ex) {
				log.error("Failed to add new scope description '{0}'", ex, this.scopeDescription.getId());
				return Configuration.RESULT_FAILURE;
			}

			this.update = true;
		}

		saveIconImage();

		log.debug("Scope description were {0} successfully", (this.update ? "added" : "updated"));
		return Configuration.RESULT_SUCCESS;
	}

	@Restrict("#{s:hasPermission('uma', 'access')}")
	public String delete() {
		if (update) {
			// Remove scope description
			try {
				scopeDescriptionService.removeScopeDescription(this.scopeDescription);
				return Configuration.RESULT_SUCCESS;
			} catch (LdapMappingException ex) {
				log.error("Failed to remove scope description {0}", ex, this.scopeDescription.getId());
			}
		}

		return Configuration.RESULT_FAILURE;
	}

	@Restrict("#{s:hasPermission('uma', 'access')}")
	public void removeIconImage() {
		cancelIconImage();

		this.curIconImage = null;
		this.scopeDescription.setIcon(null);
	}

	@Destroy
	public void destroy() throws Exception {
		// When user decided to leave form without saving we must remove added
		// logo image from disk
		cancel();
	}

	@Restrict("#{s:hasPermission('uma', 'access')}")
	public void setIconImage(FileUploadEvent event) {
		UploadedFile uploadedFile = event.getUploadedFile();
		try {
			setIconImageImpl(uploadedFile);
		} finally {
			try {
				uploadedFile.delete();
			} catch (IOException ex) {
				log.error("Failed to remove temporary image", ex);
			}
		}
	}

	public byte[] getIconImageThumbData() {
		if (this.curIconImage != null) {
			return imageService.getThumIconData(this.curIconImage);
		}

		return imageService.getBlankImageData();
	}

	private void initIconImage() {
		this.oldIconImage = this.curIconImage = imageService.getGluuImageFromXML(this.scopeDescription.getIcon());
	}

	private void saveIconImage() {
		// Remove old icon image if user upload new image
		if ((this.oldIconImage != null)
				&& ((this.curIconImage == null) || !this.oldIconImage.getUuid().equals(this.curIconImage.getUuid()))) {
			try {
				imageService.deleteImage(this.oldIconImage);
			} catch (Exception ex) {
				log.error("Failed to remove old icon image: '{0}'", ex, this.oldIconImage);
			}
		}

		// Move added photo to persistent location
		if ((this.curIconImage != null) && this.curIconImage.isStoreTemporary()) {
			try {
				imageService.moveImageToPersistentStore(this.curIconImage);
			} catch (Exception ex) {
				log.error("Failed to move new icon image to persistence store: '{0}'", ex, this.curIconImage);
			}
		}
	}

	private void setIconImageImpl(UploadedFile uploadedFile) {
		removeIconImage();

		GluuImage newIcon = imageService.constructImage(currentPerson, uploadedFile);
		newIcon.setStoreTemporary(true);
		try {
			if (imageService.createImageFiles(newIcon, 16, 16)) {
				this.curIconImage = newIcon;
			}

			this.scopeDescription.setIcon(imageService.getXMLFromGluuImage(this.curIconImage));
		} catch (Exception ex) {
			log.error("Failed to store icon image: '{0}'", ex, newIcon);
		}
	}

	public String getIconImageSourceName() {
		if (this.curIconImage != null) {
			return this.curIconImage.getSourceName();
		}

		return null;
	}

	private void cancelIconImage() {
		if ((this.curIconImage != null) && this.curIconImage.isStoreTemporary()) {
			try {
				imageService.deleteImage(this.curIconImage);
			} catch (Exception ex) {
				log.error("Failed to delete temporary icon image: '{0}'", ex, this.curIconImage);
			}
		}
	}

	public boolean isUpdate() {
		return update;
	}

	public String getScopeId() {
		return scopeId;
	}

	public void setScopeId(String scopeId) {
		this.scopeId = scopeId;
	}

	public ScopeDescription getScopeDescription() {
		return scopeDescription;
	}

	public boolean isIconExist() {
		return this.curIconImage != null;
	}

	public String getHostInum() {
		return hostInum;
	}

	public void setHostInum(String hostInum) {
		this.hostInum = hostInum;
	}

}
