package org.gluu.oxtrust.action.uma;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Set;

import org.gluu.oxtrust.ldap.service.GroupService;
import org.gluu.oxtrust.ldap.service.HostService;
import org.gluu.oxtrust.ldap.service.LookupService;
import org.gluu.oxtrust.ldap.service.uma.ResourceSetService;
import org.gluu.oxtrust.ldap.service.uma.ResourceSetUmaSyncService;
import org.gluu.oxtrust.ldap.service.uma.ScopeDescriptionService;
import org.gluu.oxtrust.model.DisplayNameEntry;
import org.gluu.oxtrust.model.GluuCustomPerson;
import org.gluu.oxtrust.model.GluuGroup;
import org.gluu.oxtrust.model.uma.ResourceSet;
import org.gluu.oxtrust.model.uma.ScopeDescription;
import org.gluu.oxtrust.util.Configuration;
import org.gluu.site.ldap.persistence.exception.LdapMappingException;
import org.jboss.seam.ScopeType;
import org.jboss.seam.annotations.Destroy;
import org.jboss.seam.annotations.In;
import org.jboss.seam.annotations.Logger;
import org.jboss.seam.annotations.Name;
import org.jboss.seam.annotations.Scope;
import org.jboss.seam.annotations.security.Restrict;
import org.jboss.seam.log.Log;
import org.xdi.util.StringHelper;
import org.xdi.util.Util;

/**
 * Action class for view and update resource sets
 * 
 * @author Yuriy Movchan Date: 11/21/2012
 */
@Name("updateResourceSetAction")
@Scope(ScopeType.CONVERSATION)
@Restrict("#{identity.loggedIn}")
public class UpdateResourceSetAction implements Serializable {

	private static final long serialVersionUID = 9180729281938167478L;

	@Logger
	private Log log;

	@In
	protected GluuCustomPerson currentPerson;

	@In
	private HostService hostService;

	@In
	private ResourceSetService resourceSetService;

	@In
	private ScopeDescriptionService scopeDescriptionService;

	@In
	private GroupService groupService;

	@In
	private LookupService lookupService;

	@In
	private ResourceSetUmaSyncService resourceSetUmaSyncService;

	private String resourceId;
	private String hostInum, hostDn;

	private ResourceSet resourceSet;
	private List<DisplayNameEntry> scopes;
	private List<DisplayNameEntry> groups;
	private List<String> resources;

	private List<ScopeDescription> availableScopes;
	private String searchAvailableScopePattern, oldSearchAvailableScopePattern;

	private List<GluuGroup> availableGroups;
	private String searchAvailableGroupPattern, oldSearchAvailableGroupPattern;

	private String newResource;

	private boolean update;

	@Restrict("#{s:hasPermission('uma', 'access')}")
	public String modify() {
		if (this.resourceSet != null) {
			return Configuration.RESULT_SUCCESS;
		}

		this.update = StringHelper.isNotEmpty(this.resourceId);

		try {
			this.hostDn = hostService.getDnForHost(this.hostInum);
			if (!hostService.contains(this.hostDn)) {
				return Configuration.RESULT_FAILURE;
			}

			resourceSetService.prepareResourceSetBranch(this.hostDn);
		} catch (Exception ex) {
			log.error("Failed to initialize form", ex);
			return Configuration.RESULT_FAILURE;
		}

		if (update) {
			return update();
		} else {
			return add();
		}
	}

	private String add() {
		this.resourceSet = new ResourceSet();

		this.scopes = new ArrayList<DisplayNameEntry>();
		this.groups = new ArrayList<DisplayNameEntry>();
		this.resources = new ArrayList<String>();

		return Configuration.RESULT_SUCCESS;
	}

	private String update() {
		log.debug("Loading UMA resource set '{0}' for host '{1}'", this.resourceId, this.hostInum);
		try {
			List<ResourceSet> resourceSets = resourceSetService.findResourceSetsById(this.hostDn, this.resourceId);
			if (resourceSets.size() != 1) {
				log.error("Failed to find resource set '{0}'. Found: '{1}'", this.resourceId, resourceSets.size());
				return Configuration.RESULT_FAILURE;
			}

			this.resourceSet = resourceSets.get(0);
		} catch (LdapMappingException ex) {
			log.error("Failed to find resource set '{0}'", ex, this.resourceId);
			return Configuration.RESULT_FAILURE;
		}

		if (this.resourceSet == null) {
			log.error("Resource set is null");
			return Configuration.RESULT_FAILURE;
		}

		this.scopes = getScopeDisplayNameEntries();
		this.groups = getGroupDisplayNameEntries();

		if (this.resourceSet.getResources() == null) {
			this.resources = new ArrayList<String>();
		} else {
			this.resources = new ArrayList<String>(this.resourceSet.getResources());
		}

		return Configuration.RESULT_SUCCESS;
	}

	@Restrict("#{s:hasPermission('uma', 'access')}")
	public void cancel() {
	}

	@Restrict("#{s:hasPermission('uma', 'access')}")
	public String save() {
		updateScopes();
		updateGroups();
		updateResources();

		boolean updateResourceSet = this.update;

		if (this.update) {
			resourceSet.setRev(String.valueOf(StringHelper.toInteger(resourceSet.getRev(), 0) + 1));
			// Update resource set
			try {
				resourceSetService.updateResourceSet(this.resourceSet);
			} catch (LdapMappingException ex) {
				log.error("Failed to update resource set '{0}'", ex, this.resourceSet.getId());
				return Configuration.RESULT_FAILURE;
			}
		} else {
			// Check if resource set with this name already exist
			ResourceSet exampleResourceSet = new ResourceSet();
			exampleResourceSet.setDn(resourceSetService.getDnForResourceSet(this.hostDn, null));
			exampleResourceSet.setId(resourceSet.getId());
			if (resourceSetService.containsResourceSet(exampleResourceSet)) {
				return Configuration.RESULT_DUPLICATE;
			}

			// Prepare score description
			this.resourceSet.setRev(String.valueOf(0));

			String inum = resourceSetService.generateInumForNewResourceSet(this.hostDn, this.hostInum);
			String resourceSetDn = resourceSetService.getDnForResourceSet(this.hostDn, inum);

			this.resourceSet.setInum(inum);
			this.resourceSet.setDn(resourceSetDn);
			this.resourceSet.setOwner(currentPerson.getDn());

			// Save resource set
			try {
				resourceSetService.addResourceSet(this.resourceSet);
			} catch (LdapMappingException ex) {
				log.error("Failed to add new resource set '{0}'", ex, this.resourceSet.getId());
				return Configuration.RESULT_FAILURE;
			}

			this.update = true;
		}

		boolean syncResult = resourceSetUmaSyncService.syncResourceSet(hostInum, this.resourceSet, updateResourceSet);
		if (!syncResult) {
			return "sync_error";
		}

		log.debug("Resource set were {0} successfully", (this.update ? "added" : "updated"));
		return Configuration.RESULT_SUCCESS;
	}

	@Restrict("#{s:hasPermission('uma', 'access')}")
	public String delete() {
		if (update) {
			// Remove resource set
			try {
				resourceSetService.removeResourceSet(this.resourceSet);
				return Configuration.RESULT_SUCCESS;
			} catch (LdapMappingException ex) {
				log.error("Failed to remove resource set {0}", ex, this.resourceSet.getId());
			}
		}

		return Configuration.RESULT_FAILURE;
	}

	@Destroy
	public void destroy() throws Exception {
		cancel();
	}

	@Restrict("#{s:hasPermission('uma', 'access')}")
	public void searchAvailableScopes() {
		if (Util.equals(this.oldSearchAvailableScopePattern, this.searchAvailableScopePattern)) {
			return;
		}

		try {
			this.availableScopes = scopeDescriptionService.searchScopeDescriptions(this.hostDn, this.searchAvailableScopePattern, 100);
			this.oldSearchAvailableScopePattern = this.searchAvailableScopePattern;

			selectAddedScopes();
		} catch (Exception ex) {
			log.error("Failed to find scopes", ex);
		}
	}

	public void selectAddedScopes() {
		Set<String> addedScopeInums = getAddedScopesInums();

		for (ScopeDescription availableScope : this.availableScopes) {
			availableScope.setSelected(addedScopeInums.contains(availableScope.getInum()));
		}
	}

	@Restrict("#{s:hasPermission('uma', 'access')}")
	public void acceptSelectScopes() {
		Set<String> addedScopeInums = getAddedScopesInums();

		for (ScopeDescription availableScope : this.availableScopes) {
			if (availableScope.isSelected() && !addedScopeInums.contains(availableScope.getInum())) {
				addScope(availableScope);
			}
		}

	}

	private Set<String> getAddedScopesInums() {
		Set<String> addedScopeInums = new HashSet<String>();

		if (this.availableScopes == null) {
			return addedScopeInums;
		}

		for (DisplayNameEntry scope : this.scopes) {
			addedScopeInums.add(scope.getInum());
		}
		return addedScopeInums;
	}

	@Restrict("#{s:hasPermission('uma', 'access')}")
	public void cancelSelectScopes() {
	}

	@Restrict("#{s:hasPermission('uma', 'access')}")
	public void addScope(ScopeDescription scope) {
		DisplayNameEntry oneScope = new DisplayNameEntry(scope.getDn(), scope.getId(), scope.getDisplayName());
		this.scopes.add(oneScope);
	}

	@Restrict("#{s:hasPermission('uma', 'access')}")
	public void removeScope(String id) throws Exception {
		if (StringHelper.isEmpty(id)) {
			return;
		}

		String removeScopeId = scopeDescriptionService.getDnForScopeDescription(this.hostDn, id);

		for (Iterator<DisplayNameEntry> iterator = this.scopes.iterator(); iterator.hasNext();) {
			DisplayNameEntry oneScope = iterator.next();
			if (removeScopeId.equals(oneScope.getDn())) {
				iterator.remove();
				break;
			}
		}
	}

	private void updateScopes() {
		if ((this.scopes == null) || (this.scopes.size() == 0)) {
			this.resourceSet.setAssociatedScopes(null);
			return;
		}

		List<String> tmpScopes = new ArrayList<String>();
		for (DisplayNameEntry scope : this.scopes) {
			tmpScopes.add(scope.getDn());
		}

		this.resourceSet.setAssociatedScopes(tmpScopes);
	}

	private List<DisplayNameEntry> getScopeDisplayNameEntries() {
		List<DisplayNameEntry> result = new ArrayList<DisplayNameEntry>();
		List<DisplayNameEntry> tmp = lookupService.getDisplayNameEntries(
				scopeDescriptionService.getDnForScopeDescription(this.hostDn, null), this.resourceSet.getAssociatedScopes());
		if (tmp != null) {
			result.addAll(tmp);
		}

		return result;
	}

	@Restrict("#{s:hasPermission('uma', 'access')}")
	public void searchAvailableGroups() {
		if (Util.equals(this.oldSearchAvailableGroupPattern, this.searchAvailableGroupPattern)) {
			return;
		}

		try {
			this.availableGroups = groupService.searchGroups(this.searchAvailableGroupPattern, 100);
			this.oldSearchAvailableGroupPattern = this.searchAvailableGroupPattern;

			selectAddedGroups();
		} catch (Exception ex) {
			log.error("Failed to find groups", ex);
		}
	}

	public void selectAddedGroups() {
		Set<String> addedGroupInums = getAddedGroupsInums();

		for (GluuGroup availableGroup : this.availableGroups) {
			availableGroup.setSelected(addedGroupInums.contains(availableGroup.getInum()));
		}
	}

	@Restrict("#{s:hasPermission('uma', 'access')}")
	public void acceptSelectGroups() {
		Set<String> addedGroupInums = getAddedGroupsInums();

		for (GluuGroup availableGroup : this.availableGroups) {
			if (availableGroup.isSelected() && !addedGroupInums.contains(availableGroup.getInum())) {
				addGroup(availableGroup);
			}
		}
	}

	private Set<String> getAddedGroupsInums() {
		Set<String> addedGroupInums = new HashSet<String>();

		if (this.availableGroups == null) {
			return addedGroupInums;
		}

		for (DisplayNameEntry group : this.groups) {
			addedGroupInums.add(group.getInum());
		}

		return addedGroupInums;
	}

	@Restrict("#{s:hasPermission('uma', 'access')}")
	public void cancelSelectGroups() {
	}

	@Restrict("#{s:hasPermission('uma', 'access')}")
	public void addGroup(GluuGroup group) {
		DisplayNameEntry oneGroup = new DisplayNameEntry(group.getDn(), group.getInum(), group.getDisplayName());
		this.groups.add(oneGroup);
	}

	@Restrict("#{s:hasPermission('uma', 'access')}")
	public void removeGroup(String inum) throws Exception {
		if (StringHelper.isEmpty(inum)) {
			return;
		}

		String removeGroupInum = groupService.getDnForGroup(inum);

		for (Iterator<DisplayNameEntry> iterator = this.groups.iterator(); iterator.hasNext();) {
			DisplayNameEntry oneGroup = iterator.next();
			if (removeGroupInum.equals(oneGroup.getDn())) {
				iterator.remove();
				break;
			}
		}
	}

	private void updateGroups() {
		if ((this.groups == null) || (this.groups.size() == 0)) {
			this.resourceSet.setAssociatedGroups(null);
			return;
		}

		List<String> tmpGroups = new ArrayList<String>();
		for (DisplayNameEntry group : this.groups) {
			tmpGroups.add(group.getDn());
		}

		this.resourceSet.setAssociatedGroups(tmpGroups);
	}

	private List<DisplayNameEntry> getGroupDisplayNameEntries() {
		List<DisplayNameEntry> result = new ArrayList<DisplayNameEntry>();
		List<DisplayNameEntry> tmp = lookupService.getDisplayNameEntries(groupService.getDnForGroup(null),
				this.resourceSet.getAssociatedGroups());
		if (tmp != null) {
			result.addAll(tmp);
		}

		return result;
	}

	private void updateResources() {
		if ((this.resources == null) || (this.resources.size() == 0)) {
			this.resourceSet.setAssociatedGroups(null);
			return;
		}

		this.resourceSet.setResources(this.resources);
	}

	@Restrict("#{s:hasPermission('uma', 'access')}")
	public void acceptResource() {
		if (!this.resources.contains(this.newResource)) {
			this.resources.add(this.newResource);
		}

		cancelResource();
	}

	@Restrict("#{s:hasPermission('uma', 'access')}")
	public void cancelResource() {
		this.newResource = null;
	}

	@Restrict("#{s:hasPermission('uma', 'access')}")
	public void removeResource(String resource) {
		if (StringHelper.isNotEmpty(resource)) {
			this.resources.remove(resource);
		}
	}

	public boolean isUpdate() {
		return update;
	}

	public String getResourceId() {
		return resourceId;
	}

	public void setResourceId(String resourceId) {
		this.resourceId = resourceId;
	}

	public ResourceSet getResourceSet() {
		return resourceSet;
	}

	public List<DisplayNameEntry> getScopes() {
		return scopes;
	}

	public List<ScopeDescription> getAvailableScopes() {
		return availableScopes;
	}

	public String getSearchAvailableScopePattern() {
		return searchAvailableScopePattern;
	}

	public void setSearchAvailableScopePattern(String searchAvailableScopePattern) {
		this.searchAvailableScopePattern = searchAvailableScopePattern;
	}

	public List<DisplayNameEntry> getGroups() {
		return groups;
	}

	public List<GluuGroup> getAvailableGroups() {
		return this.availableGroups;
	}

	public String getSearchAvailableGroupPattern() {
		return searchAvailableGroupPattern;
	}

	public void setSearchAvailableGroupPattern(String searchAvailableGroupPattern) {
		this.searchAvailableGroupPattern = searchAvailableGroupPattern;
	}

	public String getNewResource() {
		return newResource;
	}

	public void setNewResource(String newResource) {
		this.newResource = newResource;
	}

	public String getHostInum() {
		return hostInum;
	}

	public void setHostInum(String hostInum) {
		this.hostInum = hostInum;
	}

	public List<String> getResources() {
		return resources;
	}

}
