package org.gluu.oxtrust.action.uma;

import java.io.Serializable;
import java.util.Date;

import org.gluu.oxtrust.ldap.service.HostService;
import org.gluu.oxtrust.ldap.service.uma.ClientUmaWebService;
import org.gluu.oxtrust.ldap.service.uma.HostClientRegistrationService;
import org.gluu.oxtrust.ldap.service.uma.HostClientService;
import org.gluu.oxtrust.model.GluuCustomPerson;
import org.gluu.oxtrust.model.uma.HostClient;
import org.gluu.oxtrust.util.Configuration;
import org.gluu.site.ldap.persistence.exception.LdapMappingException;
import org.jboss.seam.ScopeType;
import org.jboss.seam.annotations.Destroy;
import org.jboss.seam.annotations.In;
import org.jboss.seam.annotations.Logger;
import org.jboss.seam.annotations.Name;
import org.jboss.seam.annotations.Scope;
import org.jboss.seam.annotations.security.Restrict;
import org.jboss.seam.log.Log;
import org.xdi.oxauth.model.uma.MetadataConfiguration;
import org.xdi.util.StringHelper;

/**
 * Action class for view and update host clients
 * 
 * @author Yuriy Movchan Date: 11/21/2012
 */
@Name("updateHostClientAction")
@Scope(ScopeType.CONVERSATION)
@Restrict("#{identity.loggedIn}")
public class UpdateHostClientAction implements Serializable {

	private static final long serialVersionUID = 9180729281938167478L;

	@Logger
	private Log log;

	@In
	protected GluuCustomPerson currentPerson;

	@In
	private HostService hostService;

	@In
	private HostClientService hostClientService;

	@In
	private HostClientRegistrationService hostClientRegistrationService;

	@In
	private ClientUmaWebService clientUmaWebService;

	private String clientId;
	private String hostInum, hostDn;

	private HostClient client;

	private boolean update;

	@Restrict("#{s:hasPermission('uma', 'access')}")
	public String modify() {
		if (this.client != null) {
			return Configuration.RESULT_SUCCESS;
		}

		this.update = StringHelper.isNotEmpty(this.clientId);

		try {
			this.hostDn = hostService.getDnForHost(this.hostInum);
			if (!hostService.contains(this.hostDn)) {
				return Configuration.RESULT_FAILURE;
			}

			hostClientService.prepareHostClientBranch(this.hostDn);
		} catch (Exception ex) {
			log.error("Failed to initialize form", ex);
			return Configuration.RESULT_FAILURE;
		}

		if (update) {
			return update();
		} else {
			return add();
		}
	}

	private String add() {
		this.client = new HostClient();

		return Configuration.RESULT_SUCCESS;
	}

	private String update() {
		log.debug("Loading UMA host client '{0}' for host '{1}'", this.clientId, this.hostInum);
		try {
			String clientDn = hostClientService.getDnForHostClient(this.hostDn, this.clientId);
			this.client = hostClientService.getHostClientByDn(clientDn);
		} catch (LdapMappingException ex) {
			log.error("Failed to find host client '{0}'", ex, this.clientId);
			return Configuration.RESULT_FAILURE;
		}

		if (this.client == null) {
			log.error("Client is null");
			return Configuration.RESULT_FAILURE;
		}

		return Configuration.RESULT_SUCCESS;
	}

	@Restrict("#{s:hasPermission('uma', 'access')}")
	public void cancel() {
	}

	@Restrict("#{s:hasPermission('uma', 'access')}")
	public String save() {
		if (this.update) {
			client.setRev(String.valueOf(StringHelper.toInteger(client.getRev(), 0) + 1));
			// Update host client
			try {
				hostClientService.updateHostClient(this.client);
			} catch (LdapMappingException ex) {
				log.error("Failed to update host client '{0}'", ex, this.client.getId());
				return Configuration.RESULT_FAILURE;
			}
		} else {
			MetadataConfiguration metadataConfiguration = null;
			try {
				metadataConfiguration = clientUmaWebService.obtainUmaMetaDataConfigurationByIssuer(client.getId());
			} catch (Exception ex) {
				log.error("Failed to obtain metadata due to invalid UMA AM meta data configuration from '{0}'", ex, client.getId());
			}

			if (metadataConfiguration == null) {
				return "invalid_meta_data";
			}

			// Check if host client with this name already exist
			HostClient exampleClient = new HostClient();
			exampleClient.setDn(hostClientService.getDnForHostClient(this.hostDn, null));
			exampleClient.setId(metadataConfiguration.getIssuer());
			if (hostClientService.containsHostClient(exampleClient)) {
				return Configuration.RESULT_DUPLICATE;
			}

			// Prepare score description
			HostClient registeredClient = null;
			try {
				registeredClient = hostClientRegistrationService.registerUmaAmHostClient(metadataConfiguration);
			} catch (Exception ex) {
				log.error("Failed to register UMA AM client at '{0}'", ex, client.getId());
			}

			if (registeredClient == null) {
				return "failed_register_client";
			}

			String inum = hostClientService.generateInumForNewHostClient(this.hostDn, this.hostInum);
			String clientDn = hostClientService.getDnForHostClient(this.hostDn, inum);

			this.client.setInum(inum);
			this.client.setDn(clientDn);
			this.client.setId(registeredClient.getId());
			this.client.setClientId(registeredClient.getClientId());
			this.client.setEncodedClientSecret(registeredClient.getEncodedClientSecret());
			this.client.setOwner(currentPerson.getDn());
			this.client.setRegistrationDate(new Date());
			this.client.setRev(String.valueOf(0));

			// Save host client
			try {
				hostClientService.addHostClient(this.client);
			} catch (LdapMappingException ex) {
				log.error("Failed to add new host client '{0}'", ex, this.client.getId());
				return Configuration.RESULT_FAILURE;
			}

			this.update = true;
		}

		log.debug("Client were {0} successfully", (this.update ? "added" : "updated"));
		return Configuration.RESULT_SUCCESS;
	}

	@Restrict("#{s:hasPermission('uma', 'access')}")
	public String delete() {
		if (update) {
			// Remove host client
			try {
				hostClientService.removeHostClient(this.client);
				return Configuration.RESULT_SUCCESS;
			} catch (LdapMappingException ex) {
				log.error("Failed to remove host client {0}", ex, this.client.getId());
			}
		}

		return Configuration.RESULT_FAILURE;
	}

	@Destroy
	public void destroy() throws Exception {
		cancel();
	}

	public boolean isUpdate() {
		return update;
	}

	public String getClientId() {
		return clientId;
	}

	public void setClientId(String clientId) {
		this.clientId = clientId;
	}

	public HostClient getClient() {
		return client;
	}

	public String getHostInum() {
		return hostInum;
	}

	public void setHostInum(String hostInum) {
		this.hostInum = hostInum;
	}

}
