package org.gluu.oxtrust.action.uma;

import java.io.Serializable;
import java.util.List;

import javax.validation.constraints.NotNull;
import javax.validation.constraints.Size;

import org.gluu.oxtrust.ldap.service.ClientService;
import org.gluu.oxtrust.ldap.service.HostService;
import org.gluu.oxtrust.ldap.service.ImageService;
import org.gluu.oxtrust.ldap.service.uma.ScopeDescriptionService;
import org.gluu.oxtrust.model.GluuImage;
import org.gluu.oxtrust.model.uma.ScopeDescription;
import org.gluu.oxtrust.util.Configuration;
import org.jboss.seam.ScopeType;
import org.jboss.seam.annotations.In;
import org.jboss.seam.annotations.Logger;
import org.jboss.seam.annotations.Name;
import org.jboss.seam.annotations.Scope;
import org.jboss.seam.annotations.security.Restrict;
import org.jboss.seam.log.Log;
import org.xdi.util.StringHelper;
import org.xdi.util.Util;

/**
 * Action class for search scope descriptions
 * 
 * @author Yuriy Movchan Date: 12/05/2012
 */
@Name("searchScopeDescriptionAction")
@Scope(ScopeType.CONVERSATION)
@Restrict("#{identity.loggedIn}")
public class SearchScopeDescriptionAction implements Serializable {

	private static final long serialVersionUID = 1361095046179474395L;

	@Logger
	private Log log;

	@In
	protected HostService hostService;

	@In
	private ScopeDescriptionService scopeDescriptionService;

	@In
	protected ClientService clientService;

	@In
	protected ImageService imageService;

	private String hostInum, hostDn;

	@NotNull
	@Size(min = 0, max = 30, message = "Length of search string should be less than 30")
	private String searchPattern;

	private String oldSearchPattern;

	private List<ScopeDescription> scopeDescriptionList;

	@Restrict("#{s:hasPermission('uma', 'access')}")
	public String start() {
		try {
			this.hostDn = hostService.getDnForHost(this.hostInum);
			if (!hostService.contains(this.hostDn)) {
				return Configuration.RESULT_FAILURE;
			}

			scopeDescriptionService.prepareScopeDescriptionBranch(this.hostDn);
		} catch (Exception ex) {
			log.error("Failed to initialize form", ex);
			return Configuration.RESULT_FAILURE;
		}

		if (StringHelper.isEmpty(this.searchPattern)) {
			searchPattern = "";
		}

		search();

		return Configuration.RESULT_SUCCESS;
	}

	@Restrict("#{s:hasPermission('uma', 'access')}")
	public String search() {
		if (Util.equals(this.oldSearchPattern, this.searchPattern)) {
			return Configuration.RESULT_SUCCESS;
		}

		try {
			this.scopeDescriptionList = scopeDescriptionService.searchScopeDescriptions(hostDn, this.searchPattern, 100);
			this.oldSearchPattern = this.searchPattern;
		} catch (Exception ex) {
			log.error("Failed to find scope descriptions", ex);

			return Configuration.RESULT_FAILURE;
		}

		return Configuration.RESULT_SUCCESS;
	}

	public byte[] getIcon(ScopeDescription scopeDescription) {
		GluuImage gluuImage = imageService.getGluuImageFromXML(scopeDescription.getIcon());

		if (gluuImage != null) {
			return imageService.getThumIconData(gluuImage);
		}

		return imageService.getBlankIconData();
	}

	public String getSearchPattern() {
		return searchPattern;
	}

	public void setSearchPattern(String searchPattern) {
		this.searchPattern = searchPattern;
	}

	public List<ScopeDescription> getScopeDescriptionList() {
		return scopeDescriptionList;
	}

	public String getHostInum() {
		return hostInum;
	}

	public void setHostInum(String hostInum) {
		this.hostInum = hostInum;
	}

}
