package org.gluu.oxtrust.action;

import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.FileReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.Serializable;
import java.math.BigInteger;
import java.security.GeneralSecurityException;
import java.security.KeyFactory;
import java.security.KeyPair;
import java.security.PublicKey;
import java.security.Security;
import java.security.cert.X509Certificate;
import java.security.spec.RSAPublicKeySpec;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;

import javax.faces.context.FacesContext;
import javax.security.auth.x500.X500Principal;

import org.apache.commons.io.FileUtils;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang.WordUtils;
import org.bouncycastle.jce.PKCS10CertificationRequest;
import org.bouncycastle.jce.provider.BouncyCastleProvider;
import org.bouncycastle.jce.provider.JCERSAPrivateCrtKey;
import org.bouncycastle.openssl.PEMReader;
import org.bouncycastle.openssl.PasswordFinder;
import org.bouncycastle.util.encoders.Base64;
import org.gluu.oxtrust.ldap.service.ApplianceService;
import org.gluu.oxtrust.ldap.service.OrganizationService;
import org.gluu.oxtrust.ldap.service.SSLService;
import org.gluu.oxtrust.util.Configuration;
import org.jboss.seam.Component;
import org.jboss.seam.ScopeType;
import org.jboss.seam.annotations.In;
import org.jboss.seam.annotations.Logger;
import org.jboss.seam.annotations.Name;
import org.jboss.seam.annotations.Scope;
import org.jboss.seam.annotations.security.Restrict;
import org.jboss.seam.annotations.web.RequestParameter;
import org.jboss.seam.faces.FacesMessages;
import org.jboss.seam.international.StatusMessage.Severity;
import org.jboss.seam.log.Log;
import org.richfaces.event.FileUploadEvent;
import org.richfaces.model.UploadedFile;
import org.xdi.util.StringHelper;
import org.xdi.util.io.FileHelper;
import org.xdi.util.io.FileUploadWrapper;
import org.xdi.util.io.ResponseHelper;

/**
 * Manages ssl certificates.
 * 
 * @author �Oleksiy Tataryn�
 * 
 */
@Name("updateCertificatesAction")
@Scope(ScopeType.CONVERSATION)
@Restrict("#{identity.loggedIn}")
public class UpdateCertificatesAction implements Serializable {
	public static final String BEGIN_CERT_REQ = "-----BEGIN CERTIFICATE REQUEST-----";
	public static final String END_CERT_REQ = "-----END CERTIFICATE REQUEST-----";

	@Logger
	private Log log;
	private static final long serialVersionUID = 4012709440384265524L;

	private String orgInumFN = StringHelper.removePunctuation(OrganizationService.instance().getOrganizationInum());

	private String tomcatCertFN = orgInumFN + "-java.crt";
	private String idpCertFN = orgInumFN + "-shib.crt";

	@In(value = "#{facesContext}")
	FacesContext facesContext;

	@In
	private FacesMessages facesMessages;

	@In
	private SSLService sslService;

	private FileUploadWrapper tomcatCertWrapper = new FileUploadWrapper();
	private FileUploadWrapper idpCertWrapper = new FileUploadWrapper();
	private FileUploadWrapper spCertWrapper = new FileUploadWrapper();
	private HashMap<String, String> issuer;
	private HashMap<String, String> subject;

	private String uploadMarker;

	/**
	 * Fills issuer and subject maps with data about currently selected
	 * certificate
	 */
	public void getCert(String fileName) {
		X509Certificate cert = sslService.getCertificate(getTempCertDir() + fileName);
		if (cert != null) {
			issuer = new HashMap<String, String>();
			subject = new HashMap<String, String>();
			String issuerDN = cert.getIssuerX500Principal().getName();
			String[] values = issuerDN.split("(?<!\\\\),");
			for (String value : values) {
				String[] keyValue = value.split("=");
				issuer.put(keyValue[0], keyValue[1]);
			}
			String subjectDN = cert.getSubjectX500Principal().getName();
			values = subjectDN.split("(?<!\\\\),");
			for (String value : values) {
				String[] keyValue = value.split("=");
				subject.put(keyValue[0], keyValue[1]);
			}
			subject.put("validUntil", cert.getNotAfter().toString());
			subject.put("validAfter", cert.getNotBefore().toString());
		}
	}

	public String generateCSR(String fileName) {
		if (Security.getProvider(BouncyCastleProvider.PROVIDER_NAME) == null) {
			Security.addProvider(new BouncyCastleProvider());
		}

		KeyPair pair = getKeyPair(fileName);
		boolean result = false;
		if (pair != null) {

			String url = Configuration.instance().getIdpUrl().replaceFirst(".*//", "");
			String csrPrincipal = String.format("CN=%s", url);
			X500Principal principal = new X500Principal(csrPrincipal);

			PKCS10CertificationRequest csr = null;
			try {
				csr = new PKCS10CertificationRequest("SHA1withRSA", principal, pair.getPublic(), null, pair.getPrivate());
			} catch (GeneralSecurityException e) {
				log.error(e.getMessage(), e);
				return Configuration.RESULT_FAILURE;
			}

			// Form download responce
			StringBuilder response = new StringBuilder();

			response.append(BEGIN_CERT_REQ + "\n");
			response.append(WordUtils.wrap(new String(Base64.encode(csr.getDEREncoded())), 64, "\n", true) + "\n");
			response.append(END_CERT_REQ + "\n");

			result = ResponseHelper.downloadFile("csr.pem", Configuration.CONTENT_TYPE_TEXT_PLAIN, response.toString().getBytes(),
					facesContext);
		}
		return result ? Configuration.RESULT_SUCCESS : Configuration.RESULT_FAILURE;
	}

	public void listener(FileUploadEvent event) {
		UploadedFile item = event.getUploadedFile();

		updateCert(item);
	}

	public void keyListener(FileUploadEvent event) {
		UploadedFile item = event.getUploadedFile();

		updateKey(item);
	}

	public boolean compare(String fileName) {
		KeyPair pair = getKeyPair(fileName);
		X509Certificate cert = sslService.getCertificate(getTempCertDir() + fileName);

		boolean noFilesPresent = (pair == null) && (cert == null);

		boolean filesPresent = (pair != null) && (cert != null);
		boolean filesValid = false;
		if (filesPresent) {
			filesValid = (pair.getPublic() != null) && (pair.getPublic().equals(cert.getPublicKey()));
		}
		
		boolean compareResult = (noFilesPresent || (filesPresent && filesValid));
		log.debug(fileName + " compare result: " + compareResult);
		return compareResult;
	}

	private KeyPair getKeyPair(String fileName) {
		KeyPair pair = null;
		JCERSAPrivateCrtKey privateKey = null;
		PEMReader r = null;
		FileReader fileReader = null;

		File keyFile = new File(getTempCertDir() + fileName.replace("crt", "key"));
		if (keyFile.isFile()) {
			try {
				fileReader = new FileReader(keyFile);
				r = new PEMReader(fileReader, new PasswordFinder() {
					public char[] getPassword() {
						// Since keys are stored without a password this
						// function should not be called.
						return null;
					}
				});

				Object keys = r.readObject();
				if(keys == null){
					log.error(" Unable to read keys from: " + keyFile.getAbsolutePath());
					return null;
				}

				if(keys instanceof KeyPair){
					pair = (KeyPair) keys;
					log.debug( keyFile.getAbsolutePath() + "contains KeyPair");
				}else if(keys instanceof JCERSAPrivateCrtKey){

					privateKey = (JCERSAPrivateCrtKey) keys;
					log.debug( keyFile.getAbsolutePath() + "contains JCERSAPrivateCrtKey");
					BigInteger exponent = privateKey.getPublicExponent();
					BigInteger modulus = privateKey.getModulus();
					
					RSAPublicKeySpec publicKeySpec = new java.security.spec.RSAPublicKeySpec( modulus, exponent);   
					PublicKey publicKey = null;
					try {   
					     KeyFactory keyFactory = KeyFactory.getInstance("RSA");   

					     publicKey  = keyFactory.generatePublic(publicKeySpec);   
					} catch (Exception e) {   
					     e.printStackTrace();   
					} 
					
					pair = new KeyPair(publicKey, privateKey);
				}else{
					log.error(keyFile.getAbsolutePath() + " Contains unsupported key type: " + keys.getClass().getName());
					return null;
				}

			} catch (IOException e) {
				log.error(e.getMessage(), e);
				return null;
			} finally {
				try {
					r.close();
					fileReader.close();
				} catch (Exception e) {
					log.error(e.getMessage(), e);
					return null;
				}
			}
		} else {
			log.error("Key file does not exist : " + keyFile.getAbsolutePath());
		}
		log.debug("KeyPair successfully extracted from: " + keyFile.getAbsolutePath());
		return pair;
	}

	public boolean certPresent(String filename) {
		KeyPair pair = getKeyPair(filename);
		X509Certificate cert = sslService.getCertificate(getTempCertDir() + filename);

		boolean filesPresent = (pair != null) && (cert != null);

		return filesPresent;
	}

	public String getIdpCertFN() {
		return idpCertFN;
	}

	public String getTomcatCertFN() {
		return tomcatCertFN;
	}

	public FileUploadWrapper getTomcatCertWrapper() {
		return tomcatCertWrapper;
	}

	public FileUploadWrapper getIdpCertWrapper() {
		return idpCertWrapper;
	}

	public FileUploadWrapper getSpCertWrapper() {
		return spCertWrapper;
	}

	public String getTempCertDir() {
		return Configuration.instance().getTempCertDir() + File.separator;
	}

	public HashMap<String, String> getIssuer() {
		return issuer;
	}

	public HashMap<String, String> getSubject() {
		return subject;
	}

	public void setUploadMarker(String uploadMarker) {
		this.uploadMarker = uploadMarker;
	}

	public String getUploadMarker() {
		return uploadMarker;
	}

	public boolean prepareTempWorkspace() {
		String tempDirFN = Configuration.instance().getTempCertDir();
		String dirFN = Configuration.instance().getCertDir();
		File certDir = new File(dirFN);
		if (tempDirFN == null || dirFN == null || !certDir.isDirectory() || StringHelper.isEmpty(tempDirFN)) {

			return false;
		} else {
			File tempDir = new File(tempDirFN);
			// If tempDir exists - empty it, if not - create. If exists, but
			// isFile - write an error and return false.
			if (tempDir.isDirectory()) {
				File[] files = tempDir.listFiles();
				for (File file : files) {
					if (file.isFile()) {
						file.delete();
					}
				}
			} else {
				if (tempDir.exists()) {
					log.error("Temporary certifcates path exists but is not a directory");
					return false;
				} else {
					tempDir.mkdirs();
				}
			}

			File[] files = certDir.listFiles();
			for (File file : files) {
				if (file.isFile()) {
					try {
						FileHelper.copy(file, new File(tempDirFN + File.separator + file.getName()));
					} catch (IOException e) {
						log.error("Unable to populate temp certs directory: ", e);
						return false;
					}
				}
			}
		}

		return true;
	}

	public static UpdateCertificatesAction instance() {
		return (UpdateCertificatesAction) Component.getInstance(UpdateCertificatesAction.class);
	}

	/**
	 * Updates certificates from temporary working directory to production and
	 * restarts services.
	 * 
	 * @return true if update was successful. false if update was aborted due to
	 *         some error (perhaps permissions issue.)
	 */
	public boolean updateCerts() {
		if (!compare(tomcatCertFN) || !compare(idpCertFN)) {
			facesMessages.add(Severity.ERROR, "Certificates and private keys should match. Certificate update aborted.");
			return false;
		}
		boolean wereAnyChanges = false;
		String tempDirFN = Configuration.instance().getTempCertDir();
		String dirFN = Configuration.instance().getCertDir();
		File certDir = new File(dirFN);
		File tempDir = new File(tempDirFN);
		if (tempDirFN == null || dirFN == null || !certDir.isDirectory() || !tempDir.isDirectory()) {
			facesMessages.add(Severity.ERROR, "Certificate update aborted due to filesystem error");
			return false;
		} else {
			File[] files = tempDir.listFiles();
			for (File file : files) {
				try {
					if (file.isFile() && !FileUtils.contentEquals(file, new File(dirFN + File.separator + file.getName()))) {
						FileHelper.copy(file, new File(dirFN + File.separator + file.getName()));
						wereAnyChanges = true;
					}
				} catch (IOException e) {
					facesMessages
							.add(Severity.FATAL,
									"Certificate update failed. Certificates may have been corrupted. Please contact a Gluu administrator for help.");
					log.fatal("Error occured on certificates update:", e);
				}
			}
			if (wereAnyChanges) {
				File pkcs12 = new File(certDir, orgInumFN + "-java.pkcs12");
				File pem = new File(certDir, orgInumFN + "-java.pem");
				File jks = new File(certDir, orgInumFN + "-java.jks");
				log.info("Deleting %s : %s", orgInumFN + "-java.pkcs12", pkcs12.delete());
				log.info("Deleting %s : %s", orgInumFN + "-java.pem", pem.delete());
				log.info("Deleting %s : %s", orgInumFN + "-java.jks", jks.delete());
				ApplianceService.instance().restartServices();
			}

		}

		return wereAnyChanges;
	}

	public void certUpload(FileUploadEvent event) {
		updateCert(event.getUploadedFile());
	}

	private void updateCert(UploadedFile item) {
		InputStream is = null;
		OutputStream os = null;
		try {
			is = item.getInputStream();
			os = new FileOutputStream(getTempCertDir() + getUploadMarker());
			BufferedOutputStream bos = new BufferedOutputStream(os);

			IOUtils.copy(is, bos);
			bos.flush();
		} catch (IOException e) {
			log.error("Failed to upload certicicate", e);
		} finally {
			IOUtils.closeQuietly(is);
			IOUtils.closeQuietly(os);
		}
	}

	public void keyUpload(FileUploadEvent event) {
		updateKey(event.getUploadedFile());
	}

	private void updateKey(UploadedFile item) {
		InputStream is = null;
		OutputStream os = null;
		try {
			is = item.getInputStream();
			os = new FileOutputStream(getTempCertDir() + getUploadMarker().replace("crt", "key"));
			BufferedOutputStream bos = new BufferedOutputStream(os);

			IOUtils.copy(is, bos);
			bos.flush();
		} catch (IOException e) {
			log.error("Failed to upload key", e);
		} finally {
			IOUtils.closeQuietly(is);
			IOUtils.closeQuietly(os);
		}
	}

}
