package org.gluu.oxtrust.ws.rs;

import java.io.File;
import java.io.IOException;
import java.net.URI;
import java.net.URISyntaxException;

import javax.servlet.http.HttpServletRequest;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.Response;

import org.apache.commons.io.FileUtils;
import org.gluu.oxtrust.model.scim.Error;
import org.gluu.oxtrust.model.scim.Errors;
import org.gluu.oxtrust.util.Configuration;
import org.jboss.seam.annotations.In;
import org.jboss.seam.annotations.Logger;
import org.jboss.seam.annotations.Name;
import org.jboss.seam.annotations.security.Restrict;
import org.jboss.seam.log.Log;
import org.jboss.seam.security.Identity;
import org.xdi.util.process.ProcessHelper;

/**
 * Self testing service
 * 
 * @author Yuriy Movchan Date: 04.27.2012
 */
@Name("selfTestWebService")
@Restrict("#{identity.loggedIn}")
public class SelfTestWebServiceImpl implements SelfTestWebService {

	@Logger
	private Log log;

	@In
	private Identity identity;

	@Override
	public Response runTest(HttpServletRequest request) {
		Configuration conf = Configuration.instance();

		if (!Boolean.TRUE.equals(conf.isSelfTestEnabled())) {
			String message = "Disabled by administrator";

			log.error(message);
			return getErrorResponse(message, Response.Status.BAD_REQUEST.getStatusCode());
		}

		if (!identity.hasPermission("scim_test", "access")) {
			String message = "User isn't authorized";

			log.error(message);
			return getErrorResponse(message, Response.Status.FORBIDDEN.getStatusCode());
		}

		String programToStart = conf.getSelfTestProgramToStart();
		String workingFolder = conf.getSelfTestProgramWorkingFolder();
		try {
			boolean result = ProcessHelper.executeProgram(programToStart, workingFolder, false, 0, null);
			if (!result) {
				return null;
			}
		} catch (Exception ex) {
			String message = String.format("Failed to run '%s' in working directory '%s'", programToStart, workingFolder);

			log.error(message, ex);
			return getErrorResponse(message, Response.Status.INTERNAL_SERVER_ERROR.getStatusCode());
		}

		String reportFile = workingFolder + File.separator + "test-report" + File.separator + "testng-results.xml";
		String reportFileContext = null;
		try {
			reportFileContext = FileUtils.readFileToString(new File(reportFile));
		} catch (IOException ex) {
			String message = String.format("Failed to read report file '%s'", reportFile);

			log.error(message, ex);
			return getErrorResponse(message, Response.Status.INTERNAL_SERVER_ERROR.getStatusCode());
		}

		URI location;
		try {
			location = new URI("/self_test/run");
		} catch (URISyntaxException ex) {
			String message = "Failed to build location URI";

			log.error(message, ex);
			return getErrorResponse(message, Response.Status.INTERNAL_SERVER_ERROR.getStatusCode());
		}

		return Response.ok(reportFileContext, MediaType.APPLICATION_XML).location(location).build();
	}

	private Response getErrorResponse(String errMsg, int statusCode) {
		Errors errors = new Errors();
		Error error = new org.gluu.oxtrust.model.scim.Error(errMsg, statusCode, "");
		errors.getErrors().add(error);

		return Response.status(statusCode).entity(errors).build();
	}
	/*
	 * public static void main(String[] args) { CommandLine commandLine = new
	 * CommandLine("ant.bat"); commandLine.addArguments("test");
	 * 
	 * boolean result = ProcessHelper.executeProgram(commandLine,
	 * "D:\\Workspaces\\ox\\oxTrust", false, 0, null);
	 * System.out.println(result); }
	 */
}
