package org.gluu.oxtrust.util;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.Serializable;
import java.util.Enumeration;
import java.util.Hashtable;
import java.util.Properties;

import org.apache.commons.io.IOUtils;
import org.gluu.oxtrust.model.LdapConfig;
import org.jboss.seam.log.Log;
import org.jboss.seam.log.Logging;

/**
 * managing LDAP connection properties
 * 
 * @author Reda Zerrad Date: 08.14.2012
 */

public class LDAPProperties implements Serializable {

	/**
     *
     */
	private static final long serialVersionUID = 7517493629095645850L;

	public static final String CONFIGURATION_FILE_APPLICATION_CONFIGURATION = "oxTrustLdap.properties";
	private static Log log = Logging.getLog(LDAPProperties.class);
	private Properties properties;
	private Hashtable<String, String> propertyList;

	public LDAPProperties() {
		readProperties();
	}

	public Hashtable<String, String> readProperties() {
		FileInputStream fileInput = null;
		try {

			String tomcatHome = System.getProperty("catalina.home");
			String path = "";
			if (tomcatHome != null) {
				path = System.getProperty("catalina.home") + File.separator + "conf" + File.separator
						+ CONFIGURATION_FILE_APPLICATION_CONFIGURATION;

			}
			File file = new File(path);
			fileInput = new FileInputStream(file);
			this.properties = new Properties();
			properties.load(fileInput);
			this.propertyList = new Hashtable<String, String>();
			Enumeration enuKeys = properties.keys();
			while (enuKeys.hasMoreElements()) {
				String key = (String) enuKeys.nextElement();
				String value = properties.getProperty(key);
				this.propertyList.put(key, value);
			}
			return this.propertyList;
		} catch (FileNotFoundException e) {
			e.printStackTrace();
		} catch (IOException e) {
			e.printStackTrace();
		} finally {
			IOUtils.closeQuietly(fileInput);
		}
		return null;
	}

	public void writeProperties() {
		if (this.properties == null) {
			return;
		}

		FileOutputStream fileOut = null;
		try {
			String tomcatHome = System.getProperty("catalina.home");
			String path = "";
			if (tomcatHome != null) {
				path = System.getProperty("catalina.home") + File.separator + "conf" + File.separator
						+ this.CONFIGURATION_FILE_APPLICATION_CONFIGURATION;
			}

			File file = new File(path);
			fileOut = new FileOutputStream(file);
			properties.store(fileOut, "");
		} catch (FileNotFoundException e) {
			e.printStackTrace();
		} catch (IOException e) {
			e.printStackTrace();
		} finally {
			IOUtils.closeQuietly(fileOut);
		}

	}

	public void propertiesMaping(Hashtable<String, String> propertyList) {

		if (propertyList == null) {
			log.error("propertyList is null!");
			return;
		}

		Enumeration enuKeys = propertyList.keys();

		while (enuKeys.hasMoreElements()) {
			String key = (String) enuKeys.nextElement();
			String value = properties.getProperty(key);
			this.properties.setProperty(key, value);
		}
	}

	public Hashtable<String, String> getProperties() {
		return this.propertyList = readProperties();
	}

	public void setProperties(Hashtable<String, String> propertyList) {
		if (propertyList == null) {
			log.error("propertyList is null!");
			return;
		}
		propertiesMaping(propertyList);
		writeProperties();
		readProperties();
	}

	public String getProperty(String key) {
		return this.properties.getProperty(key);
	}

	public void addProperty(String key, String value) {
		this.properties.setProperty(key, value);
		writeProperties();
		readProperties();

	}

	public void deleteProperty(String key) {
		if (key == null) {
			log.error("key is null!");
			return;
		}
		if (!this.properties.containsKey(key)) {
			return;
		}
		this.properties.remove(key);
		writeProperties();
		readProperties();
	}

	public LdapConfig getLDAPConfig() {

		LdapConfig conf = new LdapConfig();
		conf.setLdapBindDn(getProperty("bindDN"));
		conf.setLdapBindPw(getProperty("bindPassword"));
		conf.setLdapHost(getProperty("servers").split(":")[0]);
		conf.setLdapPort(getProperty("servers").split(":")[1]);
		conf.setLdapUseSSL(Boolean.valueOf(("useSSL")));
		return conf;

	}

	public void saveLDAPConfig(LdapConfig ldapConfig) {
		if (ldapConfig == null) {
			log.error("ldapConfig is null!");
			return;
		}

		this.properties.setProperty("bindDN", ldapConfig.getLdapBindDn());
		this.properties.setProperty("bindPassword", ldapConfig.getLdapBindPw());
		this.properties.setProperty("servers", ldapConfig.getLdapHost() + ":" + ldapConfig.getLdapPort());
		this.properties.setProperty("useSSL", Boolean.toString(ldapConfig.isLdapUseSSL()));
		writeProperties();
		readProperties();

	}

}
