package org.gluu.oxtrust.model;

import java.io.Serializable;
import java.text.ParseException;
import java.util.Arrays;
import java.util.Date;

import com.unboundid.util.StaticUtils;

/**
 * Attribute
 * 
 * @author Yuriy Movchan Date: 10.07.2010
 */
public class GluuCustomAttribute implements Serializable, Comparable<GluuCustomAttribute> {

	private static final long serialVersionUID = 1468440094325406153L;

	private String name;
	private String[] values;

	private transient GluuAttribute metadata;

	private transient boolean newAttribute = false;

	private transient boolean mandatory = false;

	public GluuCustomAttribute() {
	}

	public GluuCustomAttribute(String name, String value) {
		this.name = name;
		setValue(value);
	}

	public GluuCustomAttribute(String name, String value, boolean newAttribute) {
		this.name = name;
		setValue(value);
		this.newAttribute = newAttribute;
	}

	public GluuCustomAttribute(String name, String value, boolean newAttribute, boolean mandatory) {
		this.name = name;
		setValue(value);
		this.newAttribute = newAttribute;
		this.mandatory = mandatory;
	}

	public GluuCustomAttribute(String name, String[] values) {
		this.name = name;
		this.values = values;
	}

	public String getValue() {
		if (this.values == null) {
			return null;
		}

		if (this.values.length > 0) {
			return this.values[0];
		}

		return null;
	}

	public void setValue(String value) {
		if (this.values == null) {
			this.values = new String[0];
		}

		if (this.values.length != 1) {
			this.values = new String[1];
		}
		this.values[0] = value;
	}

	public Date getDate() {
		if (this.values == null) {
			return null;
		}

		if (this.values.length > 0 && values[0] != null) {
			try {
				return StaticUtils.decodeGeneralizedTime((String) values[0]);
			} catch (ParseException e) {
				e.printStackTrace();
			}
		}

		return null;
	}

	public void setDate(Date date) {
		if (this.values == null) {
			this.values = new String[0];
		}

		if (this.values.length != 1) {
			this.values = new String[1];
		}
		this.values[0] = StaticUtils.encodeGeneralizedTime(date);
	}

	public String[] getValues() {
		return values;
	}

	public void setValues(String[] values) {
		this.values = values;
	}

	public final String getName() {
		return name;
	}

	public final void setName(String name) {
		this.name = name;
	}

	public void setMetadata(GluuAttribute metadata) {
		this.metadata = metadata;
	}

	public GluuAttribute getMetadata() {
		return metadata;
	}

	public boolean isNew() {
		return newAttribute;
	}

	public void setNew(boolean newAttribute) {
		this.newAttribute = newAttribute;
	}

	public boolean isMandatory() {
		return mandatory;
	}

	public void setMandatory(boolean mandatory) {
		this.mandatory = mandatory;
	}

	public String getDisplayValue() {
		if (values == null) {
			return "";
		}

		if (values.length == 1) {
			return values[0];
		}

		StringBuilder sb = new StringBuilder(values[0]);
		for (int i = 1; i < values.length; i++) {
			sb.append(", ").append(values[i]);
		}

		return sb.toString();
	}

	public boolean isAdminCanAccess() {
		return (this.metadata != null) && this.metadata.isAdminCanAccess();
	}

	public boolean isAdminCanView() {
		return (this.metadata != null) && this.metadata.isAdminCanView();
	}

	public boolean isAdminCanEdit() {
		return (this.metadata != null) && this.metadata.isAdminCanEdit();
	}

	public boolean isUserCanAccess() {
		return (this.metadata != null) && this.metadata.isUserCanAccess();
	}

	public boolean isUserCanView() {
		return (this.metadata != null) && this.metadata.isUserCanView();
	}

	public boolean isUserCanEdit() {
		return (this.metadata != null) && this.metadata.isUserCanEdit();
	}

	// public boolean equals(Object attribute) {
	// return (attribute instanceof GluuCustomAttribute) &&
	// (((GluuCustomAttribute) attribute).getName().equals(getName()));
	// }

	@Override
	public boolean equals(Object o) {
		if (this == o)
			return true;
		if (o == null || getClass() != o.getClass())
			return false;

		GluuCustomAttribute that = (GluuCustomAttribute) o;

		return !(name != null ? !name.equalsIgnoreCase(that.name) : that.name != null);

	}

	@Override
	public int hashCode() {
		return name != null ? name.hashCode() : 0;
	}

	@Override
	public String toString() {
		return String.format("Attribute [name=%s, values=%s, metadata=%s]", name, Arrays.toString(values), metadata);
	}

	public int compareTo(GluuCustomAttribute o) {
		return name.compareTo(o.name);
	}

}
