package org.gluu.oxtrust.ldap.service.uma;

import java.io.Serializable;
import java.util.List;

import org.gluu.oxtrust.model.uma.HostClient;
import org.gluu.oxtrust.util.Configuration;
import org.gluu.site.ldap.persistence.LdapEntryManager;
import org.hibernate.annotations.common.util.StringHelper;
import org.jboss.seam.Component;
import org.jboss.seam.ScopeType;
import org.jboss.seam.annotations.AutoCreate;
import org.jboss.seam.annotations.In;
import org.jboss.seam.annotations.Logger;
import org.jboss.seam.annotations.Name;
import org.jboss.seam.annotations.Scope;
import org.jboss.seam.log.Log;
import org.xdi.ldap.model.SimpleBranch;
import org.xdi.util.INumGenerator;

import com.unboundid.ldap.sdk.Filter;

/**
 * Provides operations with UMA host AM clients
 * 
 * @author Yuriy Movchan Date: 12/08/2012
 */
@Scope(ScopeType.STATELESS)
@Name("hostClientService")
@AutoCreate
public class HostClientService implements Serializable {

	private static final long serialVersionUID = -2537567020929600777L;

	@In
	private LdapEntryManager ldapEntryManager;

	@Logger
	private Log log;

	public void addBranch(String hostDn) {
		SimpleBranch branch = new SimpleBranch();
		branch.setOrganizationalUnitName("clients");
		branch.setDn(getDnForHostClient(hostDn, null));

		ldapEntryManager.persist(branch);
	}

	/**
	 * Add new host AM client entry
	 * 
	 * @param group
	 *            Group
	 */
	public void addHostClient(HostClient hostClient) {
		ldapEntryManager.persist(hostClient);
	}

	/**
	 * Update host AM client entry
	 * 
	 * @param group
	 *            Group
	 */
	public void updateHostClient(HostClient hostClient) {
		ldapEntryManager.merge(hostClient);
	}

	/**
	 * Remove host AM client entry
	 * 
	 * @param group
	 *            Group
	 */
	public void removeHostClient(HostClient hostClient) {
		ldapEntryManager.remove(hostClient);
	}

	/**
	 * Get all host AM clients
	 * 
	 * @return List of host AM clients
	 */
	public List<HostClient> getAllHostClients(String hostDn, String... ldapReturnAttributes) {
		return ldapEntryManager.findEntries(getDnForHostClient(hostDn, null), HostClient.class, ldapReturnAttributes, null);
	}

	/**
	 * Search host AM clients by pattern
	 * 
	 * @param pattern
	 *            Pattern
	 * @param sizeLimit
	 *            Maximum count of results
	 * @return List of host AM clients
	 */
	public List<HostClient> searchHostClients(String hostDn, String pattern, int sizeLimit) {
		String[] targetArray = new String[] { pattern };
		Filter inumFilter = Filter.createSubstringFilter(Configuration.inum, null, targetArray, null);
		Filter oxIdFilter = Filter.createSubstringFilter("oxId", null, targetArray, null);
		Filter displayNameFilter = Filter.createSubstringFilter(Configuration.displayName, null, targetArray, null);
		Filter oxAuthClientIdFilter = Filter.createSubstringFilter("oxAuthClientId", null, targetArray, null);
		Filter searchFilter = Filter.createORFilter(inumFilter, oxIdFilter, displayNameFilter, oxAuthClientIdFilter);

		List<HostClient> result = ldapEntryManager.findEntries(getDnForHostClient(hostDn, null), HostClient.class, searchFilter, sizeLimit);

		return result;
	}

	/**
	 * Get host AM clients by example
	 * 
	 * @param hostClient
	 *            HostClient
	 * @return HostClient which conform example
	 */
	public List<HostClient> findHostClients(HostClient hostClient) {
		return ldapEntryManager.findEntries(hostClient);
	}

	/**
	 * Get resource sets by Id
	 * 
	 * @param id
	 *            Id
	 * @return List of HostClients which specified id
	 */
	public List<HostClient> findHostClientsById(String hostDn, String id) {
		return ldapEntryManager.findEntries(getDnForHostClient(hostDn, null), HostClient.class, Filter.createEqualityFilter("oxId", id));
	}

	public boolean containsBranch(String hostDn) {
		return ldapEntryManager.contains(SimpleBranch.class, getDnForHostClient(hostDn, null));
	}

	/**
	 * Check if LDAP server contains host AM client with specified attributes
	 * 
	 * @return True if host AM client with specified attributes exist
	 */
	public boolean containsHostClient(HostClient hostClient) {
		return ldapEntryManager.contains(hostClient);
	}

	public void prepareHostClientBranch(String hostDn) {
		// Create host AM client branch if needed
		if (!containsBranch(hostDn)) {
			addBranch(hostDn);
		}
	}

	/**
	 * Get host AM client by DN
	 * 
	 * @param DN
	 *            Host AM client DN
	 * @return Host AM client
	 */
	public HostClient getHostClientByDn(String dn) {
		return ldapEntryManager.find(HostClient.class, dn);
	}

	/**
	 * Generate new inum for host client
	 * 
	 * @return New inum for host client
	 * @throws Exception
	 */
	public String generateInumForNewHostClient(String hostDn, String baseInum) {
		HostClient hostClient = new HostClient();
		String newInum = null;
		do {
			newInum = generateInumForNewGroupImpl(hostDn, baseInum);
			String newDn = getDnForHostClient(hostDn, newInum);
			hostClient.setDn(newDn);
		} while (ldapEntryManager.contains(hostClient));

		return newInum;
	}

	/**
	 * Generate new inum for host client
	 * 
	 * @return New inum for host client
	 */

	private String generateInumForNewGroupImpl(String hostDn, String baseInum) {
		return baseInum + Configuration.inumDelimiter + INumGenerator.generate(2);
	}

	/**
	 * Build DN string for host AM client
	 */
	public String getDnForHostClient(String hostDn, String inum) {
		if (StringHelper.isEmpty(inum)) {
			return String.format("ou=clients,%s", hostDn);
		}

		return String.format("inum=%s,ou=clients,%s", inum, hostDn);
	}

	/**
	 * Get HostClientService instance
	 * 
	 * @return HostClientService instance
	 */
	public static HostClientService instance() {
		return (HostClientService) Component.getInstance(HostClientService.class);
	}

}
