package org.gluu.oxtrust.ldap.service.uma;

import org.gluu.oxtrust.model.uma.HostClient;
import org.jboss.seam.ScopeType;
import org.jboss.seam.annotations.AutoCreate;
import org.jboss.seam.annotations.In;
import org.jboss.seam.annotations.Logger;
import org.jboss.seam.annotations.Name;
import org.jboss.seam.annotations.Scope;
import org.jboss.seam.faces.FacesMessages;
import org.jboss.seam.log.Log;
import org.xdi.oxauth.client.RegisterClient;
import org.xdi.oxauth.client.RegisterRequest;
import org.xdi.oxauth.client.RegisterResponse;
import org.xdi.oxauth.model.register.ApplicationType;
import org.xdi.oxauth.model.uma.MetadataConfiguration;
import org.xdi.oxauth.model.util.StringUtils;
import org.xdi.util.StringHelper;
import org.xdi.util.security.StringEncrypter;
import org.xdi.util.security.StringEncrypter.EncryptionException;

import java.util.Date;

/**
 * Provides operations with UMA host AM client registrations
 * 
 * @author Yuriy Movchan Date: 12/11/2012
 */
@Scope(ScopeType.STATELESS)
@Name("hostClientRegistrationService")
@AutoCreate
public class HostClientRegistrationService {

	@Logger
	private Log log;

	@In
	private FacesMessages facesMessages;

	@In
	private ClientUmaWebService clientUmaWebService;

	@In
	private HostClientService hostClientService;

	public HostClient registerUmaAmHostClient(MetadataConfiguration metadataConfiguration) {
		if (metadataConfiguration == null) {
			log.error("UMA meta data is empty");
			return null;
		}

		String issuer = metadataConfiguration.getIssuer();

		// Register client
		String registerUri = metadataConfiguration.getDynamicClientEndpoint();
		RegisterClient registerClient = new RegisterClient(registerUri);
		RegisterRequest registerRequest = new RegisterRequest(ApplicationType.WEB, "oxAuth test app",
                StringUtils.spaceSeparatedToList("http://localhost:8080/oxTrust"));
		registerClient.setRequest(registerRequest);
		RegisterResponse response = registerClient.exec();
		if (response == null) {
			log.error("Failed to register client at UMA AM '{0}'", issuer);
			return null;
		}

		int status = response.getStatus();
		String clientId = response.getClientId();
		String clientSecret = response.getClientSecret();
		Date expiresAt = response.getExpiresAt();

		// Validate response
		if ((response.getStatus() != 200) || StringHelper.isEmpty(clientId) || StringHelper.isEmpty(clientSecret) || (expiresAt == null)) {
			log.error("Failed to register client. Status: '{0}', clientId: '{1}', clientSecret: '{2}', expiresAt: '{3}'", status, clientId,
					clientSecret, expiresAt);
			return null;
		}

		HostClient hostClient = new HostClient();
		hostClient.setId(issuer);
		hostClient.setDisplayName(String.format("oxTrust UMA AM client at '%s'", issuer));
		hostClient.setClientId(clientId);
		hostClient.setRegistrationDate(new Date());

		// Encrypt password
		if (StringHelper.isNotEmpty(clientSecret)) {
			try {
				hostClient.setEncodedClientSecret(StringEncrypter.defaultInstance().encrypt(clientSecret));
			} catch (EncryptionException exx) {
				log.error("Failed to encrypt client password: '{0}'", clientSecret);
			}
		}

		log.info("oxTrust successfully registered UMA AM client '{0}' at UMA AM Host: '{1}'", response.getClientId(), issuer);

		return hostClient;
	}

}