package org.gluu.oxtrust.ldap.service;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import net.sf.ehcache.CacheManager;

import org.gluu.oxtrust.model.GluuBoolean;
import org.gluu.oxtrust.model.GluuOrganization;
import org.gluu.oxtrust.util.Configuration;
import org.gluu.site.ldap.persistence.LdapEntryManager;
import org.jboss.seam.Component;
import org.jboss.seam.ScopeType;
import org.jboss.seam.annotations.AutoCreate;
import org.jboss.seam.annotations.In;
import org.jboss.seam.annotations.Logger;
import org.jboss.seam.annotations.Name;
import org.jboss.seam.annotations.Observer;
import org.jboss.seam.annotations.Scope;
import org.jboss.seam.log.Log;
import org.xdi.util.ArrayHelper;
import org.xdi.util.StringHelper;

/**
 * Provides operations with organization
 * 
 * @author Yuriy Movchan Date: 11.02.2010
 */
@Scope(ScopeType.STATELESS)
@Name("organizationService")
@AutoCreate
public class OrganizationService implements Serializable {

	private static final long serialVersionUID = 5537567020929600777L;

	@Logger
	private Log log;

	@In
	private LdapEntryManager ldapEntryManager;

	@In
	private CacheService cacheService;

	/**
	 * Update organization entry
	 * 
	 * @param organization
	 *            Organization
	 */
	public void updateOrganization(GluuOrganization organization) {
		ldapEntryManager.merge(organization);

	}

	/**
	 * Check if LDAP server contains organization with specified attributes
	 * 
	 * @return True if organization with specified attributes exist
	 */
	public boolean containsOrganization(GluuOrganization organization) {
		return ldapEntryManager.contains(organization);
	}

	/**
	 * Get organization
	 * 
	 * @return Organization entry
	 */
	public GluuOrganization getOrganization() {
		return getOrganizationByInum(getInumForOrganization());
	}

	/**
	 * Get organization by DN
	 * 
	 * @param inum
	 *            inum
	 * @return Organization
	 */
	public GluuOrganization getOrganizationByInum(String inum) {
		String key = Configuration.CACHE_ORGANIZATION_KEY + "_" + inum;
		GluuOrganization organization = (GluuOrganization) cacheService.get(Configuration.CACHE_APPLICATION_NAME, key);
		if (organization == null) {
			organization = ldapEntryManager.find(GluuOrganization.class, getDnForOrganization(inum));
			cacheService.put(Configuration.CACHE_APPLICATION_NAME, key, organization);
		}

		return organization;
	}

	/**
	 * Returns custom message defined for the organization
	 * 
	 * @param customMessageId
	 *            message id
	 * @return custom message
	 */
	public String getOrganizationCustomMessage(String customMessageId) {
		GluuOrganization organization = getOrganization();

		String key = Configuration.CACHE_ORGANIZATION_CUSTOM_MESSAGE_KEY + "_" + organization.getInum();
		@SuppressWarnings("unchecked")
		Map<String, String> organizationCustomMessage = (Map<String, String>) cacheService.get(Configuration.CACHE_APPLICATION_NAME, key);
		if (organizationCustomMessage == null) {
			organizationCustomMessage = new HashMap<String, String>();

			String[] customMessages = organization.getCustomMessages();
			if (ArrayHelper.isNotEmpty(customMessages)) {
				for (String customMessage : customMessages) {
					int idx = customMessage.indexOf(':');
					if ((idx > 0) && (idx + 1 < customMessage.length())) {
						String msgKey = customMessage.substring(0, idx).trim();
						String msgValue = customMessage.substring(idx + 1).trim();

						if (StringHelper.isNotEmpty(msgKey) && StringHelper.isNotEmpty(msgValue)) {
							organizationCustomMessage.put(msgKey, msgValue);
						}
					}
				}
			}
			cacheService.put(Configuration.CACHE_APPLICATION_NAME, key, organizationCustomMessage);
		}

		return organizationCustomMessage.get(customMessageId);
	}

	public String[] buildOrganizationCustomMessages(String[][] customMessages) {
		List<String> result = new ArrayList<String>();

		for (String[] customMessage : customMessages) {
			if (ArrayHelper.isEmpty(customMessage) || customMessage.length != 2) {
				continue;
			}
			String msgKey = customMessage[0];
			String msgValue = customMessage[1];

			if (StringHelper.isNotEmpty(msgKey) && StringHelper.isNotEmpty(msgValue)) {
				result.add(msgKey + ": " + msgValue);
			}
		}

		return result.toArray(new String[0]);
	}

	/**
	 * Remove organization from cache after receiving event that organization
	 * were changed
	 */
	@Observer(Configuration.EVENT_CLEAR_ORGANIZATION)
	public void clearOrganizationCache() throws Exception {
		log.debug("Removing organization from cache");
		cacheService.removeAll(Configuration.CACHE_APPLICATION_NAME);
	}

	/**
	 * Build DN string for organization
	 * 
	 * @return DN string for organization
	 */
	public String getDnForOrganization() {
		return getDnForOrganization(getOrganizationInum());
	}

	/**
	 * Build DN string for organization
	 * 
	 * @return DN string for organization
	 */
	public String getDnForOrganization(String inum) {
		Configuration gluuConfig = Configuration.instance();
		return String.format("o=%s,%s", inum, gluuConfig.getBaseDN());
	}

	/**
	 * Build DN string for organization
	 * 
	 * @return DN string for organization
	 */
	public String getBaseDn() {
		return Configuration.instance().getBaseDN();
	}

	/**
	 * Get Inum for organization
	 * 
	 * @return Inum for organization
	 */
	public String getInumForOrganization() {
		return Configuration.instance().getOrgInum();
	}

	public boolean isAllowPersonModification() {
		return Configuration.instance().isAllowPersonModification(); // todo &&
																		// ApplianceService.instance().getAppliance().getManageIdentityPermission()
																		// !=
																		// null
																		// &&
																		// ApplianceService.instance().getAppliance().getProfileManagment().isBooleanValue();
	}

	public String getOrganizationInum() {
		return Configuration.instance().getOrgInum();
	}

	/**
	 * Get organizationService instance
	 * 
	 * @return OrganizationService instance
	 */
	public static OrganizationService instance() {
		return (OrganizationService) Component.getInstance(OrganizationService.class);
	}

	public GluuBoolean[] getBooleanSelectionTypes() {
		return new GluuBoolean[] { GluuBoolean.DISABLED, GluuBoolean.ENABLED };
	}

}
