package org.gluu.oxtrust.action;

import java.io.File;
import java.io.Serializable;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import org.gluu.oxtrust.ldap.service.OrganizationService;
import org.gluu.oxtrust.ldap.service.Shibboleth2ConfService;
import org.gluu.oxtrust.ldap.service.TrustService;
import org.gluu.oxtrust.model.GluuMetadataSourceType;
import org.gluu.oxtrust.model.GluuSAMLTrustRelationship;
import org.gluu.oxtrust.model.GluuStatus;
import org.gluu.oxtrust.util.Configuration;
import org.jboss.seam.ScopeType;
import org.jboss.seam.annotations.In;
import org.jboss.seam.annotations.Name;
import org.jboss.seam.annotations.Scope;
import org.jboss.seam.annotations.security.Restrict;
import org.xdi.util.StringHelper;

@Name("federationDeconstructionAction")
@Scope(ScopeType.CONVERSATION)
@Restrict("#{identity.loggedIn}")
public class FederationDeconstructionAction implements Serializable {
	private static final long serialVersionUID = 1216276324815043884L;

	private List<String> bulkEntities;
	private List<String> managedEntities;
	private String filterString;
	private List<String> filteredEntities;

	@In
	private Shibboleth2ConfService shibboleth2ConfService;

	private List<String> bulkFiltered;

	private List<String> managedFiltered;

	@In
	private TrustService trustService;

	private Set<String> selectedList = new HashSet<String>();

	private GluuSAMLTrustRelationship selectedTR;

	private boolean updateDescrInProgress;

	private boolean updateNameInProgress;

	private GluuSAMLTrustRelationship trustRelationship;

	public String initFederationDeconstructions(GluuSAMLTrustRelationship trustRelationship) {
		this.trustRelationship = trustRelationship;

		return Configuration.RESULT_SUCCESS;
	}

	public void setBulkEntities(List<String> bulkEntities) {

		this.bulkEntities.removeAll(bulkFiltered);
		this.bulkEntities.addAll(bulkEntities);
	}

	public List<String> getBulkEntities() {
		if (bulkEntities == null) {
			bulkEntities = new ArrayList<String>();
			if (trustRelationship.getContainerFederation() != null) {
				trustRelationship = trustRelationship.getContainerFederation();
			}

			bulkEntities.addAll(trustRelationship.getGluuEntityId());
			List<GluuSAMLTrustRelationship> currentDeconstruction = trustService.getDeconstructedTrustRelationships(trustRelationship);
			for (GluuSAMLTrustRelationship configuredTR : currentDeconstruction) {
				bulkEntities.remove(configuredTR.getEntityId());
			}

		}
		bulkFiltered = new ArrayList<String>();
		bulkFiltered.addAll(bulkEntities);

		if (filteredEntities != null) {
			bulkFiltered.retainAll(filteredEntities);

		}
		return bulkFiltered;
	}

	public void setManagedEntities(List<String> managedEntities) {
		this.managedEntities.removeAll(managedFiltered);
		this.managedEntities.addAll(managedEntities);
	}

	public List<String> getManagedEntities() {
		if (managedEntities == null) {
			List<GluuSAMLTrustRelationship> currentDeconstruction = trustService.getDeconstructedTrustRelationships(trustRelationship);
			managedEntities = new ArrayList<String>();
			for (GluuSAMLTrustRelationship configuredTR : currentDeconstruction) {
				managedEntities.add(configuredTR.getEntityId());
			}
		}

		managedFiltered = new ArrayList<String>();
		managedFiltered.addAll(managedEntities);
		if (filteredEntities != null) {
			managedFiltered.retainAll(filteredEntities);
		}
		return managedFiltered;

	}

	public void setFilterString(String filterString) {
		this.filterString = filterString;
	}

	public String getFilterString() {
		return filterString;
	}

	public String saveEntityList() {
		List<GluuSAMLTrustRelationship> currentDeconstruction = trustService.getDeconstructedTrustRelationships(trustRelationship);
		List<String> existingTRs = new ArrayList<String>();
		for (GluuSAMLTrustRelationship configuredTR : currentDeconstruction) {
			if (managedEntities.contains(configuredTR.getEntityId())) {
				// Filter not changed entities out.
				managedEntities.remove(configuredTR.getEntityId());
				existingTRs.add(configuredTR.getEntityId());
			} else {
				// Remove removed entities
				trustService.removeTrustRelationship(configuredTR);
			}
		}
		// Add new entities
		for (String entityName : managedEntities) {
			GluuSAMLTrustRelationship newTR = new GluuSAMLTrustRelationship();
			newTR.setInum(trustService.generateInumForNewTrustRelationship());
			String dn = trustService.getDnForTrustRelationShip(newTR.getInum());
			newTR.setDn(dn);
			newTR.setMaxRefreshDelay("PT8H");
			newTR.setOwner(OrganizationService.instance().getOrganization().getDn());
			newTR.setSpMetaDataSourceType(GluuMetadataSourceType.FEDERATION);
			newTR.setContainerFederation(trustRelationship);
			newTR.setEntityId(entityName);
			newTR.setDisplayName(entityName);
			newTR.setDescription(entityName);
			newTR.setStatus(GluuStatus.ACTIVE);
			trustService.addTrustRelationship(newTR);
		}
		// Get final List
		managedEntities.addAll(existingTRs);
		if (selectedTR != null && managedEntities.contains(selectedTR.getEntityId())) {
			trustService.updateTrustRelationship(selectedTR);
		}
		return Configuration.RESULT_SUCCESS;
	}

	public void filterEntities() {
		filteredEntities = null;
		if (StringHelper.isNotEmpty(filterString)) {
			filteredEntities = new ArrayList<String>();
			String idpMetadataFolder = Configuration.instance().getShibboleth2IdpRootDir() + File.separator
					+ Shibboleth2ConfService.SHIB2_IDP_METADATA_FOLDER + File.separator;
			File metadataFile = new File(idpMetadataFolder + trustRelationship.getSpMetaDataFN());
			for (String entity : shibboleth2ConfService.getEntityIdFromMetadataFile(metadataFile)) {
				if (entity.toLowerCase().contains(filterString.toLowerCase())) {
					filteredEntities.add(entity);
				}
			}
		}
	}

	public boolean isFiltered(String entity) {
		return filteredEntities == null || filteredEntities.contains(entity);
	}

	public void setSelectedList(Set<String> selectedList) {
		if (selectedList.size() > 0) {
			for (GluuSAMLTrustRelationship trust : trustService.getDeconstructedTrustRelationships(trustRelationship)) {
				if (selectedList.toArray(new String[] {})[0].equals(trust.getEntityId())) {
					if (getSelectedTR() != null && !trust.equals(getSelectedTR())) {
						/*
						 * This flag is used to counter JSF behavior of
						 * automatic submit of a form on re-render. When text
						 * inputs are being re-rendered after new selection has
						 * been made - first of all they submit their current
						 * values, which leads to situation where values of
						 * previous object are assigned to the new object.
						 * 
						 * To counter this we pass first submit after changed
						 * selection.
						 */
						updateNameInProgress = true;
						updateDescrInProgress = true;
					}
					setSelectedTR(trust);
					break;
				}
			}
		} else {
			updateNameInProgress = true;
			updateDescrInProgress = true;
			setSelectedTR(null);
		}

	}

	public Set<String> getSelectedList() {
		return selectedList;
	}

	public GluuSAMLTrustRelationship getSelectedTR() {
		return selectedTR;
	}

	private void setSelectedTR(GluuSAMLTrustRelationship selectedTR) {
		this.selectedTR = selectedTR;
	}

	public String getSelectedTRDisplayName() {
		return selectedTR == null ? null : selectedTR.getDisplayName();
	}

	public void setSelectedTRDisplayName(String displayName) {
		if (!updateNameInProgress) {
			selectedTR.setDisplayName(displayName);
		} else {
			updateNameInProgress = false;
		}
	}

	public String getSelectedTRDescription() {
		return selectedTR == null ? null : selectedTR.getDescription();
	}

	public void setSelectedTRDescription(String description) {
		if (!updateDescrInProgress) {
			selectedTR.setDescription(description);
		} else {
			updateDescrInProgress = false;
		}
	}

}
