package org.gluu.oxtrust.ldap.cache.service;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;

import org.apache.commons.io.FileUtils;
import org.apache.commons.io.IOUtils;
import org.apache.commons.io.filefilter.WildcardFileFilter;
import org.jboss.seam.ScopeType;
import org.jboss.seam.annotations.AutoCreate;
import org.jboss.seam.annotations.In;
import org.jboss.seam.annotations.Logger;
import org.jboss.seam.annotations.Name;
import org.jboss.seam.annotations.Scope;
import org.jboss.seam.annotations.Startup;
import org.jboss.seam.log.Log;
import org.xdi.util.ArrayHelper;

/**
 * Helper service to work with snapshots
 * 
 * @author Yuriy Movchan Date: 06.09.2011
 */
@Name("cacheRefreshSnapshotFileService")
@Scope(ScopeType.APPLICATION)
@AutoCreate
@Startup(depends = "appInitializer")
public class CacheRefreshSnapshotFileService {

	@Logger
	Log log;

	private static final String SNAPSHOT_FILE_NAME_PATTERN = "inum-snapshot-%s.txt";
	private static final String PROBLEM_LIST_FILE_NAME = "problem-inum-list.txt";
	private static final String SNAPSHOT_FILE_NAME_DATE_PATTERN = "yyyy-MM-dd-HH-mm";

	@In(required = false)
	private CacheRefreshConfiguration cacheRefreshConfiguration;

	public boolean prepareSnapshotsFolder() {
		String snapshotFolder = cacheRefreshConfiguration.getSnapshotFolder();

		try {
			File dir = new File(snapshotFolder);
			if (!dir.exists()) {
				FileUtils.forceMkdir(dir);
			}
		} catch (IOException ex) {
			log.error("Failed to create snapshot folder '{0}'", ex, snapshotFolder);
			return false;
		}

		return true;
	}

	public boolean createSnapshot(Map<String, Integer> inumWithEntryHashCodeMap) {
		if (!prepareSnapshotsFolder()) {
			return false;
		}

		DateFormat fileNameDateFormat = new SimpleDateFormat(SNAPSHOT_FILE_NAME_DATE_PATTERN);
		String snapshotFileName = String.format(SNAPSHOT_FILE_NAME_PATTERN, fileNameDateFormat.format(new Date()));

		File file = new File(cacheRefreshConfiguration.getSnapshotFolder() + File.separator + snapshotFileName);
		BufferedWriter bos;
		try {
			bos = new BufferedWriter(new FileWriter(file));
		} catch (IOException ex) {
			log.error("Failed to create snapshot file '{0}'", ex, file.getAbsolutePath());
			return false;
		}

		try {
			for (Entry<String, Integer> entry : inumWithEntryHashCodeMap.entrySet()) {
				bos.write(String.format("%s:%d\n", entry.getKey(), entry.getValue()));
			}
			bos.flush();
		} catch (IOException ex) {
			log.error("Failed to create snapshot file '{0}'", ex, file.getAbsolutePath());
			return false;
		} finally {
			IOUtils.closeQuietly(bos);
		}

		return true;
	}

	public Map<String, Integer> readSnapshot(String snapshotFileName) {
		if (!prepareSnapshotsFolder()) {
			return null;
		}

		File file = new File(cacheRefreshConfiguration.getSnapshotFolder() + File.separator + snapshotFileName);
		if (!file.exists()) {
			return null;
		}

		BufferedReader bis;
		try {
			bis = new BufferedReader(new FileReader(file));
		} catch (FileNotFoundException ex) {
			log.error("Failed to load snapshot file '{0}'", ex, file.getAbsolutePath());
			return null;
		}

		Map<String, Integer> result = new HashMap<String, Integer>();
		try {
			String line;
			while ((line = bis.readLine()) != null) {
				String[] lineValues = line.split(":");
				if (lineValues.length != 2) {
					log.error("Failed to parse line: {0}", line);
					return null;
				}

				try {
					result.put(lineValues[0], Integer.valueOf(lineValues[1]));
				} catch (RuntimeException ex) {
					log.error("Failed to parse '%s' to integer", ex, lineValues[1]);
					return null;
				}
			}
		} catch (IOException ex) {
			log.error("Failed to load snapshot file '{0}'", ex, file.getAbsolutePath());
			return null;
		} finally {
			IOUtils.closeQuietly(bis);
		}

		return result;
	}

	public Map<String, Integer> readLastSnapshot() {
		if (!prepareSnapshotsFolder()) {
			return null;
		}

		String[] snapshots = getSnapshotsList();
		if (ArrayHelper.isEmpty(snapshots)) {
			return null;
		}

		return readSnapshot(snapshots[snapshots.length - 1]);
	}

	private String[] getSnapshotsList() {
		File file = new File(cacheRefreshConfiguration.getSnapshotFolder());
		String[] files = file.list(new WildcardFileFilter(String.format(SNAPSHOT_FILE_NAME_PATTERN, "*")));
		Arrays.sort(files);

		return files;
	}

	public boolean retainSnapshots(int count) {
		if (!prepareSnapshotsFolder()) {
			return false;
		}

		String[] snapshots = getSnapshotsList();
		if (ArrayHelper.isEmpty(snapshots)) {
			return true;
		}

		for (int i = 0; i < snapshots.length - count; i++) {
			File file = new File(cacheRefreshConfiguration.getSnapshotFolder() + File.separator + snapshots[i]);
			if (!file.delete()) {
				log.error("Failed to remove snaphost file '{0}'", file.getAbsolutePath());
			}
		}

		return true;
	}

	public List<String> readProblemList() {
		if (!prepareSnapshotsFolder()) {
			return null;
		}

		File file = new File(cacheRefreshConfiguration.getSnapshotFolder() + File.separator + PROBLEM_LIST_FILE_NAME);
		if (!file.exists()) {
			return null;
		}

		BufferedReader bis;
		try {
			bis = new BufferedReader(new FileReader(file));
		} catch (FileNotFoundException ex) {
			log.error("Failed to load problem list from file '{0}'", ex, file.getAbsolutePath());
			return null;
		}

		List<String> result = new ArrayList<String>();
		try {
			String line;
			while ((line = bis.readLine()) != null) {
				result.add(line);
			}
		} catch (IOException ex) {
			log.error("Failed to load problem list from file '{0}'", ex, file.getAbsolutePath());
			return null;
		} finally {
			IOUtils.closeQuietly(bis);
		}

		return result;
	}

	public boolean writeProblemList(Set<String> changedInums) {
		if (!prepareSnapshotsFolder()) {
			return false;
		}

		File file = new File(cacheRefreshConfiguration.getSnapshotFolder() + File.separator + PROBLEM_LIST_FILE_NAME);
		BufferedWriter bos;
		try {
			bos = new BufferedWriter(new FileWriter(file));
		} catch (IOException ex) {
			log.error("Failed to write problem list to file '{0}'", ex, file.getAbsolutePath());
			return false;
		}

		try {
			for (String changedInum : changedInums) {
				bos.write(String.format("%s\n", changedInum));
			}
			bos.flush();
		} catch (IOException ex) {
			log.error("Failed to write problem list to file '{0}'", ex, file.getAbsolutePath());
			return false;
		} finally {
			IOUtils.closeQuietly(bos);
		}

		return true;
	}

}
