package org.gluu.oxtrust.ldap.service;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.gluu.oxtrust.model.GluuAttribute;
import org.gluu.oxtrust.model.GluuAttributeDataType;
import org.gluu.oxtrust.model.GluuAttributePrivacyLevel;
import org.gluu.oxtrust.model.GluuAttributeUsageType;
import org.gluu.oxtrust.model.GluuCustomAttribute;
import org.gluu.oxtrust.model.GluuCustomPerson;
import org.gluu.oxtrust.model.GluuSAMLTrustRelationship;
import org.gluu.oxtrust.model.GluuUserRole;
import org.gluu.oxtrust.model.OxMultivalued;
import org.gluu.oxtrust.model.ScimCustomAtribute;
import org.gluu.oxtrust.util.OxTrustConstants;
import org.gluu.site.ldap.persistence.LdapEntryManager;
import org.jboss.seam.Component;
import org.jboss.seam.ScopeType;
import org.jboss.seam.annotations.AutoCreate;
import org.jboss.seam.annotations.In;
import org.jboss.seam.annotations.Logger;
import org.jboss.seam.annotations.Name;
import org.jboss.seam.annotations.Observer;
import org.jboss.seam.annotations.Scope;
import org.jboss.seam.core.Events;
import org.jboss.seam.log.Log;
import org.xdi.config.oxtrust.ApplicationConfiguration;
import org.xdi.service.CacheService;
import org.xdi.util.INumGenerator;
import org.xdi.util.StringHelper;

import com.unboundid.ldap.sdk.Filter;
import com.unboundid.ldap.sdk.LDAPException;

/**
 * Provides operations with attributes
 * 
 * @author Yuriy Movchan Date: 10.13.2010
 */
@Scope(ScopeType.STATELESS)
@Name("attributeService")
@AutoCreate
public class AttributeService {

	@Logger
	private Log log;

	@In
	private LdapEntryManager ldapEntryManager;

	@In
	private CacheService cacheService;

	@In
	private SchemaService schemaService;

	@In(value = "#{oxTrustConfiguration.applicationConfiguration}")
	private ApplicationConfiguration applicationConfiguration;

	public static final String CUSTOM_ATTRIBUTE_OBJECTCLASS_PREFIX = "ox-";

	/**
	 * Get all attributes
	 * 
	 * @return List of attributes
	 */
	@SuppressWarnings("unchecked")
	public List<GluuAttribute> getAllAttributes() {
		List<GluuAttribute> attributeList = (List<GluuAttribute>) cacheService.get(OxTrustConstants.CACHE_ATTRIBUTE_NAME,
				OxTrustConstants.CACHE_ATTRIBUTE_KEY_LIST);
		if (attributeList == null) {
			attributeList = getAllAtributesImpl();
			cacheService.put(OxTrustConstants.CACHE_ATTRIBUTE_NAME, OxTrustConstants.CACHE_ATTRIBUTE_KEY_LIST, attributeList);
		}

		return attributeList;
	}

	/**
	 * Get all attributes
	 * 
	 * @return List of attributes
	 */
	private List<GluuAttribute> getAllAtributesImpl() {
		List<GluuAttribute> attributeList = ldapEntryManager.findEntries(getDnForAttribute(null), GluuAttribute.class, null);
		String customOrigin = getCustomOrigin();
		for (GluuAttribute attribute : attributeList) {
			attribute.setCustom(customOrigin.equals(attribute.getOrigin()));
		}

		return attributeList;
	}

	/**
	 * Get all person attributes
	 * 
	 * @param gluuUserRole
	 *            User role
	 * @return List of person attributes
	 */
	@SuppressWarnings("unchecked")
	public List<GluuAttribute> getAllPersonAttributes(GluuUserRole gluuUserRole) {
		String key = OxTrustConstants.CACHE_ATTRIBUTE_PERSON_KEY_LIST + "_" + gluuUserRole.getValue();
		List<GluuAttribute> attributeList = (List<GluuAttribute>) cacheService.get(OxTrustConstants.CACHE_ATTRIBUTE_NAME, key);
		if (attributeList == null) {
			attributeList = getAllPersonAtributesImpl(gluuUserRole, getAllAttributes());
			cacheService.put(OxTrustConstants.CACHE_ATTRIBUTE_NAME, key, attributeList);
		}

		return attributeList;
	}

	/**
	 * Get all organization attributes
	 * 
	 * @param attributes
	 *            List of attributes
	 * @return List of organization attributes
	 */
	private List<GluuAttribute> getAllPersonAtributesImpl(GluuUserRole gluuUserRole, Collection<GluuAttribute> attributes) {
		List<GluuAttribute> returnAttributeList = new ArrayList<GluuAttribute>();

		String[] objectClassTypes = applicationConfiguration.getPersonObjectClassTypes();
		log.debug("objectClassTypes={0}", Arrays.toString(objectClassTypes));
		for (GluuAttribute attribute : attributes) {
			for (String objectClassType : objectClassTypes) {
				if (attribute.getOrigin().equals(objectClassType)
						&& ((attribute.allowViewBy(gluuUserRole) || attribute.allowEditBy(gluuUserRole)))) {
					returnAttributeList.add(attribute);
					break;
				}
			}
		}

		return returnAttributeList;
	}

	/**
	 * Get all contact attributes
	 * 
	 * @return List of contact attributes
	 */
	@SuppressWarnings("unchecked")
	public List<GluuAttribute> getAllContactAttributes(GluuUserRole gluuUserRole) {
		String key = OxTrustConstants.CACHE_ATTRIBUTE_CONTACT_KEY_LIST + "_" + gluuUserRole.getValue();
		List<GluuAttribute> attributeList = (List<GluuAttribute>) cacheService.get(OxTrustConstants.CACHE_ATTRIBUTE_NAME, key);
		if (attributeList == null) {
			attributeList = getAllContactAtributesImpl(gluuUserRole, getAllAttributes());
			cacheService.put(OxTrustConstants.CACHE_ATTRIBUTE_NAME, key, attributeList);
		}

		return attributeList;
	}

	/**
	 * Get all contact attributes
	 * 
	 * @param attributes
	 *            List of attributes
	 * @return List of contact attributes
	 */
	private List<GluuAttribute> getAllContactAtributesImpl(GluuUserRole gluuUserRole, Collection<GluuAttribute> attributes) {
		List<GluuAttribute> returnAttributeList = new ArrayList<GluuAttribute>();

		String[] objectClassTypes = applicationConfiguration.getContactObjectClassTypes();
		for (GluuAttribute attribute : attributes) {
			for (String objectClassType : objectClassTypes) {
				if (attribute.getOrigin().equals(objectClassType)
						&& (attribute.allowViewBy(gluuUserRole) || attribute.allowEditBy(gluuUserRole))) {
					returnAttributeList.add(attribute);
					break;
				}
			}
		}

		return returnAttributeList;
	}

	/**
	 * Get all origins
	 * 
	 * @return List of origins
	 */
	@SuppressWarnings("unchecked")
	public List<String> getAllAttributeOrigins() {
		List<String> attributeOriginList = (List<String>) cacheService.get(OxTrustConstants.CACHE_ATTRIBUTE_NAME,
				OxTrustConstants.CACHE_ATTRIBUTE_ORIGIN_KEY_LIST);
		if (attributeOriginList == null) {
			attributeOriginList = getAllAttributeOrigins(getAllAttributes());
			cacheService.put(OxTrustConstants.CACHE_ATTRIBUTE_NAME, OxTrustConstants.CACHE_ATTRIBUTE_ORIGIN_KEY_LIST, attributeOriginList);
		}

		return attributeOriginList;
	}

	/**
	 * Get all origins
	 * 
	 * @param attributes
	 *            List of attributes
	 * @return List of origins
	 */
	public List<String> getAllAttributeOrigins(Collection<GluuAttribute> attributes) {
		List<String> attributeOriginList = new ArrayList<String>();

		for (GluuAttribute attribute : attributes) {
			String origin = attribute.getOrigin();
			if (!attributeOriginList.contains(origin)) {
				attributeOriginList.add(attribute.getOrigin());
			}
		}

		String customOrigin = getCustomOrigin();
		if (!attributeOriginList.contains(customOrigin)) {
			attributeOriginList.add(customOrigin);
		}

		return attributeOriginList;
	}

	/**
	 * Get origin display names
	 * 
	 * @param attributes
	 *            List of origins
	 * @param objectClassTypes
	 *            List of objectClasses
	 * @param objectClassDisplayNames
	 *            List of display names for objectClasses
	 * @return Map with key = origin and value = display name
	 */
	public Map<String, String> getAllAttributeOriginDisplayNames(List<String> attributeOriginList, String[] objectClassTypes,
			String[] objectClassDisplayNames) {
		// Put displayNames = origins if user don't specify right mapping in
		// properties file
		Map<String, String> attributeOriginDisplayNameMap = new HashMap<String, String>();
		for (String origin : attributeOriginList) {
			attributeOriginDisplayNameMap.put(origin, origin);
		}

		if (objectClassTypes.length == objectClassDisplayNames.length) {
			for (int i = 0; i < objectClassTypes.length; i++) {
				String objectClass = objectClassTypes[i];
				if (attributeOriginDisplayNameMap.containsKey(objectClass)) {
					attributeOriginDisplayNameMap.put(objectClass, objectClassDisplayNames[i]);
				}
			}
		}

		return attributeOriginDisplayNameMap;
	}

	/**
	 * Get custom attributes
	 * 
	 * @return List of cusomt attributes
	 */
	@SuppressWarnings("unchecked")
	public List<GluuAttribute> getCustomAttributes() {
		List<GluuAttribute> attributeList = (List<GluuAttribute>) cacheService.get(OxTrustConstants.CACHE_ATTRIBUTE_NAME,
				OxTrustConstants.CACHE_ATTRIBUTE_CUSTOM_KEY_LIST);
		if (attributeList == null) {
			attributeList = new ArrayList<GluuAttribute>();
			for (GluuAttribute attribute : getAllAttributes()) {
				if (attribute.isCustom()) {
					attributeList.add(attribute);
				}
			}

			cacheService.put(OxTrustConstants.CACHE_ATTRIBUTE_NAME, OxTrustConstants.CACHE_ATTRIBUTE_CUSTOM_KEY_LIST, attributeList);
		}

		return attributeList;
	}

	/**
	 * Get attribute by inum
	 * 
	 * @param inum
	 *            Inum
	 * @return Attribute
	 */
	public GluuAttribute getAttributeByInum(String inum) {
		return getAttributeByInum(inum, getAllAttributes());
	}

	public GluuAttribute getAttributeByInum(String inum, List<GluuAttribute> attributes) {
		for (GluuAttribute attribute : attributes) {
			if (attribute.getInum().equals(inum)) {
				return attribute;
			}
		}

		return null;
	}

	/**
	 * Get attribute by name
	 * 
	 * @param name
	 *            Name
	 * @return Attribute
	 */
	public GluuAttribute getAttributeByName(String name) {
		return getAttributeByName(name, getAllAttributes());
	}

	/**
	 * Get attribute by name
	 * 
	 * @param name
	 *            Name
	 * @param attributes
	 *            List of attributes
	 * @return Attribute
	 */
	public GluuAttribute getAttributeByName(String name, List<GluuAttribute> attributes) {
		for (GluuAttribute attribute : attributes) {
			if (attribute.getName().equals(name)) {
				return attribute;
			}
		}

		return null;
	}

	/**
	 * Get SCIM related attributes
	 * 
	 * @return Attribute
	 */
	public List<GluuAttribute> getSCIMRelatedAttributes() throws Exception {
		return getSCIMRelatedAttributesImpl(getAllAttributes());
	}

	/**
	 * Get SCIMAttributes
	 * 
	 * @param attributes
	 *            List of attributes
	 * @return list of Attributes
	 */
	public List<GluuAttribute> getSCIMRelatedAttributesImpl(List<GluuAttribute> attributes) throws Exception {

		List<GluuAttribute> result = new ArrayList<GluuAttribute>();

		for (GluuAttribute attribute : attributes) {

			boolean isEmpty = attribute.getOxSCIMCustomAttribute() == null;

			if (!isEmpty) {

				if (attribute.getOxSCIMCustomAttribute().getValue().equalsIgnoreCase("true")) {
					result.add(attribute);
				}
			}
		}

		return result;
	}

	/**
	 * Add new custom attribute
	 * 
	 * @param attribute
	 *            Attribute
	 */
	public void addAttribute(GluuAttribute attribute) {
		ldapEntryManager.persist(attribute);

		Events.instance().raiseEvent(OxTrustConstants.EVENT_CLEAR_ATTRIBUTES);
	}

	/**
	 * Remove attribute with specified Inum
	 * 
	 * @param inum
	 *            Inum
	 */
	public void removeAttribute(String inum) {
		GluuAttribute attribute = new GluuAttribute();
		attribute.setDn(getDnForAttribute(inum));

		removeAttribute(attribute);
	}

	/**
	 * Remove attribute
	 * 
	 * @param attribute
	 *            Attribute
	 */
	public boolean removeAttribute(GluuAttribute attribute) {
		log.info("Attribute removal started");
		log.trace("Getting attribute information");
		
		String objectClassName = getCustomOrigin();
		log.debug("objectClassName is " + objectClassName);
		
		String attributeName = attribute.getName();
		log.debug("attributeName is " + attributeName);
		
		log.trace("Removing attribute from people");
		List<GluuCustomPerson> people = PersonService.instance().searchPersons("", OxTrustConstants.searchPersonsSizeLimit);
		log.trace(String.format("Iterating %d people", people.size()));
		for (GluuCustomPerson person : people) {
			log.trace(String.format("Analyzing %s.", person.getUid()));
			List<GluuCustomAttribute> customAttrs = person.getCustomAttributes();
			for (GluuCustomAttribute attr : customAttrs) {
				log.trace(String.format("%s has custom attribute %s", person.getUid(), attr.getName()));
				if (attr.getName().equals(attributeName)) {
					log.trace(String.format("%s matches %s .  deleting it.", attr.getName(), attributeName));
					customAttrs.remove(attr);
					person.setCustomAttributes(customAttrs);
					PersonService.instance().updatePerson(person);
					break;
				}
			}
		}

		log.trace("Removing attribute from trustRelationships");
		List<GluuSAMLTrustRelationship> trustRelationships = TrustService.instance().getAllTrustRelationships();
		log.trace(String.format("Iterating %d trustRelationships", trustRelationships.size()));
		for (GluuSAMLTrustRelationship trustRelationship : trustRelationships) {
			log.trace(String.format("Analyzing %s.", trustRelationship.getDisplayName()));
			List<String> customAttrs = trustRelationship.getReleasedAttributes();
			if (customAttrs != null) {
				for (String attrDN : customAttrs) {
					log.trace(String.format("%s has custom attribute %s", trustRelationship.getDisplayName(), attrDN));
					if (attrDN.equals(attribute.getDn())) {
						log.trace(String.format("%s matches %s .  deleting it.", attrDN, attribute.getDn()));
						List<String> updatedAttrs = new ArrayList<String>();
						updatedAttrs.addAll(customAttrs);
						updatedAttrs.remove(attrDN);
						if (updatedAttrs.size() == 0) {
							trustRelationship.setReleasedAttributes(null);
						} else {
							trustRelationship.setReleasedAttributes(updatedAttrs);
						}

						TrustService.instance().updateTrustRelationship(trustRelationship);
						break;
					}
				}
			}
		}
		log.trace("Removing attribute from objectclass");
		// Unregister new attribute type from custom object class
		try {
			schemaService.removeAttributeTypeFromObjectClass(objectClassName, attributeName);
		} catch (Exception ex) {
			log.error("Failed to remove attribute type from LDAP schema's object class", ex);
			return false;
		}
		
		log.trace("Removing attribute from schema");
		// Remove attribute type from LDAP schema
		try {
			schemaService.removeStringAttribute(attributeName);
		} catch (Exception ex) {
			log.error("Failed to remove attribute type from LDAP schema", ex);
			return false;
		}

		log.trace("Removing attribute for good.");
		ldapEntryManager.remove(attribute);
		Events.instance().raiseEvent(OxTrustConstants.EVENT_CLEAR_ATTRIBUTES);
		return true;
	}

	/**
	 * Update specified attribute
	 * 
	 * @param attribute
	 *            Attribute
	 */
	public void updateAttribute(GluuAttribute attribute) {
		ldapEntryManager.merge(attribute);
		Events.instance().raiseEvent(OxTrustConstants.EVENT_CLEAR_ATTRIBUTES);
	}

	/**
	 * Clear attributes cache after receiving event that attributes were changed
	 */
	@Observer(OxTrustConstants.EVENT_CLEAR_ATTRIBUTES)
	public void clearAttributesCache() {
		log.debug("Removing attributes from cache");
		cacheService.removeAll(OxTrustConstants.CACHE_ATTRIBUTE_NAME);
	}

	/**
	 * Get all available data types
	 * 
	 * @return Array of data types
	 */
	public GluuAttributeDataType[] getDataTypes() {
		return GluuAttributeDataType.values();
	}

	/**
	 * Get all available Scim Custom Atributes
	 * 
	 * @return Array of data types
	 */
	public ScimCustomAtribute[] getScimCustomAttribute() throws Exception {
		return ScimCustomAtribute.values();
	}

	/**
	 * Get all available oxMultivalued attributes
	 * 
	 * @return Array of data types
	 */
	public OxMultivalued[] getOxMultivalued() throws Exception {
		return OxMultivalued.values();
	}

	/**
	 * Get all available attribute user roles
	 * 
	 * @return Array of attribute user roles
	 */
	public GluuUserRole[] getAttributeUserRoles() throws Exception {
		return new GluuUserRole[] { GluuUserRole.ADMIN, GluuUserRole.USER };
	}

	/**
	 * Get all available attribute view types
	 * 
	 * @return Array of attribute user roles
	 */
	public GluuUserRole[] getViewTypes() {
		if (ApplianceService.instance().getAppliance().getWhitePagesEnabled() != null
				&& ApplianceService.instance().getAppliance().getWhitePagesEnabled().isBooleanValue()) {
			return new GluuUserRole[] { GluuUserRole.ADMIN, GluuUserRole.USER, GluuUserRole.WHITEPAGES };
		}
		return new GluuUserRole[] { GluuUserRole.ADMIN, GluuUserRole.USER };
	}

	/**
	 * Get all usage types
	 * 
	 * @return Array of Usage types
	 */
	public GluuAttributeUsageType[] getAttributeUsageTypes() {
		return GluuAttributeUsageType.values();
	}

	/**
	 * Get all available privacy levels
	 * 
	 * @return Array of Privacy levels
	 */
	public GluuAttributePrivacyLevel[] getPrivacyLevels() {
		return GluuAttributePrivacyLevel.values();
	}

	/**
	 * Check if LDAP server contains attribute with specified attributes
	 * 
	 * @return True if attribute with specified attributes exist
	 */
	public boolean containsAttribute(GluuAttribute attribute) {
		return ldapEntryManager.contains(attribute);
	}

	/**
	 * Generate new inum for attribute
	 * 
	 * @return New inum for attribute
	 */
	public String generateInumForNewAttribute() {
		GluuAttribute attribute = null;
		String newInum = null;
		do {
			newInum = generateInumForNewAttributeImpl();
			String newDn = getDnForAttribute(newInum);
			attribute = new GluuAttribute();
			attribute.setDn(newDn);
		} while (containsAttribute(attribute));

		return newInum;
	}

	/**
	 * Convert inum to string without delimiters
	 * 
	 * @return Inum string without delimiters
	 */
	public String toInumWithoutDelimiters(String inum) {
		return inum.replace(".", "").replace(OxTrustConstants.inumDelimiter, "").replace("@", "");
	}

	/**
	 * Generate new inum for attribute
	 * 
	 * @return New inum for attribute
	 * @throws Exception
	 */
	private String generateInumForNewAttributeImpl() {
		String orgInum = OrganizationService.instance().getInumForOrganization();
		return orgInum + OxTrustConstants.inumDelimiter + "0005" + OxTrustConstants.inumDelimiter + generateInum();
	}

	private String generateInum() {
		String inum = "";
		int value;
		while (true) {
			inum = INumGenerator.generate(1);
			try {
				value = Integer.parseInt(inum, 16);
				if (value < 7) {
					continue;
				}
			} catch (Exception ex) {
				log.error("Error generating inum: ", ex);

			}
			break;
		}
		return inum;
	}

	/**
	 * Build DN string for attribute
	 * 
	 * @param inum
	 *            Inum
	 * @return DN string for specified attribute or DN for attributes branch if
	 *         inum is null
	 */
	public String getDnForAttribute(String inum) {
		String organizationDn = OrganizationService.instance().getDnForOrganization();
		if (StringHelper.isEmpty(inum)) {
			return String.format("ou=attributes,%s", organizationDn);
		}

		return String.format("inum=%s,ou=attributes,%s", inum, organizationDn);
	}

	/**
	 * Return current custom origin
	 * 
	 * @return Current custom origin
	 */
	public String getCustomOrigin() {
		return applicationConfiguration.getPersonCustomObjectClass();
		// return CUSTOM_ATTRIBUTE_OBJECTCLASS_PREFIX +
		// toInumWithoutDelimiters(organizationService.getInumForOrganization());
	}

	/**
	 * Set metadata for every custom attribute
	 * 
	 * @param customAttributes
	 *            List of custom attributes
	 * @param attributes
	 *            List of attributes
	 */
	public void setAttributeMetadata(List<GluuCustomAttribute> customAttributes, List<GluuAttribute> attributes) {
		if ((customAttributes == null) || (attributes == null)) {
			return;
		}

		for (GluuCustomAttribute personAttribute : customAttributes) {
			GluuAttribute tmpAttribute = getAttributeByName(personAttribute.getName(), attributes);
			if (tmpAttribute == null) {
				log.error("Failed to find attribute '{0}' metadata", personAttribute.getName());
			}

			personAttribute.setMetadata(tmpAttribute);
		}
	}

	/**
	 * Get custom attributes by attribute DNs
	 * 
	 * @param customAttributes
	 *            List of attribute DNs
	 * @param attributes
	 *            List of custom attributes
	 */
	public List<GluuCustomAttribute> getCustomAttributesByAttributeDNs(List<String> attributeDNs,
			HashMap<String, GluuAttribute> attributesByDNs) {
		List<GluuCustomAttribute> customAttributes = new ArrayList<GluuCustomAttribute>();
		if (attributeDNs == null) {
			return customAttributes;
		}

		for (String releasedAttributeDn : attributeDNs) {
			GluuAttribute attribute = attributesByDNs.get(releasedAttributeDn);
			if (attribute != null) {
				GluuCustomAttribute customAttribute = new GluuCustomAttribute(attribute.getName(), releasedAttributeDn);
				customAttribute.setMetadata(attribute);
				customAttributes.add(customAttribute);
			}
		}

		return customAttributes;
	}

	public HashMap<String, GluuAttribute> getAttributeMapByDNs(List<GluuAttribute> attributes) {
		HashMap<String, GluuAttribute> attributeDns = new HashMap<String, GluuAttribute>();
		for (GluuAttribute attribute : attributes) {
			attributeDns.put(attribute.getDn(), attribute);
		}

		return attributeDns;
	}

	public void sortCustomAttributes(List<GluuCustomAttribute> customAttributes, String sortByProperties) {
		ldapEntryManager.sortListByProperties(GluuCustomAttribute.class, customAttributes, sortByProperties);
	}

	public static AttributeService instance() {
		return (AttributeService) Component.getInstance(AttributeService.class);
	}

	/**
	 * Build DN string for group
	 * 
	 * @param inum
	 *            Group Inum
	 * @return DN string for specified group or DN for groups branch if inum is
	 *         null
	 */
	public String getDnForGroup(String inum) throws Exception {
		String orgDn = OrganizationService.instance().getDnForOrganization();
		if (StringHelper.isEmpty(inum)) {
			return String.format("ou=groups,%s", orgDn);
		}

		return String.format("inum=%s,ou=groups,%s", inum, orgDn);
	}

	/**
	 * @param admin
	 * @return
	 */
	public List<GluuAttribute> getAllActivePersonAttributes(GluuUserRole admin) {
		@SuppressWarnings("unchecked")
		List<GluuAttribute> activeAttributeList = (List<GluuAttribute>) cacheService.get(OxTrustConstants.CACHE_ACTIVE_ATTRIBUTE_NAME,
				OxTrustConstants.CACHE_ACTIVE_ATTRIBUTE_KEY_LIST);
		if (activeAttributeList == null) {
			try {
				activeAttributeList = getAllActiveAtributesImpl(admin);
			} catch (LDAPException e) {
				// TODO Auto-generated catch block
				e.printStackTrace();
			}
			cacheService.put(OxTrustConstants.CACHE_ACTIVE_ATTRIBUTE_NAME, OxTrustConstants.CACHE_ATTRIBUTE_KEY_LIST, activeAttributeList);
		}

		return activeAttributeList;
	}

	/**
	 * @return
	 * @throws LDAPException
	 */
	private List<GluuAttribute> getAllActiveAtributesImpl(GluuUserRole gluuUserRole) throws LDAPException {
		List<GluuAttribute> attributeList = ldapEntryManager.findEntries(getDnForAttribute(null), GluuAttribute.class,
				Filter.create("gluuStatus=active"));
		String customOrigin = getCustomOrigin();
		String[] objectClassTypes = applicationConfiguration.getPersonObjectClassTypes();
		log.debug("objectClassTypes={0}", Arrays.toString(objectClassTypes));
		List<GluuAttribute> returnAttributeList = new ArrayList<GluuAttribute>();
		for (GluuAttribute attribute : attributeList) {
			for (String objectClassType : objectClassTypes) {
				if (attribute.getOrigin().equals(objectClassType)
						&& ((attribute.allowViewBy(gluuUserRole) || attribute.allowEditBy(gluuUserRole)))) {
					attribute.setCustom(customOrigin.equals(attribute.getOrigin()));
					returnAttributeList.add(attribute);
					break;
				}
			}
		}

		return returnAttributeList;
	}

	/**
	 * Search groups by pattern
	 * 
	 * @param pattern
	 *            Pattern
	 * @param sizeLimit
	 *            Maximum count of results
	 * @return List of groups
	 * @throws Exception
	 */
	public List<GluuAttribute> searchAttributes(String pattern, int sizeLimit) throws Exception {
		String[] targetArray = new String[] { pattern };
		Filter displayNameFilter = Filter.createSubstringFilter(OxTrustConstants.displayName, null, targetArray, null);
		Filter descriptionFilter = Filter.createSubstringFilter(OxTrustConstants.description, null, targetArray, null);
		Filter inameFilter = Filter.createSubstringFilter(OxTrustConstants.iname, null, targetArray, null);
		Filter searchFilter = Filter.createORFilter(displayNameFilter, descriptionFilter, inameFilter);

		List<GluuAttribute> result = ldapEntryManager.findEntries(getDnForAttribute(null), GluuAttribute.class, searchFilter, sizeLimit);

		return result;
	}

}
