package org.xdi.graphmodel.impl.operation;

import org.xdi.graphmodel.api.Symbols;
import org.xdi.graphmodel.api.graph.XdiStatement;
import org.xdi.graphmodel.api.operation.Operation;
import org.xdi.graphmodel.api.operation.OperationFactory;
import org.xdi.graphmodel.impl.XdiStatementImpl;
import org.xdi.graphmodel.impl.xri.XriImpl;

/**
 * Operation factory implementation.
 *
 * @author Yuriy Zabrovarnyy
 * @version 0.9, 05/05/2011
 */
public class OperationFactoryImpl implements OperationFactory {

    /**
     * Instance of operation factory.
     */
    private static final OperationFactory INSTANCE = new OperationFactoryImpl();

    /**
     * Private constructor to avoid instance creation outside.
     */
    private OperationFactoryImpl() {
    }

    /**
     * Returns operation factory instance.
     *
     * @return operation factory instance
     */
    public static OperationFactory getInstance() {
        return INSTANCE;
    }

    /**
     * Creates operation object based on command xdi statement.
     *
     * @param p_statement xdi statement with operation (e.g. fromStatement$do/$get/nodexri)
     * @return operation object
     */
    public Operation create(XdiStatement p_statement) {
        if (p_statement != null && p_statement.getSubject().asString().endsWith(Symbols.DO_OPERATION.getValue())) {
            final String predicate = p_statement.getPredicate().asString();
            final String object = p_statement.getObject().asString();
            if (predicate.equals(Symbols.ADD_OPERATION.getValue()) &&
                    object.startsWith(Symbols.OPENING_PARENTHESIS.getValue()) &&
                    object.endsWith(Symbols.CLOSING_PARENTHESIS.getValue())) {
                final XdiStatement xdiStatement = XdiStatementImpl.parseSilently(object.substring(1, object.length() - 1));
                if (xdiStatement != null) {
                    return new AddOperationImpl(p_statement, xdiStatement);
                }
            } else if (predicate.equals(Symbols.DEL_OPERATION.getValue())) {
                if (object.startsWith(Symbols.OPENING_PARENTHESIS.getValue()) &&
                        object.endsWith(Symbols.CLOSING_PARENTHESIS.getValue()) &&
                        XdiStatementImpl.canParse(object.substring(1, object.length() - 1))) {
                    return new DeleteRelArcOperationImpl(p_statement, XdiStatementImpl.parseSilently(object.substring(1, object.length() - 1)));
                } else {
                    return new DeleteOperationImpl(p_statement, new XriImpl(object));
                }
            } else if (predicate.equals(Symbols.GET_OPERATION.getValue())) {
                return new GetOperationImpl(p_statement, new XriImpl(object));
            } else if (predicate.equals(Symbols.MOD_OPERATION.getValue())) {
                final XdiStatement xdiStatement = XdiStatementImpl.parseSilently(object.substring(1, object.length() - 1));
                if (xdiStatement != null && xdiStatement.getType() == XdiStatement.ArcType.LITERAL) {
                    return new ModifyOperationImpl(p_statement, xdiStatement);
                }
            } else if (predicate.equals(Symbols.COPY_OPERATION.getValue())) {
                throw new UnsupportedOperationException(); // not supported
            } else if (predicate.equals(Symbols.MOVE_OPERATION.getValue())) {
                throw new UnsupportedOperationException(); // not supported
            }
        }
        return null;
    }
}
