package org.xdi.graphmodel.impl.operation;

import org.xdi.graphmodel.api.LiteralNode;
import org.xdi.graphmodel.api.RootNode;
import org.xdi.graphmodel.api.Symbols;
import org.xdi.graphmodel.api.graph.XdiNode;
import org.xdi.graphmodel.api.graph.XdiStatement;
import org.xdi.graphmodel.api.operation.Operation;
import org.xdi.graphmodel.api.operation.OperationResult;
import org.xdi.graphmodel.api.operation.OperationType;
import org.xdi.graphmodel.api.xri.Xri;
import org.xdi.graphmodel.impl.GraphBuilderImpl;
import org.xdi.graphmodel.impl.GraphExtractorImpl;
import org.xdi.graphmodel.impl.LiteralNodeImpl;
import org.xdi.graphmodel.impl.xri.XriImpl;

/**
 * Modify operation implementation.
 *
 * @author Yuriy Zabrovarnyy
 * @version 0.9, 09/05/2011
 */
public class ModifyOperationImpl implements Operation<XdiStatement> {

    /**
     * Command statement
     */
    private final XdiStatement m_commandStatement;

    /**
     * Operand
     */
    private final XdiStatement m_operand;

    /**
     * Constructor
     *
     * @param commandStatement command statement
     * @param operand          operand
     */
    public ModifyOperationImpl(XdiStatement commandStatement, XdiStatement operand) {
        m_commandStatement = commandStatement;
        m_operand = operand;
    }

    /**
     * Apply operation to graph.
     *
     * @param p_graph root of graph
     * @return whether operation is applied successfully or not
     */
    public OperationResult apply(RootNode p_graph) {
        final XdiNode node = GraphExtractorImpl.getInstance().getNodeByXri(p_graph, getLiteralNodeXri());
        if (node instanceof LiteralNode) {
            final LiteralNode literalNode = (LiteralNode) node;
            literalNode.setValue(new LiteralNodeImpl.ValueImpl(GraphBuilderImpl.extractLiteralValueString(m_operand.getObject().asString())));
            return new OperationResultImpl(true);
        }
        return new OperationResultImpl(false);
    }

    /**
     * Gets literal node xri.
     *
     * @return literal node xri
     */
    private XriImpl getLiteralNodeXri() {
        return new XriImpl(m_operand.getSubject().asString(), Symbols.SLASH.getValue(), m_operand.getPredicate().asString());
    }

    /**
     * Original xdi statement with operation (e.g. {from segment}$do/$add/({statement})).
     *
     * @return original xdi statement with operation
     */
    public XdiStatement getOriginalCommandStatement() {
        return m_commandStatement;
    }

    /**
     * Returns operand of operation.
     *
     * @return operand of operation
     */
    public XdiStatement getOperand() {
        return m_operand;
    }

    /**
     * Gets type of operation.
     *
     * @return type of operation
     */
    public OperationType getType() {
        return OperationType.MOD;
    }

    /**
     * Gets link contract node xri.
     *
     * @return Gets link contract node xri
     */
    public Xri getLinkContractNode() {
        return getLiteralNodeXri();
    }

    /**
     * Generates command statement object.
     *
     * @return command statement object
     */
    public String generateCommandStatementObject() {
        final StringBuilder b = new StringBuilder();
        b.append("(").append(m_operand.asString()).append(")");
        return b.toString();
    }
}
