package org.xdi.graphmodel.impl.operation;

import org.xdi.graphmodel.api.RootNode;
import org.xdi.graphmodel.api.Symbols;
import org.xdi.graphmodel.api.graph.XdiArc;
import org.xdi.graphmodel.api.graph.XdiNode;
import org.xdi.graphmodel.api.graph.XdiStatement;
import org.xdi.graphmodel.api.operation.Operation;
import org.xdi.graphmodel.api.operation.OperationResult;
import org.xdi.graphmodel.api.operation.OperationType;
import org.xdi.graphmodel.api.xri.Xri;
import org.xdi.graphmodel.common.Utils;
import org.xdi.graphmodel.impl.GraphExtractorImpl;
import org.xdi.graphmodel.impl.xri.XriImpl;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Set;

/**
 * Get operation implementation.
 *
 * @author Yuriy Zabrovarnyy
 * @version 0.9, 09/05/2011
 */
public class GetOperationImpl implements Operation<Xri> {

    /**
     * Command statement
     */
    private final XdiStatement m_commandStatement;

    /**
     * Operation operand
     */
    private final Xri m_operand;

    /**
     * Link contract xri
     */
    private Xri m_linkContractXri = null;

    /**
     * Constructor
     *
     * @param commandStatement command statement
     * @param operand          operation operand
     */
    public GetOperationImpl(XdiStatement commandStatement, Xri operand) {
        m_commandStatement = commandStatement;
        m_operand = operand;
    }

    /**
     * Original xdi statement with operation (e.g. {from segment}$do/$add/({statement})).
     *
     * @return original xdi statement with operation
     */
    public XdiStatement getOriginalCommandStatement() {
        return m_commandStatement;
    }

    /**
     * Returns operand of operation.
     *
     * @return operand of operation
     */
    public Xri getOperand() {
        return m_operand;
    }

    /**
     * Gets type of operation.
     *
     * @return type of operation
     */
    public OperationType getType() {
        return OperationType.GET;
    }

    /**
     * Apply operation to graph.
     *
     * @param p_graph root of graph
     * @return whether operation is applied successfully or not
     */
    public OperationResult apply(RootNode p_graph) {
        // literal node
        final String operandStr = m_operand.asString();
        if (Utils.isEnclosedInParenthesis(operandStr) && operandStr.indexOf(Symbols.SLASH.getValue()) != -1) {
            final XdiNode targetNode = GraphExtractorImpl.getInstance().getNodeByXri(p_graph, new XriImpl(Utils.cutParenthesis(operandStr)));
            if (targetNode != null) {
                final Set<XdiArc> literalArcList = targetNode.getLiteralArcList();
                if (literalArcList != null && !literalArcList.isEmpty()) {
                    final XdiArc litArc = literalArcList.iterator().next();
                    return new OperationResultImpl(true, new ArrayList<XdiStatement>(Arrays.asList(
                            GraphExtractorImpl.constructLiteralStatementFromLiteralArc(litArc)
                    )));
                }
            }
        }

        // context node
        final List<XdiStatement> result = getStatementListByNodeXri(m_operand, p_graph);
        if (result != null && !result.isEmpty()) {
            return new OperationResultImpl(true, result);
        }

        return new OperationResultImpl(false, new ArrayList<XdiStatement>());
    }

    /**
     * Returns sub graph as list of xdi statements.
     *
     * @param p_nodeXri xri of node
     * @param p_graph   root node of graph
     * @return sub graph as list of xdi statements
     */
    public static List<XdiStatement> getStatementListByNodeXri(Xri p_nodeXri, RootNode p_graph) {
        final List<XdiStatement> result = new ArrayList<XdiStatement>();
        if (p_nodeXri != null && p_graph != null) {
            final XdiNode nodeByXri = GraphExtractorImpl.getInstance().getNodeByXri(p_graph, p_nodeXri);
            if (nodeByXri != null) {
                result.addAll(GraphExtractorImpl.getInstance().extractXdiStatementList(nodeByXri));
            }
        }
        return result;
    }

    /**
     * Gets link contract node xri.
     *
     * @return Gets link contract node xri
     */
    public Xri getLinkContractNode() {
        if (m_linkContractXri == null) {
            m_linkContractXri = Utils.normalize(m_operand);
        }
        return m_linkContractXri;
    }

    /**
     * Generates command statement object.
     *
     * @return command statement object
     */
    public String generateCommandStatementObject() {
        final String s = m_operand.asString();
        if (s.contains(Symbols.SLASH.getValue())) {
            final StringBuilder b = new StringBuilder();
            b.append("(").append(s).append(")");
            return b.toString();
        }
        return s;
    }
}
