package org.xdi.graphmodel.impl.operation;

import org.xdi.graphmodel.api.RootNode;
import org.xdi.graphmodel.api.graph.XdiStatement;
import org.xdi.graphmodel.api.operation.Operation;
import org.xdi.graphmodel.api.operation.OperationResult;
import org.xdi.graphmodel.api.operation.OperationType;
import org.xdi.graphmodel.api.xri.Xri;

/**
 * Delete Relational arc operation.
 *
 * @author Yuriy Z
 * @version 0.9, 15/12/2011
 */

public class DeleteRelArcOperationImpl implements Operation<XdiStatement> {

    /**
     * Command statement
     */
    private final XdiStatement m_commandStatement;

    /**
     * Operand
     */
    private final XdiStatement m_operand;

    /**
     * Constructor
     *
     * @param p_commandStatement command statement
     * @param p_operand          operand
     */
    public DeleteRelArcOperationImpl(XdiStatement p_commandStatement, XdiStatement p_operand) {
        m_commandStatement = p_commandStatement;
        m_operand = p_operand;
    }

    /**
     * Apply operation to graph.
     *
     * @param p_graph root of graph
     * @return result of operation execution
     */
    @Override
    public OperationResult apply(RootNode p_graph) {
        // remove only relational arc
        if (p_graph != null && m_operand.getType() == XdiStatement.ArcType.RELATIONAL) {
            final boolean ok = DeleteOperationImpl.removeRelationalArc(m_operand, p_graph);
            return new OperationResultImpl(ok);
        }
        return new OperationResultImpl(false);
    }

    /**
     * Original xdi statement with operation (e.g. {from segment}$do/$add/({statement})).
     *
     * @return original xdi statement with operation
     */
    @Override
    public XdiStatement getOriginalCommandStatement() {
        return m_commandStatement;
    }

    /**
     * Returns operand of operation.
     *
     * @return operand of operation
     */
    @Override
    public XdiStatement getOperand() {
        return m_operand;
    }

    /**
     * Gets type of operation.
     *
     * @return type of operation
     */
    @Override
    public OperationType getType() {
        return OperationType.DEL;
    }

    /**
     * Gets link contract node xri.
     *
     * @return Gets link contract node xri
     */
    @Override
    public Xri getLinkContractNode() {
        return m_operand.getSubject();
    }

    /**
     * Generates command statement object.
     *
     * @return command statement object
     */
    public String generateCommandStatementObject() {
        final StringBuilder b = new StringBuilder();
        b.append("(").append(m_operand.asString()).append(")");
        return b.toString();
    }
}
