package org.xdi.graphmodel.impl.operation;

import org.xdi.graphmodel.api.ContextNode;
import org.xdi.graphmodel.api.RootNode;
import org.xdi.graphmodel.api.Symbols;
import org.xdi.graphmodel.api.graph.XdiArc;
import org.xdi.graphmodel.api.graph.XdiNode;
import org.xdi.graphmodel.api.graph.XdiStatement;
import org.xdi.graphmodel.api.operation.Operation;
import org.xdi.graphmodel.api.operation.OperationResult;
import org.xdi.graphmodel.api.operation.OperationType;
import org.xdi.graphmodel.api.xri.Xri;
import org.xdi.graphmodel.common.Utils;
import org.xdi.graphmodel.impl.GraphExtractorImpl;
import org.xdi.graphmodel.impl.xri.XriImpl;

import java.util.ArrayList;
import java.util.List;
import java.util.Set;

/**
 * Delete operation implementation.
 *
 * @author Yuriy Zabrovarnyy
 * @version 0.9, 09/05/2011
 */
public class DeleteOperationImpl implements Operation<Xri> {

    /**
     * Command statement
     */
    private final XdiStatement m_commandStatement;

    /**
     * Operand
     */
    private final Xri m_operand;

    /**
     * Link contract xri
     */
    private Xri m_linkContractXri = null;

    /**
     * Constructor
     *
     * @param commandStatement command statement
     * @param operand          delete operation operand
     */
    public DeleteOperationImpl(XdiStatement commandStatement, Xri operand) {
        m_commandStatement = commandStatement;
        m_operand = operand;
    }

    /**
     * Apply operation to graph.
     *
     * @param p_graph root of graph
     * @return whether operation is applied successfully or not
     */
    public OperationResult apply(RootNode p_graph) {
        if (p_graph != null) {

            // check maybe it's literal node xri
            final String operandStr = m_operand.asString();
            if (Utils.isEnclosedInParenthesis(operandStr) && operandStr.contains(Symbols.SLASH.getValue())) {
                final XdiNode targetNode = getNode(p_graph, new XriImpl(Utils.cutParenthesis(operandStr)));
                if (targetNode != null) {
                    final Set<XdiArc> literalArcList = targetNode.getLiteralArcList();
                    for (XdiArc arc : literalArcList) {
                        remove(arc.getSource(), arc.getTarget(), XdiStatement.ArcType.LITERAL);
                    }
                    return new OperationResultImpl(true);
                }
            }

            // remove sub graph if exist
            final List<XdiStatement> list = GetOperationImpl.getStatementListByNodeXri(m_operand, p_graph);
            if (list != null && !list.isEmpty()) {
                // order of removing is important:
                // 1. remove relational arcs
                // 2. remove literal arcs
                // 3. and only then contextual arcs

                for (XdiStatement s : list) {
                    if (s.getType() == XdiStatement.ArcType.RELATIONAL) {
                        removeRelationalArc(s, p_graph);
                    }
                }

                for (XdiStatement s : list) {
                    if (s.getType() == XdiStatement.ArcType.LITERAL) {
                        removeLiteralArc(s, p_graph);
                    }
                }

                for (XdiStatement s : list) {
                    if (s.getType() == XdiStatement.ArcType.CONTEXTUAL) {
                        removeContextualArc(s, p_graph);
                    }
                }

            }

            // try to remove contextual node itself
            final XdiNode node = getNode(p_graph, m_operand);
            if (node instanceof ContextNode) {
                final Set<XdiArc> contextualArcs = node.getContextualArcList();
                for (XdiArc arc : contextualArcs) {
                    if (arc.getTarget().equals(node)) {
                        remove(arc.getSource(), arc.getTarget(), XdiStatement.ArcType.CONTEXTUAL);
                    }
                }
                return new OperationResultImpl(true);
            }
        }
        return new OperationResultImpl(false);
    }

    /**
     * Removes contextual arc from graph
     *
     * @param p_statement statement to remove
     * @param p_graph     root node of graph
     */
    private void removeContextualArc(XdiStatement p_statement, RootNode p_graph) {
        final XdiNode sourceNode = getNode(p_graph, p_statement.getSubject());
        final XdiNode targetNode = getNode(p_graph, new XriImpl(p_statement.getSubject().asString(), p_statement.getObject().asString()));
        if (sourceNode != null && targetNode != null) {
            remove(sourceNode, targetNode, XdiStatement.ArcType.CONTEXTUAL);
        }
    }

    /**
     * Removes literal arc from graph
     *
     * @param p_statement statement to remove
     * @param p_graph     root node of graph
     */
    private void removeLiteralArc(XdiStatement p_statement, RootNode p_graph) {
        final XdiNode sourceNode = getNode(p_graph, p_statement.getSubject());
        final XdiNode targetNode = getNode(p_graph, new XriImpl(p_statement.getSubject().asString(), Symbols.SLASH.getValue(), p_statement.getPredicate().asString()));
        if (sourceNode != null && targetNode != null) {
            remove(sourceNode, targetNode, XdiStatement.ArcType.LITERAL);
        }
    }

    /**
     * Gets node by xri from graph.
     *
     * @param p_graph root node of graph
     * @param p_xri   xri of node
     * @return node by xri
     */
    private static XdiNode getNode(RootNode p_graph, Xri p_xri) {
        return GraphExtractorImpl.getInstance().getNodeByXri(p_graph, p_xri);
    }

    /**
     * Removes relational arc from graph.
     *
     * @param p_statement xdi statement to remove (must have relational arc type)
     * @param p_graph     root node of graph
     * @return whether relational arc was removed
     */
    public static boolean removeRelationalArc(XdiStatement p_statement, RootNode p_graph) {
        final Xri xriObject = p_statement.getObject();
        final XdiNode sourceNode = getNode(p_graph, p_statement.getSubject());

        XdiNode targetNode = getNode(p_graph, xriObject);

        if (sourceNode == null) {
            return false;
        }

        if (targetNode != null) {
            remove(sourceNode, targetNode, XdiStatement.ArcType.RELATIONAL);
            return true;
        }

        // try whether it's local context symbol
        targetNode = getNode(p_graph, new XriImpl(sourceNode.getXri().asString(), Symbols.SLASH.getValue(), xriObject.asString()));
        if (targetNode != null) {
            remove(sourceNode, targetNode, XdiStatement.ArcType.RELATIONAL);
            return true;
        }

        // try whether it's cross-reference
        final String xriObjectStr = xriObject.asString();
        if (Utils.isEnclosedInParenthesis(xriObjectStr)) {
            targetNode = getNode(p_graph, new XriImpl(Utils.cutParenthesis(xriObjectStr)));
            if (targetNode != null) {
                remove(sourceNode, targetNode, XdiStatement.ArcType.RELATIONAL);
                return true;
            }
        }
        return false;
    }

    /**
     * Removes arc between source and target nodes
     *
     * @param p_source source node
     * @param p_target target node
     * @param p_type   type of arc
     */
    private static void remove(XdiNode p_source, XdiNode p_target, XdiStatement.ArcType p_type) {
        switch (p_type) {
            case RELATIONAL:
                for (XdiArc arc : new ArrayList<XdiArc>(p_source.getRelationalArcList())) {
                    if (arc.getSource().equals(p_source) && arc.getTarget().equals(p_target)) {
                        p_source.getRelationalArcList().remove(arc);
                        p_target.getRelationalArcList().remove(arc);
                    }
                }
                return;
            case LITERAL:
                for (XdiArc arc : new ArrayList<XdiArc>(p_source.getLiteralArcList())) {
                    if (arc.getSource().equals(p_source) && arc.getTarget().equals(p_target)) {
                        p_source.getLiteralArcList().remove(arc);
                        p_target.getLiteralArcList().remove(arc);
                    }
                }
                return;
            case CONTEXTUAL:
                for (XdiArc arc : new ArrayList<XdiArc>(p_source.getContextualArcList())) {
                    if (arc.getSource().equals(p_source) && arc.getTarget().equals(p_target)) {
                        p_source.getContextualArcList().remove(arc);
                        p_target.getContextualArcList().remove(arc);
                    }
                }
                return;
            default:
        }
    }

    /**
     * Original xdi statement with operation (e.g. {from segment}$do/$add/({statement})).
     *
     * @return original xdi statement with operation
     */
    public XdiStatement getOriginalCommandStatement() {
        return m_commandStatement;
    }

    /**
     * Returns operand of operation.
     *
     * @return operand of operation
     */
    public Xri getOperand() {
        return m_operand;
    }

    /**
     * Gets type of operation.
     *
     * @return type of operation
     */
    public OperationType getType() {
        return OperationType.DEL;
    }

    /**
     * Gets link contract node xri.
     *
     * @return Gets link contract node xri
     */
    public Xri getLinkContractNode() {
        if (m_linkContractXri == null) {
            m_linkContractXri = Utils.normalize(m_operand);
        }
        return m_linkContractXri;
    }

    /**
     * Generates command statement object.
     *
     * @return command statement object
     */
    public String generateCommandStatementObject() {
        final String s = m_operand.asString();
        if (s.contains(Symbols.SLASH.getValue())) {
            final StringBuilder b = new StringBuilder();
            b.append("(").append(s).append(")");
            return b.toString();
        }
        return s;
    }
}
