package org.xdi.graphmodel.impl.operation;

import org.xdi.graphmodel.api.GraphBuilder;
import org.xdi.graphmodel.api.GraphExtractor;
import org.xdi.graphmodel.api.RootNode;
import org.xdi.graphmodel.api.Symbols;
import org.xdi.graphmodel.api.graph.XdiNode;
import org.xdi.graphmodel.api.graph.XdiStatement;
import org.xdi.graphmodel.api.operation.Operation;
import org.xdi.graphmodel.api.operation.OperationResult;
import org.xdi.graphmodel.api.operation.OperationType;
import org.xdi.graphmodel.api.xri.Xri;
import org.xdi.graphmodel.impl.AbstractXdiNode;
import org.xdi.graphmodel.impl.GraphBuilderImpl;
import org.xdi.graphmodel.impl.GraphExtractorImpl;
import org.xdi.graphmodel.impl.xri.XriImpl;

/**
 * Add operation implementation.
 *
 * @author Yuriy Zabrovarnyy
 * @version 0.9, 05/05/2011
 */
public class AddOperationImpl implements Operation<XdiStatement> {

    /**
     * Command statement
     */
    private final XdiStatement m_commandStatement;

    /**
     * Operand
     */
    private final XdiStatement m_operand;

    /**
     * Constructor
     *
     * @param p_commandStatement command statement
     * @param p_operand          operand
     */
    public AddOperationImpl(XdiStatement p_commandStatement, XdiStatement p_operand) {
        m_commandStatement = p_commandStatement;
        m_operand = p_operand;
    }

    /**
     * Apply operation to graph.
     *
     * @param p_graph root of graph
     * @return whether operation is applied successfully or not
     */
    public OperationResult apply(RootNode p_graph) {
        return new OperationResultImpl(applyInternally(p_graph));
    }

    /**
     * Internal implementation of apply method.
     *
     * @param p_graph root of graph
     * @return whether operation is applied successfully or not
     */
    private boolean applyInternally(RootNode p_graph) {
        if (p_graph != null) {
            final GraphExtractor extractor = GraphExtractorImpl.getInstance();
            final GraphBuilder builder = GraphBuilderImpl.getInstance();
            final XdiNode sourceNode = extractor.getNodeByXri(p_graph, m_operand.getSubject());

            if (sourceNode != null) {
                switch (m_operand.getType()) {
                    case CONTEXTUAL:
                        final Xri targetNodeXri = new XriImpl(sourceNode.getXri().asString(), m_operand.getObject().asString());
                        XdiNode targetNode = extractor.getNodeByXri(p_graph, targetNodeXri);

                        if (targetNode != null) {
                            return false; // arc already exist
                        } else {
                            targetNode = builder.createContextNode();
                            ((AbstractXdiNode) targetNode).setXri(targetNodeXri);
                        }
                        builder.attachContextualArc(sourceNode, targetNode, m_operand.getObject().asString());
                        return true;
                    case LITERAL:
                        final Xri targetLitNodeXri = new XriImpl(m_operand.getSubject().asString(), Symbols.SLASH.getValue(), m_operand.getPredicate().asString());
                        XdiNode targetLitNode = extractor.getNodeByXri(p_graph, targetLitNodeXri);
                        if (targetLitNode != null) {
                            return false; // arc already exist
                        } else {
                            final String value = GraphBuilderImpl.extractLiteralValueString(m_operand.getObject().asString());
                            targetLitNode = builder.createLiteralNode(value);
                            ((AbstractXdiNode) targetLitNode).setXri(targetLitNodeXri);
                        }
                        builder.attachLiteralArc(sourceNode, targetLitNode, m_operand.getPredicate().asString());
                        return true;
                    case RELATIONAL:
                        XdiNode targetRelNode = extractor.getNodeByXri(p_graph, new XriImpl(m_operand.getObject().asString()));
                        if (targetRelNode != null) {
                            builder.attachRelationalArc(sourceNode, targetRelNode, m_operand.getPredicate().asString());
                            return true;
                        }

                        // try whether it's local context symbol
                        targetRelNode = extractor.getNodeByXri(p_graph, new XriImpl(sourceNode.getXri().asString(), Symbols.SLASH.getValue(), m_operand.getObject().asString()));
                        if (targetRelNode != null) {
                            builder.attachRelationalArc(sourceNode, targetRelNode, m_operand.getPredicate().asString());
                            return true;
                        }

                        // try whether it's cross-reference
                        final String xriObjectStr = m_operand.getObject().asString();
                        if (xriObjectStr.startsWith(Symbols.OPENING_PARENTHESIS.getValue()) &&
                                xriObjectStr.endsWith(Symbols.CLOSING_PARENTHESIS.getValue())) {
                            targetRelNode = extractor.getNodeByXri(p_graph, new XriImpl(xriObjectStr.substring(1, xriObjectStr.length() - 1)));
                            if (targetRelNode != null) {
                                builder.attachRelationalArc(sourceNode, targetRelNode, m_operand.getPredicate().asString());
                                return true;
                            }
                        }

                        return false;
                }
            }
        }
        return false;
    }


    /**
     * Original xdi statement with operation (e.g. {from segment}$do/$add/({statement})).
     *
     * @return original xdi statement with operation
     */
    public XdiStatement getOriginalCommandStatement() {
        return m_commandStatement;
    }

    /**
     * Returns operand of operation.
     *
     * @return operand of operation
     */
    public XdiStatement getOperand() {
        return m_operand;
    }

    /**
     * Gets type of operation.
     *
     * @return type of operation
     */
    public OperationType getType() {
        return OperationType.ADD;
    }

    /**
     * Gets link contract node xri.
     *
     * @return Gets link contract node xri
     */
    public Xri getLinkContractNode() {
        return m_operand.getSubject();
    }

    /**
     * Generates command statement object.
     *
     * @return command statement object
     */
    public String generateCommandStatementObject() {
        final StringBuilder b = new StringBuilder();
        b.append("(").append(m_operand.asString()).append(")");
        return b.toString();
    }
}
