package org.xdi.graphmodel.impl;

import org.xdi.graphmodel.api.Symbols;
import org.xdi.graphmodel.api.Variable;
import org.xdi.graphmodel.api.graph.XdiStatement;
import org.xdi.graphmodel.api.xri.Xri;
import org.xdi.graphmodel.common.Utils;
import org.xdi.graphmodel.impl.xri.XriImpl;

import java.util.Set;

/**
 * Xdi statement implementation. Contains 3 basic elements of xdi statements: subject, predicate, object.
 *
 * @author Yuriy Zabrovarnyy
 * @version 0.9, 3/27/11
 */
public class XdiStatementImpl implements XdiStatement {

    /**
     * Delimiter for string representation of xdi statement ('/' e.g. =yuriy/+friend/=mike).
     */
    private static final String DELIMITER = Symbols.SLASH.getValue();

    /**
     * Null xdi statement
     */
    public static final XdiStatement NULL_XDI_STATEMENT = new XdiStatementImpl(new XriImpl(""), new XriImpl(""), new XriImpl());

    /**
     * Subject
     */
    private final Xri m_subject;

    /**
     * Predicate
     */
    private final Xri m_predicate;

    /**
     * Object
     */
    private final Xri m_object;

    /**
     * Constructor. (If subject or predicate or object is null IllegalArgumentException exception is thrown.)
     *
     * @param subject   subject of xdi statement in xri format
     * @param predicate predicate of xdi statement in xri format
     * @param object    object of xdi statement in xri format
     */
    public XdiStatementImpl(Xri subject, Xri predicate, Xri object) {
        if (subject == null || predicate == null || object == null) {
            throw new IllegalArgumentException();
        }
        m_subject = subject;
        m_predicate = predicate;
        m_object = object;
    }

    /**
     * @return subject of xdi statement
     */
    public Xri getSubject() {
        return m_subject;
    }

    /**
     * @return predicate of xdi statement
     */
    public Xri getPredicate() {
        return m_predicate;
    }

    /**
     * @return object of xdi statement
     */
    public Xri getObject() {
        return m_object;
    }

    /**
     * Returns type of xdi statement
     *
     * @return type of xdi statement
     */
    public ArcType getType() {
        final String predicateStr = getPredicate().asString();
        final String objectStr = getObject().asString();
        if (predicateStr.equals(Symbols.PARENTHESIS.getValue())) {
            return ArcType.CONTEXTUAL;
        } else if (objectStr.startsWith(GraphBuilderImpl.LITERAL_OBJECT_PREFIX) && objectStr.endsWith(Symbols.CLOSING_PARENTHESIS.getValue())) {
            return ArcType.LITERAL;
        } else {
            return ArcType.RELATIONAL;
        }
    }

    /**
     * Gets variables.
     *
     * @return variables
     */
    @Override
    public Set<Variable> getVariables() {
        return Utils.getVariables(asString());
    }

    /**
     * Returns whether statement is "add variable". E.g. =!schwartz/()/{$1}
     *
     * @return whether statement is "add variable"
     */
    @Override
    public boolean isAddVariableStatement() {
        final Set<Variable> variables = getVariables();
        if (variables != null && variables.size() == 1) {
            final Variable v = variables.iterator().next();
            if (v.asString().equals(m_object.asString())) {
                return true;
            }
        }
        return false;
    }

    /**
     * Constructs xdi statement object from string representation of subject, predicate and object.
     *
     * @param p_subject   subject of xdi statement as string
     * @param p_predicate predicate of xdi statement as string
     * @param p_object    object of xdi statement as string
     * @return xdi statement object
     */
    public static XdiStatement valueOf(String p_subject, String p_predicate, String p_object) {
        return new XdiStatementImpl(new XriImpl(p_subject), new XriImpl(p_predicate), new XriImpl(p_object));
    }

    /**
     * Constructs xdi statement object from string representation.
     *
     * @param p_xdiStatementString xdi statement string (e.g. =mike/+friend/=yuriy)
     * @return xdi statement object
     */
    public static XdiStatement valueOf(String p_xdiStatementString) {
        if (p_xdiStatementString != null) {
            final int firstIndex = firstIndex(p_xdiStatementString);
            final int lastIndex = lastIndex(p_xdiStatementString);
            if (firstIndex == -1 || lastIndex == -1) {
                throw new IllegalArgumentException();
            }

            final String subject = p_xdiStatementString.substring(0, firstIndex);
            final String predicate = p_xdiStatementString.substring(firstIndex + 1, lastIndex);
            final String object = p_xdiStatementString.substring(lastIndex + 1);
            return valueOf(subject, predicate, object);
        }
        return null;
    }

    /**
     * Parses xdi statement string representation silently (no exceptions are thrown).
     *
     * @param p_str string representation of xdi statement
     * @return xdi statement object
     */
    public static XdiStatement parseSilently(String p_str) {
        try {
            return XdiStatementImpl.valueOf(p_str);
        } catch (Exception e) {
            return null;
        }
    }

    /**
     * Returns whether string is valid representation of xdi statement.
     *
     * @param p_xdiStatementString xdi statement string (e.g. =mike/+friend/=yuriy)
     * @return whether string is valid representation of xdi statement
     */
    public static boolean canParse(String p_xdiStatementString) {
        try {
            return valueOf(p_xdiStatementString) != null;
        } catch (Exception e) {
            return false;
        }
    }

    /**
     * Calculates first delimiter in xdi statement
     *
     * @param p_str xdi statement
     * @return index of first occurrence of slash delimiter
     */
    private static int firstIndex(String p_str) {
        int n = 0;
        int result = 0;
        while (true) {
            int delimiterIndex = p_str.indexOf(DELIMITER);
            final String s = p_str.substring(0, delimiterIndex);
            //System.out.println("first s=" + s);

            // make sure we return from this cycle
            n++;
            if (n == 100) {
                return -1;
            }

            final int openingParenthesis = s.indexOf(Symbols.OPENING_PARENTHESIS.getValue());
            if (openingParenthesis != -1) {
                final int closingParenthesis = p_str.indexOf(Symbols.CLOSING_PARENTHESIS.getValue());
                result += closingParenthesis + 1;
                p_str = p_str.substring(closingParenthesis + 1);
                //System.out.println("first str=" + p_str);
            } else {
                return result + delimiterIndex;
            }
        }
    }

    /**
     * Calculates last delimiter in xdi statement
     *
     * @param p_str xdi statement
     * @return index of last occurrence of slash delimiter
     */
    private static int lastIndex(String p_str) {
        int N = 0;
        while (true) {
            final int delimiter = p_str.lastIndexOf(Symbols.SLASH.getValue());
            final int closingParenth = p_str.substring(delimiter).lastIndexOf(Symbols.CLOSING_PARENTHESIS.getValue());

            if (closingParenth == -1) {
                return delimiter;
            }

            int closingCount = 0;
            int openingCount = 0;
            for (int i = p_str.length() - 1; i >= 0; i--) {
                if (p_str.charAt(i) == ')') {
                    closingCount++;
                }

                if (p_str.charAt(i) == '(') {
                    openingCount++;
                }
                if (closingCount == openingCount && closingCount != 0) {
                    p_str = p_str.substring(0, i);
                    break;
                }
            }

            // make sure we return from this cycle
            N++;
            //System.out.println(N);
            if (N == 100) {
                return -1;
            }
        }
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;

        final XdiStatementImpl that = (XdiStatementImpl) o;

        return !(m_object != null ? !m_object.equals(that.m_object) :
                that.m_object != null) && !(m_predicate != null ?
                !m_predicate.equals(that.m_predicate) : that.m_predicate != null) && !(m_subject != null ?
                !m_subject.equals(that.m_subject) : that.m_subject != null);
    }

    @Override
    public int hashCode() {
        int result = m_subject != null ? m_subject.hashCode() : 0;
        result = 31 * result + (m_predicate != null ? m_predicate.hashCode() : 0);
        result = 31 * result + (m_object != null ? m_object.hashCode() : 0);
        return result;
    }

    /**
     * @return string representation of xdi statement
     */
    public String asString() {
        final StringBuilder s = new StringBuilder();
        s.append(m_subject.asString()).append(DELIMITER);
        s.append(m_predicate.asString()).append(DELIMITER);
        s.append(m_object.asString());
        return s.toString();
    }

    @Override
    public String toString() {
        return "XdiStatementImpl{" +
                "m_subject=" + m_subject +
                ", m_predicate=" + m_predicate +
                ", m_object=" + m_object +
                '}';
    }
}
