package org.xdi.graphmodel.impl;

import org.xdi.graphmodel.api.GraphTraversFunction;
import org.xdi.graphmodel.api.GraphTraverser;
import org.xdi.graphmodel.api.LiteralNode;
import org.xdi.graphmodel.api.graph.XdiArc;
import org.xdi.graphmodel.api.graph.XdiNode;

import java.util.Set;

/**
 * Implementation of graph traverser.
 *
 * @author Yuriy Zabrovarnyy
 * @version 0.9, 09/05/2011
 */
public class GraphTraverserImpl implements GraphTraverser {

    /**
     * Graph traverser instance
     */
    private static final GraphTraverser INSTANCE = new GraphTraverserImpl();

    /**
     * Private constructor
     */
    private GraphTraverserImpl() {
    }

    /**
     * Returns instance of graph traverser.
     *
     * @return instance of graph traverser
     */
    public static GraphTraverser getInstance() {
        return INSTANCE;
    }

    /**
     * Performs graph traversing with function.
     *
     * @param p_node     start xdi node
     * @param p_function traversing function
     */
    public void traverse(XdiNode p_node, GraphTraversFunction p_function) {
        traverseInternal(p_node, p_function);
    }

    /**
     * Internal implementation of graph traversing.
     *
     * @param p_node     start xdi node
     * @param p_function traversing function
     */
    private void traverseInternal(XdiNode p_node, GraphTraversFunction p_function) {
        if (p_node != null) {
            if (p_function.getTerminateCondition(p_node)) {
                return;
            }
            p_function.apply(p_node);

            final Set<XdiArc> contextualArcList = p_node.getContextualArcList();
            for (XdiArc arc : contextualArcList) {
                final XdiNode target = arc.getTarget();
                if (!target.equals(p_node)) {
                    traverseInternal(target, p_function);
                }
            }

            final Set<XdiArc> literalArcList = p_node.getLiteralArcList();
            for (XdiArc arc : literalArcList) {
                final XdiNode target = arc.getTarget();
                if (target instanceof LiteralNode && arc.getSource().equals(p_node)) {
                    if (p_function.getTerminateCondition(target)) {
                        return;
                    }
                    p_function.apply(target);
                }
            }
        }
    }
}
