package org.xdi.graphmodel.impl;

import org.xdi.graphmodel.api.*;
import org.xdi.graphmodel.api.graph.XdiArc;
import org.xdi.graphmodel.api.graph.XdiNode;
import org.xdi.graphmodel.api.graph.XdiStatement;
import org.xdi.graphmodel.api.xri.Xri;
import org.xdi.graphmodel.impl.xri.XriImpl;

import java.util.*;

/**
 * Graph extractor implementation. Extract different type of information from graph model.
 *
 * @author Yuriy Zabrovarnyy
 * @version 0.9, 3/26/11
 */
public class GraphExtractorImpl implements GraphExtractor {

    /**
     * Static reference on instance of graph extractor (avoid new instance creation)
     */
    private static final GraphExtractor EXTRACTOR = new GraphExtractorImpl();

    /**
     * Private constructor
     */
    private GraphExtractorImpl() {
    }

    /**
     * Returns instance of graph extractor.
     *
     * @return instance of graph extractor
     */
    public static GraphExtractor getInstance() {
        return EXTRACTOR;
    }

    /**
     * Extracts node's xri list from graph model.
     *
     * @param p_node root node of graph model
     * @return xri set that contains xri of each node within the graph
     */
    public Set<Xri> extractXriList(RootNode p_node) {
        final Set<Xri> result = new HashSet<Xri>();
        extractXriList(p_node, result);
        return result;
    }

    /**
     * Extracts xdi statement list from graph model.
     *
     * @param p_xdiNode xdi node of graph model
     * @return xdi statement list from graph
     */
    public List<XdiStatement> extractXdiStatementList(XdiNode p_xdiNode) {
        final List<XdiStatement> result = new ArrayList<XdiStatement>();
        extractXdiStatementList(p_xdiNode, result);
        return result;
    }

    /**
     * For xdi statement extraction only outgoing arcs should be considered.
     *
     * @param p_node node of graph
     * @param p_list list of xdi statements
     */
    private static void extractXdiStatementList(XdiNode p_node, List<XdiStatement> p_list) {
        final Set<XdiArc> contextualArcList = p_node.getContextualArcList();
        for (XdiArc arc : contextualArcList) {
            final XdiNode source = arc.getSource();
            final XdiNode target = arc.getTarget();
            if (source.equals(p_node)) {
                if (!arc.isFake()) {
                    p_list.add(new XdiStatementImpl(source.getXri(), Symbols.PARENTHESIS.getXri(), arc.getXri()));
                }
                if (target instanceof ContextNode) {
                    extractXdiStatementList(target, p_list);
                }
            }
        }

        final Set<XdiArc> relationalArcList = p_node.getRelationalArcList();
        for (XdiArc arc : relationalArcList) {
            final XdiNode source = arc.getSource();
            if (source.equals(p_node)) {
                final XdiStatement xdiStatement = constructRelationalXdiStatement(arc);
                if (xdiStatement != null) {
                    p_list.add(xdiStatement);
                }
            }
        }

        final Set<XdiArc> literalArcList = p_node.getLiteralArcList();
        for (XdiArc arc : literalArcList) {
            final XdiNode source = arc.getSource();
            if (source.equals(p_node) && arc.getTarget() instanceof LiteralNode && !arc.isFake()) {
                final XdiStatement xdiStatement = constructLiteralStatementFromLiteralArc(arc);
                if (xdiStatement != null) {
                    p_list.add(xdiStatement);
                }
            }
        }
    }

    /**
     * Constructs xdi statement that represent relational arc.
     *
     * @param p_arc arc
     * @return xdi statement
     */
    public static XdiStatement constructRelationalXdiStatement(XdiArc p_arc) {
        if (p_arc != null) {
            final XdiNode target = p_arc.getTarget();
            final XdiNode source = p_arc.getSource();
            if (target instanceof ContextNode) {
                return new XdiStatementImpl(source.getXri(), p_arc.getXri(), target.getXri());
            } else if (target instanceof LiteralNode) {
                final Iterator<XdiArc> iterator = target.getLiteralArcList().iterator();

                // local context symbol
                if (iterator.hasNext()) {
                    final XdiArc literalArc = iterator.next();
                    if (literalArc != null && literalArc.getSource().equals(source)) {
                        return new XdiStatementImpl(source.getXri(), p_arc.getXri(), literalArc.getXri());
                    } else {
                        return constructCrossReference(p_arc);
                    }
                } else {
                    return constructCrossReference(p_arc);
                }
            }
        }
        return null;
    }

    /**
     * Constructs literal xdi statement from literal arc.
     *
     * @param p_arc literal arc
     * @return xdi statement
     */
    public static XdiStatement constructLiteralStatementFromLiteralArc(XdiArc p_arc) {
        if (p_arc != null && p_arc.getTarget() instanceof LiteralNode && p_arc.getSource() instanceof ContextNode) {
            return new XdiStatementImpl(p_arc.getSource().getXri(), p_arc.getXri(), constractLiteralNodePart((LiteralNode) p_arc.getTarget()));
        }
        return null;
    }

    /**
     * Constructs xdi statement that represent cross-reference relational arc.
     *
     * @param p_arc arc
     * @return xdi statement
     */
    public static XdiStatement constructCrossReference(XdiArc p_arc) {
        // cross-reference
        final Xri objectXri = new XriImpl(
                Symbols.OPENING_PARENTHESIS.getValue(),
                p_arc.getTarget().getXri().asString(),
                Symbols.CLOSING_PARENTHESIS.getValue());
        return new XdiStatementImpl(p_arc.getSource().getXri(), p_arc.getXri(), objectXri);
    }

    /**
     * Constructs literal node part of xdi statement.
     *
     * @param p_node literal node of graph
     * @return xri constructed based on literal node
     */
    private static Xri constractLiteralNodePart(LiteralNode p_node) {
        return new XriImpl(constractLiteralValuePart(p_node.getValue().asString()));
    }

    /**
     * Constructs literal node part of xdi statement.
     *
     * @param p_literalValue literal node value
     * @return xri string constructed based on literal node value
     */
    public static String constractLiteralValuePart(String p_literalValue) {
        final StringBuilder s = new StringBuilder();
        s.append(Symbols.OPENING_PARENTHESIS.getValue()).
                append(Symbols.LITERAL_DATA.getValue()).
                append(Symbols.COLON.getValue()).
                append(Symbols.COMMA.getValue()).
                append(p_literalValue).
                append(Symbols.CLOSING_PARENTHESIS.getValue());
        return s.toString();
    }

    /**
     * Extract xri list from graph (recursively).
     *
     * @param p_node node of graph
     * @param p_list list of xri (container that collects xri's)
     */
    private static void extractXriList(XdiNode p_node, Set<Xri> p_list) {
        final Set<XdiArc> contextualArcList = p_node.getContextualArcList();
        for (XdiArc arc : contextualArcList) {
            final XdiNode target = arc.getTarget();
            if (!target.equals(p_node)) {
                p_list.add(target.getXri());
                extractXriList(target, p_list);
            }
        }
        final Set<XdiArc> literalArcList = p_node.getLiteralArcList();
        for (XdiArc arc : literalArcList) {
            final XdiNode target = arc.getTarget();
            if (target instanceof LiteralNode && arc.getSource().equals(p_node)) {
                p_list.add(target.getXri());
            }
        }
    }

    /**
     * Returns xdi node by xri from graph (or sub graph).
     *
     * @param p_xdiNode start traversing from this xdi node
     * @param p_nodeXri xri of node to find
     * @return xdi node
     */
    public XdiNode getNodeByXri(XdiNode p_xdiNode, final Xri p_nodeXri) {
        final GraphTraversFunction traversingFunction = new GraphTraversFunction() {

            private XdiNode m_terminateNode = null;

            public void apply(XdiNode p_node) {
                // nothing
            }

            public boolean getTerminateCondition(XdiNode p_node) {
                if (m_terminateNode != null) {
                    return true;
                } else {
                    if (p_node.getXri().equals(p_nodeXri)) {
                        m_terminateNode = p_node;
                        return true;
                    }
                    return false;
                }
            }

            public XdiNode getTerminateNode() {
                return m_terminateNode;
            }
        };
        GraphTraverserImpl.getInstance().traverse(p_xdiNode, traversingFunction);
        return traversingFunction.getTerminateNode();
    }
}
