package org.xdi.graphmodel.impl;

import org.xdi.graphmodel.api.ContextNode;
import org.xdi.graphmodel.api.GraphBuilder;
import org.xdi.graphmodel.api.LiteralNode;
import org.xdi.graphmodel.api.RootNode;
import org.xdi.graphmodel.api.Symbols;
import org.xdi.graphmodel.api.graph.XdiArc;
import org.xdi.graphmodel.api.graph.XdiNode;
import org.xdi.graphmodel.api.graph.XdiStatement;
import org.xdi.graphmodel.api.xri.Xri;
import org.xdi.graphmodel.common.Utils;
import org.xdi.graphmodel.impl.xri.XriImpl;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;

/**
 * Graph builder implementation.
 *
 * @author Yuriy Zabrovarnyy
 * @version 0.9, 25/03/2011
 */
public class GraphBuilderImpl implements GraphBuilder {

    /**
     * Static builder which is used internally.
     */
    private static final GraphBuilder BUILDER = new GraphBuilderImpl();

    /**
     * Literal object prefix in xdi statement object (e.g. '(data:').
     */
    public static final String LITERAL_OBJECT_PREFIX = Symbols.OPENING_PARENTHESIS.getValue() +
            Symbols.LITERAL_DATA.getValue() + Symbols.COLON.getValue();

    /**
     * Fake arc counter
     */
    private static int g_fakeArcCounter = 0;

    /**
     * Private constructor
     */
    private GraphBuilderImpl() {
    }

    /**
     * Returns instance of graph builder.
     *
     * @return instance of graph builder
     */
    public static GraphBuilder getInstance() {
        return BUILDER;
    }

    /**
     * Creates root node. (Should be start point during programmatic graph creation.)
     *
     * @return root node
     */
    public RootNode createRootNode() {
        return new RootNodeImpl();
    }

    /**
     * @return context node (without arcs)
     */
    public ContextNode createContextNode() {
        return new ContextNodeImpl();
    }

    /**
     * Creates literal node with value.
     *
     * @param value value of literal node
     * @return literal node with value
     */
    public LiteralNode createLiteralNode(LiteralNode.Value value) {
        return new LiteralNodeImpl(value);
    }

    /**
     * Creates literal node with value.
     *
     * @param p_value value of literal node
     * @return literal node with value
     */
    public LiteralNode createLiteralNode(String p_value) {
        return createLiteralNode(new LiteralNodeImpl.ValueImpl(p_value));
    }

    /**
     * Creates arc with source, target node and xri.
     *
     * @param p_source  source node of arc
     * @param p_target  target node of arc
     * @param p_xriPart xri of arc
     * @return xri with specified source, target node and xri
     */
    private static XdiArc createArc(XdiNode p_source, XdiNode p_target, String p_xriPart) {
        return new XdiArcImpl(p_source, p_target, new XriImpl(p_xriPart));
    }

    /**
     * Attach contextual arc to source and target nodes. Both nodes have reference to this arc.
     *
     * @param p_source  source node of arc
     * @param p_target  target node of arc
     * @param p_xriPart xri of arc
     * @return arc with xri and source/target specified
     */
    public XdiArc attachContextualArc(XdiNode p_source, XdiNode p_target, String p_xriPart) {
        final XdiArc arc = createArc(p_source, p_target, p_xriPart);
        p_source.getContextualArcList().add(arc);
        p_target.getContextualArcList().add(arc);
        return arc;
    }

    /**
     * Attach literal arc to source and target nodes. Both nodes have reference to this arc.
     *
     * @param p_source  source node of arc
     * @param p_target  target node of arc
     * @param p_xriPart xri of arc
     * @return arc with xri and source/target specified
     */
    public XdiArc attachLiteralArc(XdiNode p_source, XdiNode p_target, String p_xriPart) {
        final XdiArc arc = createArc(p_source, p_target, p_xriPart);
        p_source.getLiteralArcList().add(arc);
        p_target.getLiteralArcList().add(arc);
        return arc;
    }

    /**
     * Attach relational arc to source and target nodes. Both nodes have reference to this arc.
     *
     * @param p_source  source node of arc
     * @param p_target  target node of arc
     * @param p_xriPart xri of arc
     * @return arc with xri and source/target specified
     */
    public XdiArc attachRelationalArc(XdiNode p_source, XdiNode p_target, String p_xriPart) {
        final XdiArc arc = createArc(p_source, p_target, p_xriPart);
        p_source.getRelationalArcList().add(arc);
        p_target.getRelationalArcList().add(arc);
        return arc;
    }

    /**
     * Creates graph model based on xdi statement list.
     *
     * @param p_list xdi statement list
     * @return graph created based on list of xdi statements
     */
    public RootNode createGraph(List<XdiStatement> p_list) {
        return createGraphInternally(p_list, false);
    }

    /**
     * Creates graph from xdi statement list.
     *
     * @param p_list            list
     * @param p_createFakeNodes create fake nodes
     * @return root node of graph
     */
    private RootNode createGraphInternally(List<XdiStatement> p_list, boolean p_createFakeNodes) {
        final List<XdiStatement> copy = new ArrayList<XdiStatement>(p_list);
        final RootNode rootNode = createRootNode();
        ((AbstractXdiNode) rootNode).setXri(Symbols.PARENTHESIS.getXri());

        final List<XdiNode> nodeList = new ArrayList<XdiNode>(Arrays.asList(rootNode));
        if (!copy.isEmpty()) {
            final List<XdiStatement> handledSuccessfully = new ArrayList<XdiStatement>();
            for (XdiStatement statement : copy) {
                final boolean injected = injectContextAndLiteralStatements(statement, nodeList);
                if (injected) {
                    handledSuccessfully.add(statement);
                }
            }
            copy.removeAll(handledSuccessfully);
            for (XdiStatement statement : copy) {
                injectRelationalStatement(statement, nodeList, p_createFakeNodes);
            }
        }
        bind(nodeList, rootNode);
        return rootNode;
    }

    /**
     * Creates graph model based on xdi statement list. If there is nodes to which relational xdi statements points
     * but such nodes does not exist than they are created automatically and marked with "fake" indicator.
     *
     * @param p_list xdi statement list
     * @return graph created based on list of xdi statements
     */
    public RootNode createGraphWithFakeNodes(List<XdiStatement> p_list) {
        return createGraphInternally(p_list, true);
    }

    /**
     * Bind graph together and sets fake indicator.
     *
     * @param p_nodeList node list
     * @param p_rootNode root node
     */
    private void bind(List<XdiNode> p_nodeList, RootNode p_rootNode) {
        for (XdiNode n : p_nodeList) {
            if (n instanceof RootNode) {
                continue;
            }
            if (n.getParent() == null) {
                final XdiArc arc = BUILDER.attachContextualArc(p_rootNode, n, Symbols.PARENTHESIS.getValue());
                arc.setFake(true);
            }
        }
    }

    /**
     * Inject relational xdi statement into graph model (model here is inside list of nodes
     * where one of this node is root node).
     *
     * @param p_statement       xdi statement that express relational arc
     * @param p_nodeList        list of created nodes
     * @param p_createFakeNodes create fake nodes
     */
    private static void injectRelationalStatement(XdiStatement p_statement, List<XdiNode> p_nodeList, boolean p_createFakeNodes) {
        final Xri xriObject = p_statement.getObject();
        final XdiNode sourceNode = getNodeByXri(p_nodeList, p_statement.getSubject());
        XdiNode targetNode = getNodeByXri(p_nodeList, xriObject);

        if (sourceNode == null) {
            System.out.println("ERROR: enable to inject statement, unable to identify sourceNode, statement: " + p_statement);
            return;
        }

        if (targetNode != null) {
            BUILDER.attachRelationalArc(sourceNode, targetNode, p_statement.getPredicate().asString());
            return;
        }

        // try whether it's local context symbol
        targetNode = getNodeByXri(p_nodeList, new XriImpl(sourceNode.getXri().asString(), Symbols.SLASH.getValue(), xriObject.asString()));
        if (targetNode != null) {
            BUILDER.attachRelationalArc(sourceNode, targetNode, p_statement.getPredicate().asString());
            return;
        }

        // try whether it's cross-reference
        final String xriObjectStr = xriObject.asString();
        if (Utils.isEnclosedInParenthesis(xriObjectStr)) {
            targetNode = getNodeByXri(p_nodeList, new XriImpl(Utils.cutParenthesis(xriObjectStr)));
            if (targetNode != null) {
                BUILDER.attachRelationalArc(sourceNode, targetNode, p_statement.getPredicate().asString());
                return;
            }
        }

        if (p_createFakeNodes) {

            // create fake nodes
            final XdiNode rootNode = getNodeByXri(p_nodeList, Symbols.PARENTHESIS.getXri());

            final String fakeTargetNodeXriStr = Utils.isEnclosedInParenthesis(xriObjectStr) ? Utils.cutParenthesis(xriObjectStr) : xriObjectStr;
            final Xri fakeTargetNodeXri = new XriImpl(fakeTargetNodeXriStr);
            if (fakeTargetNodeXriStr.contains(Symbols.SLASH.getValue())) {
                final XdiNode fakeTargetNode = injectLiteralNode(p_nodeList, fakeTargetNodeXri, new LiteralNodeImpl.ValueImpl(""));
                fakeTargetNode.setFake(true);
                final XdiArc arc = BUILDER.attachLiteralArc(rootNode, fakeTargetNode, "+fake" + g_fakeArcCounter + "!");
                arc.setFake(true);
                g_fakeArcCounter++;
            } else {
                final XdiNode fakeTargetNode = injectContextNode(p_nodeList, fakeTargetNodeXri);
                fakeTargetNode.setFake(true);
                final XdiArc arc = BUILDER.attachContextualArc(rootNode, fakeTargetNode, Symbols.PARENTHESIS.getValue());
                arc.setFake(true);
            }
            targetNode = getNodeByXri(p_nodeList, fakeTargetNodeXri);
            if (targetNode != null) {
                BUILDER.attachRelationalArc(sourceNode, targetNode, p_statement.getPredicate().asString());
                return;
            }
        }

        System.out.println("ERROR: enable to inject statement: " + p_statement + ", createFakes: " + p_createFakeNodes);
    }

    /**
     * Inject contextual or literal xdi statement into graph model (model here is inside list of nodes
     * where one of this node is root node).
     *
     * @param p_statement xdi statement that represents contextual or literal arc
     * @param p_nodeList  list of created nodes
     * @return whether xdi statement injected successfully
     */
    private static boolean injectContextAndLiteralStatements(XdiStatement p_statement, List<XdiNode> p_nodeList) {
        final String predicateStr = p_statement.getPredicate().asString();
        final String objectStr = p_statement.getObject().asString();

        switch (p_statement.getType()) {
            case CONTEXTUAL:
                final Xri sourceNodeXri = p_statement.getSubject();
                final Xri targetNodeXri = sourceNodeXri.asString().equals(Symbols.PARENTHESIS.getValue()) ?
                        new XriImpl(objectStr) :
                        new XriImpl(sourceNodeXri.asString(), objectStr);

                final XdiNode sourceNode = injectContextNode(p_nodeList, sourceNodeXri);
                final XdiNode targetNode = injectContextNode(p_nodeList, targetNodeXri);
                BUILDER.attachContextualArc(sourceNode, targetNode, objectStr);
                return true;
            case LITERAL:
                final Xri sourceNodeXriLit = p_statement.getSubject();
                final Xri targetNodeXriLit = new XriImpl(p_statement.getSubject().asString(), Symbols.SLASH.getValue(), predicateStr);

                final LiteralNode.Value literalValue = new LiteralNodeImpl.ValueImpl(extractLiteralValueString(objectStr));
                final XdiNode sourceNodeLit = injectContextNode(p_nodeList, sourceNodeXriLit);
                final XdiNode targetNodeLit = injectLiteralNode(p_nodeList, targetNodeXriLit, literalValue);
                BUILDER.attachLiteralArc(sourceNodeLit, targetNodeLit, p_statement.getPredicate().asString());
                return true;
            default:
                return false;
        }
    }

    /**
     * Extract literal value from data uri schema.
     * <p/>
     * Notation:
     * (data:[<MIME-type>][;charset=<encoding>][;base64],<data>)
     *
     * @param p_str string (e.g. (data: <type>, <literal value>))
     * @return literal value as string (e.g. <literal value>)
     */
    public static String extractLiteralValueString(String p_str) {
        // objectStr = '(data: <type>, <literal value>)'; literalStr = '<type>, <literal value>'
        String literalStr = p_str.substring(LITERAL_OBJECT_PREFIX.length(), p_str.length() - 1).trim();
        final int commaIndex = literalStr.indexOf(Symbols.COMMA.getValue());
        if (commaIndex != -1) {
            // literalStr = '<literal value>'
            literalStr = literalStr.substring(commaIndex + 1).trim();
        }
        return literalStr;
    }

    /**
     * Inject context node into graph model.
     *
     * @param p_nodeList list of created nodes
     * @param p_xri      xri of context node
     * @return context node that is already in list or create new one and put in list
     */
    private static XdiNode injectContextNode(List<XdiNode> p_nodeList, Xri p_xri) {
        XdiNode node = getNodeByXri(p_nodeList, p_xri);
        if (node == null) {
            node = BUILDER.createContextNode();
            ((AbstractXdiNode) node).setXri(p_xri);
            p_nodeList.add(node);
        }
        return node;
    }

    /**
     * Inject literal node into graph.
     *
     * @param p_nodeList list of created nodes
     * @param p_xri      xri of context node
     * @param p_value    value of literal node
     * @return context node that is already in list or create new one and put in list
     */
    private static XdiNode injectLiteralNode(List<XdiNode> p_nodeList, Xri p_xri, LiteralNode.Value p_value) {
        XdiNode node = getNodeByXri(p_nodeList, p_xri);
        if (node == null) {
            node = BUILDER.createLiteralNode(p_value);
            ((AbstractXdiNode) node).setXri(p_xri);
            p_nodeList.add(node);
        }
        return node;
    }

    /**
     * Gets node from list by xri.
     *
     * @param p_list list of nodes where search is performed
     * @param p_xri  xri of node
     * @return node by xri
     */
    private static XdiNode getNodeByXri(Collection<? extends XdiNode> p_list, Xri p_xri) {
        if (p_list != null && !p_list.isEmpty()) {
            for (XdiNode n : p_list) {
                final AbstractXdiNode node = (AbstractXdiNode) n;
                if (p_xri.equals(node.getXriWithoutEvaluation())) {
                    return n;
                }
            }
        }
        return null;
    }
}
