package org.xdi.graphmodel.impl;

import org.xdi.graphmodel.api.graph.XdiArc;
import org.xdi.graphmodel.api.graph.XdiNode;
import org.xdi.graphmodel.api.xri.Xri;
import org.xdi.graphmodel.impl.xri.XriEvaluator;

import java.util.HashSet;
import java.util.Set;

/**
 * Base implementation of xdi node.
 *
 * @author Yuriy Zabrovarnyy
 * @version 0.9, 25/03/2011
 * @see XdiNode
 */
public abstract class AbstractXdiNode implements XdiNode {

    private final Set<XdiArc> m_contextualArcList;
    private final Set<XdiArc> m_literalArcList;
    private final Set<XdiArc> m_relationalArcList;

    private Xri m_xri;
    private boolean m_fake = false;

    /**
     * Constructor without parameters.
     */
    public AbstractXdiNode() {
        m_contextualArcList = new HashSet<XdiArc>();
        m_literalArcList = new HashSet<XdiArc>();
        m_relationalArcList = new HashSet<XdiArc>();
    }

    /**
     * @return set of contextual arcs
     */
    public Set<XdiArc> getContextualArcList() {
        return m_contextualArcList;
    }

    /**
     * @return set of literal arcs
     */
    public Set<XdiArc> getLiteralArcList() {
        return m_literalArcList;
    }

    /**
     * @return set of relational arcs
     */
    public Set<XdiArc> getRelationalArcList() {
        return m_relationalArcList;
    }

    /**
     * Returns xri of node. In cases it's not evaluated first evaluated and then returned.
     *
     * @return xri of node.
     */
    public Xri getXri() {
        if (m_xri == null) {
            m_xri = XriEvaluator.evaluate(this);
        }
        return m_xri;
    }

    /**
     * Returns xri of node. In cases it's not evaluated return null.
     *
     * @return xri of node.
     */
    public Xri getXriWithoutEvaluation() {
        return m_xri;
    }

    /**
     * Sets xri of node. Convenient methods in cases xri is known then by this set evaluation step can be skipped.
     *
     * @param p_xri xri of node.
     */
    public void setXri(Xri p_xri) {
        m_xri = p_xri;
    }

    /**
     * Gets parent node.
     *
     * @param p_arcList arc list
     * @return parent node
     */
    public XdiNode getParent(Set<XdiArc> p_arcList) {
        if (p_arcList != null && !p_arcList.isEmpty()) {
            for (XdiArc arc : p_arcList) {
                if (arc.getTarget().equals(this)) {
                    return arc.getSource();
                }
            }
        }
        return null;
    }

    /**
     * Returns whether arc is fake.
     *
     * @return whether arc is fake
     */
    @Override
    public boolean isFake() {
        return m_fake;
    }

    /**
     * Sets fake indicator.
     *
     * @param p_fake fake
     */
    @Override
    public void setFake(boolean p_fake) {
        m_fake = p_fake;
    }

    @Override
    public String toString() {
        return "AbstractXdiNode{" +
                ", m_xri: " + m_xri +
                ", m_fake: " + m_fake +
                '}';
    }
}
