package org.xdi.graphmodel.common;

import org.xdi.graphmodel.api.AsString;
import org.xdi.graphmodel.api.Symbols;
import org.xdi.graphmodel.api.Variable;
import org.xdi.graphmodel.api.graph.XdiArc;
import org.xdi.graphmodel.api.graph.XdiNode;
import org.xdi.graphmodel.api.graph.XdiStatement;
import org.xdi.graphmodel.api.xri.Xri;
import org.xdi.graphmodel.impl.VariableImpl;
import org.xdi.graphmodel.impl.XdiStatementImpl;
import org.xdi.graphmodel.impl.xri.XriImpl;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

/**
 * Utility class. Contains different utility methodss.
 *
 * @author Yuriy Z
 * @version 0.9, 3/27/11
 */
public class Utils {

    /**
     * Message parameter key
     */
    public static final String MESSAGE_PARAMETER_KEY = "message";

    /**
     * Private constructor to forbid new instance creation.
     */
    private Utils() {
    }

    /**
     * Returns whether string is empty. (can't use StringUtils directly because need conversion to JS as
     * required by GWT)
     *
     * @param str string
     * @return whether string is empty
     */
    public static boolean isEmpty(String str) {
        return str == null || str.length() == 0;
    }

    /**
     * Prints to standard output list of items. Each item is located in new line.
     *
     * @param p_list list of AsStringObjects
     * @see AsString
     */
    public static void printList(Collection<? extends AsString> p_list) {
        for (AsString s : p_list) {
            System.out.println(s.asString());
        }
    }

    /**
     * Returns whether string is enclosed in parenthesis.
     *
     * @param p_str string value to check
     * @return whether string is enclosed in parenthesis
     */
    public static boolean isEnclosedInParenthesis(String p_str) {
        return p_str != null && p_str.startsWith("(") && p_str.endsWith(")");
    }

    /**
     * Normalize xri.
     *
     * @param p_xri xri
     * @return xri
     */
    public static Xri normalize(Xri p_xri) {
        if (Utils.isEnclosedInParenthesis(p_xri.asString()) && p_xri.asString().indexOf(Symbols.SLASH.getValue()) != -1) {
            return new XriImpl(Utils.cutParenthesis(p_xri.asString()));
        }
        return p_xri;
    }

    /**
     * Cuts parenthesis from string object if exists, otherwise return string without any changes.
     *
     * @param p_str string object
     * @return string without parenthesis, if there is no parenthesis then return original string
     */
    public static String cutParenthesis(String p_str) {
        if (isEnclosedInParenthesis(p_str)) {
            return p_str.substring(1, p_str.length() - 1);
        }
        return p_str;
    }

    /**
     * Returns arc by xri.
     *
     * @param p_xdiNode xdi node as source of search
     * @param p_type    type of arc (contextual, relational or literal)
     * @param p_arcXri  arc xri
     * @return arc by xri
     */
    public static XdiArc getArcByXri(XdiNode p_xdiNode, XdiStatement.ArcType p_type, String p_arcXri) {
        return getArcByXri(p_xdiNode, p_type, new XriImpl(p_arcXri));
    }

    /**
     * Returns arc by xri.
     *
     * @param p_xdiNode xdi node as source of search
     * @param p_type    type of arc (contextual, relational or literal)
     * @param p_arcXri  arc xri
     * @return arc by xri
     */
    public static XdiArc getArcByXri(XdiNode p_xdiNode, XdiStatement.ArcType p_type, Xri p_arcXri) {
        if (p_xdiNode != null && p_type != null && p_arcXri != null) {
            switch (p_type) {
                case CONTEXTUAL:
                    return getArcByXri(p_xdiNode.getContextualArcList(), p_arcXri);
                case LITERAL:
                    return getArcByXri(p_xdiNode.getLiteralArcList(), p_arcXri);
                case RELATIONAL:
                    return getArcByXri(p_xdiNode.getRelationalArcList(), p_arcXri);
            }
        }
        return null;
    }

    /**
     * Returns arc by xri.
     *
     * @param p_arcList list of arcs
     * @param p_arcXri  arc xri
     * @return arc by xri
     */
    public static XdiArc getArcByXri(Collection<XdiArc> p_arcList, Xri p_arcXri) {
        return getArcByXri(p_arcList, p_arcXri.asString());
    }

    /**
     * Returns arc by xri.
     *
     * @param p_arcList list of arcs
     * @param p_arcXri  arc xri
     * @return arc by xri
     */
    public static XdiArc getArcByXri(Collection<XdiArc> p_arcList, String p_arcXri) {
        if (p_arcList != null && !p_arcList.isEmpty() && p_arcXri != null) {
            for (XdiArc arc : p_arcList) {
                if (arc.getXri().asString().equals(p_arcXri)) {
                    return arc;
                }
            }
        }
        return null;
    }

    /**
     * Gets target contextual node by xri.
     *
     * @param p_node source node
     * @param p_xri  target node
     * @return target contextual node
     */
    public static XdiNode targetContextualNode(XdiNode p_node, String p_xri) {
        if (p_node != null && p_xri != null) {
            for (XdiArc arc : p_node.getContextualArcList()) {
                if (arc.getSource().equals(p_node) && arc.getTarget().getXri().asString().equals(p_xri)) {
                    return arc.getTarget();
                }
            }
        }
        return null;
    }


    /**
     * Converts xdi statement list to string list.
     *
     * @param p_list xdi statement list
     * @return string list
     */
    public static List<String> convert(List<XdiStatement> p_list) {
        final List<String> result = new ArrayList<String>();
        if (p_list != null && !p_list.isEmpty()) {
            for (XdiStatement s : p_list) {
                result.add(s.asString());
            }
        }
        return result;

    }

    /**
     * Converts string list to xdi statement list.
     *
     * @param p_list string list
     * @return xdi statement list
     */
    public static List<XdiStatement> convertTo(List<String> p_list) {
        final List<XdiStatement> result = new ArrayList<XdiStatement>();
        if (p_list != null && !p_list.isEmpty()) {
            for (String s : p_list) {
                final XdiStatement st = XdiStatementImpl.parseSilently(s);
                if (st != null) {
                    result.add(st);
                }
            }
        }
        return result;
    }

    /**
     * Converts string array to xdi statement list.
     *
     * @param p_strings string array
     * @return xdi statement list
     */
    public static List<XdiStatement> convert(String... p_strings) {
        return convertTo(Arrays.asList(p_strings));
    }

    /**
     * Gets variables from string.
     *
     * @param p_string string
     * @return variables
     */
    public static Set<Variable> getVariables(String p_string) {
        final Set<Variable> variables = new HashSet<Variable>();
        int index = 0;
        int interruptCycle = 0; // just avoid forever cycle in case of bug
        while (true) {
            index = extractVariableAndAddToList(p_string, variables, index);
            if (index == -1) {
                break;
            }

            // just avoid forever cycle in case of bug
            interruptCycle++;
            if (interruptCycle >= 100) {
                break;
            }
        }
        return variables;
    }

    /**
     * Extracts variable and add to list.
     *
     * @param p_string    string
     * @param p_variables variables
     * @param p_index     index
     * @return index
     */
    private static int extractVariableAndAddToList(String p_string, Set<Variable> p_variables, int p_index) {
        // not null and more then 4, because minimum for variable is 4 chars, '($1)'
        if (p_string != null && p_string.length() > 4) {
            final int index = p_string.indexOf("($", p_index);

            if (index != -1) {
                int i = 2; // 2 because length of "($" is 2
                int endIndex = -1;
                int interruptCycle = 0; // just avoid forever cycle in case of bug
                while (true) {
                    if (Character.isDigit(p_string.substring(index + i, index + i + 1).charAt(0))) {
                        i++;
                        if (i >= p_string.length()) {
                            break;
                        }
                    } else {
                        endIndex = index + i + 1;
                        break;
                    }

                    // just avoid forever cycle in case of bug
                    interruptCycle++;
                    if (interruptCycle >= 100) {
                        break;
                    }
                }
                if (endIndex != -1 && p_string.substring(endIndex - 1, endIndex).equals(")")) {
                    // catch variable
                    p_variables.add(new VariableImpl(p_string.substring(index, endIndex)));
                    return endIndex;
                }
            }
        }
        return -1;
    }

    /**
     * Returns whether string is inum.
     *
     * @param p_str string to check
     * @return whether string is inum
     */
    public static boolean isINum(String p_str) {
        return p_str.startsWith("=!") || p_str.startsWith("@!");
    }

    /**
     * Parses string silently.
     *
     * @param p_str string to parse
     * @return integer representation or -1 if string can't be parsed
     */
    public static int parseSilently(String p_str) {
        try {
            return Integer.parseInt(p_str);
        } catch (Exception e) {
            return -1;
        }
    }

    /**
     * Lowercased xdi statement xri's except literal node value.
     *
     * @param p_xdiStatement xdi statement
     * @return lowercased xdi statement
     */
    public static XdiStatement smartLowerCase(XdiStatement p_xdiStatement) {
        if (p_xdiStatement != null) {
            final Xri object = p_xdiStatement.getObject();

            final Xri newSubject = toLowercase(p_xdiStatement.getSubject());
            final Xri newPredicate = toLowercase(p_xdiStatement.getPredicate());
            final Xri newObject;
            if (p_xdiStatement.getType() == XdiStatement.ArcType.LITERAL) {
                newObject = new XriImpl(object.asString()); // no lowercase for literal values
            } else {
                newObject = toLowercase(object);
            }
            return new XdiStatementImpl(newSubject, newPredicate, newObject);
        }
        return null;
    }

    /**
     * Lowercase xri.
     *
     * @param p_xri xri
     * @return lowercased xri
     */
    public static Xri toLowercase(Xri p_xri) {
        if (p_xri != null) {
            return new XriImpl(p_xri.asString().toLowerCase());
        }
        return null;
    }

    /**
     * As int, remove first char before parsing
     *
     * @param p_xri xri
     * @return int
     */
    public static int asInt(String p_xri) {
        if (p_xri != null && p_xri.length() > 0) {
            final String substr = p_xri.substring(1); // remove first char
            return parseSilently(substr);
        }
        return -1;
    }

    /**
     * Gets order based on arc with the same source and target that order arc.
     *
     * @param p_arc arc with the same source and target that order arc
     * @return order
     */
    public static int getOrder(XdiArc p_arc) {
        int order = -1;
        for (XdiArc orderArc : p_arc.getSource().getRelationalArcList()) {
            if (orderArc.getTarget().equals(p_arc.getTarget()) && orderArc.getSource().equals(p_arc.getSource())) {
                final String xri = orderArc.getXri().asString();
                if (xri.startsWith(Symbols.ASTERISK.getValue())) {
                    order = asInt(xri);
                    break;
                }
            }
        }
        return order;
    }
}
