package org.xdi.graphmodel.api;

import org.xdi.graphmodel.api.graph.XdiArc;
import org.xdi.graphmodel.api.graph.XdiNode;
import org.xdi.graphmodel.api.graph.XdiStatement;

import java.util.List;

/**
 * Main purpose of this class is to provide convenient methods for graph creation and easy creation of
 * graph elements (like nodes and arcs).
 *
 * @author Yuriy Zabrovarnyy
 * @version 0.9, 25/03/2011
 */
public interface GraphBuilder {

    /**
     * Creates root node. (Should be start point during programmatic graph creation.)
     *
     * @return root node (without arcs)
     */
    public RootNode createRootNode();

    /**
     * @return context node (without arcs)
     */
    public ContextNode createContextNode();

    /**
     * Creates literal node with value.
     *
     * @param value value of literal node
     * @return literal node with value (without arcs)
     */
    public LiteralNode createLiteralNode(LiteralNode.Value value);

    /**
     * Creates literal node with value.
     *
     * @param p_value value of literal node
     * @return literal node with value (without arcs)
     */
    public LiteralNode createLiteralNode(String p_value);

    /**
     * Attach contextual arc to source and target nodes. Both nodes have reference to this arc.
     *
     * @param p_source  source node of arc
     * @param p_target  target node of arc
     * @param p_xriPart xri of arc
     * @return arc with xri and source/target specified
     */
    public XdiArc attachContextualArc(XdiNode p_source, XdiNode p_target, String p_xriPart);

    /**
     * Attach literal arc to source and target nodes. Both nodes have reference to this arc.
     *
     * @param p_source  source node of arc
     * @param p_target  target node of arc
     * @param p_xriPart xri of arc
     * @return arc with xri and source/target specified
     */
    public XdiArc attachLiteralArc(XdiNode p_source, XdiNode p_target, String p_xriPart);

    /**
     * Attach relational arc to source and target nodes. Both nodes have reference to this arc.
     *
     * @param p_source  source node of arc
     * @param p_target  target node of arc
     * @param p_xriPart xri of arc
     * @return arc with xri and source/target specified
     */
    public XdiArc attachRelationalArc(XdiNode p_source, XdiNode p_target, String p_xriPart);

    /**
     * Creates graph model based on xdi statement list.
     *
     * @param p_list xdi statement list
     * @return graph created based on list of xdi statements
     */
    public RootNode createGraph(List<XdiStatement> p_list);

    /**
     * Creates graph model based on xdi statement list. If there is nodes to which relational xdi statements points
     * but such nodes does not exist than they are created automatically and marked with "fake" indicator.
     *
     * @param p_list xdi statement list
     * @return graph created based on list of xdi statements
     */
    public RootNode createGraphWithFakeNodes(List<XdiStatement> p_list);

}
