/*
 * Decompiled with CFR 0.152.
 */
package org.forgerock.audit.handlers.csv;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.io.Writer;
import java.security.PrivateKey;
import java.security.PublicKey;
import java.security.SignatureException;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.Random;
import java.util.concurrent.Executors;
import java.util.concurrent.RejectedExecutionException;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.ReentrantLock;
import javax.crypto.SecretKey;
import javax.crypto.spec.SecretKeySpec;
import org.forgerock.audit.events.handlers.FileBasedEventHandlerConfiguration;
import org.forgerock.audit.events.handlers.writers.RotatableWriter;
import org.forgerock.audit.events.handlers.writers.TextWriter;
import org.forgerock.audit.events.handlers.writers.TextWriterAdapter;
import org.forgerock.audit.handlers.csv.CsvAuditEventHandlerConfiguration;
import org.forgerock.audit.handlers.csv.CsvFormatter;
import org.forgerock.audit.handlers.csv.CsvSecureUtils;
import org.forgerock.audit.handlers.csv.CsvSecureVerifier;
import org.forgerock.audit.handlers.csv.CsvWriter;
import org.forgerock.audit.handlers.csv.HmacCalculator;
import org.forgerock.audit.rotation.RotationContext;
import org.forgerock.audit.rotation.RotationHooks;
import org.forgerock.audit.secure.JcaKeyStoreHandler;
import org.forgerock.audit.secure.KeyStoreHandler;
import org.forgerock.audit.secure.KeyStoreHandlerDecorator;
import org.forgerock.audit.secure.KeyStoreSecureStorage;
import org.forgerock.audit.secure.SecureStorage;
import org.forgerock.audit.secure.SecureStorageException;
import org.forgerock.util.Reject;
import org.forgerock.util.annotations.VisibleForTesting;
import org.forgerock.util.encode.Base64;
import org.forgerock.util.time.Duration;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.supercsv.prefs.CsvPreference;

class SecureCsvWriter
implements CsvWriter,
RotatableWriter.RolloverLifecycleHook {
    private static final Logger logger = LoggerFactory.getLogger(SecureCsvWriter.class);
    private final CsvFormatter csvFormatter;
    private final String[] headers;
    private Writer csvWriter;
    private RotatableWriter rotatableWriter;
    private HmacCalculator hmacCalculator;
    private final ScheduledExecutorService scheduler;
    private final ReentrantLock signatureLock = new ReentrantLock();
    private final Runnable signatureTask;
    private KeyStoreSecureStorage secureStorage;
    private final Duration signatureInterval;
    private ScheduledFuture<?> scheduledSignature;
    private String lastHMAC;
    private byte[] lastSignature;
    private boolean headerWritten = false;
    private final Random random;
    private File keyStoreFile;
    private String keyStorePassword;

    SecureCsvWriter(File csvFile, String[] headers, CsvPreference csvPreference, CsvAuditEventHandlerConfiguration config, KeyStoreHandler keyStoreHandler, Random random) throws IOException {
        Reject.ifFalse((boolean)config.getSecurity().isEnabled(), (String)"SecureCsvWriter should only be used if security is enabled");
        boolean fileAlreadyInitialized = csvFile.exists() && csvFile.length() > 0L;
        this.random = random;
        this.keyStoreFile = new File(csvFile.getPath() + ".keystore");
        this.headers = (String[])Reject.checkNotNull((Object)headers, (String)"The headers can't be null.");
        this.csvFormatter = new CsvFormatter(csvPreference);
        this.csvWriter = this.constructWriter(csvFile, fileAlreadyInitialized, config);
        this.hmacCalculator = new HmacCalculator("HmacSHA256");
        try {
            KeyStoreHandlerDecorator keyStoreHandlerDecorated = new KeyStoreHandlerDecorator(keyStoreHandler);
            SecretKey password = keyStoreHandlerDecorated.readSecretKeyFromKeyStore("Password");
            if (password == null) {
                throw new IllegalArgumentException(String.format("No '%s' symmetric key found in the provided keystore: %s. This key must be provided.", "Password", keyStoreHandlerDecorated.getLocation()));
            }
            this.keyStorePassword = Base64.encode((byte[])password.getEncoded());
            JcaKeyStoreHandler hmacKeyStoreHandler = new JcaKeyStoreHandler("JCEKS", this.keyStoreFile.getPath(), this.keyStorePassword);
            PublicKey publicSignatureKey = keyStoreHandlerDecorated.readPublicKeyFromKeyStore("Signature");
            PrivateKey privateSignatureKey = keyStoreHandlerDecorated.readPrivateKeyFromKeyStore("Signature");
            if (publicSignatureKey == null || privateSignatureKey == null) {
                throw new IllegalArgumentException(String.format("No '%s' signing key found in the provided keystore: %s. This key must be provided.", "Signature", keyStoreHandlerDecorated.getLocation()));
            }
            this.secureStorage = new KeyStoreSecureStorage((KeyStoreHandler)hmacKeyStoreHandler, publicSignatureKey, privateSignatureKey);
            CsvAuditEventHandlerConfiguration.CsvSecurity securityConfiguration = config.getSecurity();
            if (fileAlreadyInitialized) {
                SecretKey currentKey;
                CsvSecureVerifier verifier = new CsvSecureVerifier(csvFile, csvPreference, (SecureStorage)this.secureStorage);
                CsvSecureVerifier.VerificationResult verificationResult = verifier.verify();
                if (!verificationResult.hasPassedVerification()) {
                    throw new IOException("The CSV file was tampered: " + verificationResult.getFailureReason());
                }
                String[] actualHeaders = verifier.getHeaders();
                if (actualHeaders != null) {
                    if (actualHeaders.length != headers.length) {
                        throw new IOException("Resuming an existing CSV file but the headers do not match.");
                    }
                    for (int idx = 0; idx < actualHeaders.length; ++idx) {
                        if (actualHeaders[idx].equals(headers[idx])) continue;
                        throw new IOException("Resuming an existing CSV file but the headers do not match.");
                    }
                }
                if ((currentKey = this.secureStorage.readCurrentKey()) == null) {
                    throw new IllegalStateException("We are supposed to resume but there is not entry for CurrentKey.");
                }
                this.hmacCalculator.setCurrentKey(currentKey.getEncoded());
                this.setLastHMAC(verifier.getLastHMAC());
                this.setLastSignature(verifier.getLastSignature());
                this.headerWritten = true;
            } else {
                this.initHmacCalculatorWithRandomData();
            }
            this.signatureInterval = securityConfiguration.getSignatureIntervalDuration();
            this.scheduler = Executors.newScheduledThreadPool(1);
            this.signatureTask = new Runnable(){

                @Override
                public void run() {
                    try {
                        logger.trace("THREAD !!!");
                        if (!SecureCsvWriter.this.signatureLock.isLocked()) {
                            SecureCsvWriter.this.writeSignature(SecureCsvWriter.this.csvWriter);
                        }
                    }
                    catch (Exception ex) {
                        logger.error("An error occurred while writing the signature", (Throwable)ex);
                    }
                }
            };
        }
        catch (Exception e) {
            throw new RuntimeException("Error when initializing a secure CSV writer", e);
        }
    }

    public void beforeRollingOver() {
        this.signatureLock.lock();
    }

    public void afterRollingOver() {
        this.signatureLock.unlock();
    }

    private void initHmacCalculatorWithRandomData() throws SecureStorageException {
        this.hmacCalculator.setCurrentKey(this.getRandomBytes());
        this.secureStorage.writeInitialKey(this.hmacCalculator.getCurrentKey());
        this.secureStorage.writeCurrentKey(this.hmacCalculator.getCurrentKey());
    }

    private byte[] getRandomBytes() {
        byte[] randomBytes = new byte[32];
        this.random.nextBytes(randomBytes);
        return randomBytes;
    }

    private Writer constructWriter(File csvFile, boolean append, CsvAuditEventHandlerConfiguration config) throws IOException {
        TextWriter.Stream textWriter;
        if (config.getFileRotation().isRotationEnabled()) {
            this.rotatableWriter = new RotatableWriter(csvFile, (FileBasedEventHandlerConfiguration)config, append, (RotatableWriter.RolloverLifecycleHook)this);
            this.rotatableWriter.registerRotationHooks((RotationHooks)new SecureCsvWriterRotationHooks());
            textWriter = this.rotatableWriter;
        } else {
            textWriter = new TextWriter.Stream((OutputStream)new FileOutputStream(csvFile, append));
        }
        if (config.getBuffering().isEnabled()) {
            logger.warn("Secure CSV logging does not support buffering. Buffering config will be ignored.");
        }
        return new TextWriterAdapter((TextWriter)textWriter);
    }

    @Override
    public void flush() throws IOException {
        this.csvWriter.flush();
    }

    @Override
    public void close() throws IOException {
        this.flush();
        this.signatureLock.lock();
        try {
            this.forceWriteSignature(this.csvWriter);
        }
        finally {
            this.signatureLock.unlock();
        }
        this.scheduler.shutdown();
        try {
            while (!this.scheduler.awaitTermination(500L, TimeUnit.MILLISECONDS)) {
                logger.debug("Waiting to terminate the scheduler.");
            }
        }
        catch (InterruptedException ex) {
            logger.error("Unable to terminate the scheduler", (Throwable)ex);
            Thread.currentThread().interrupt();
        }
        this.csvWriter.close();
    }

    private void forceWriteSignature(Writer writer) throws IOException {
        if (this.scheduledSignature != null && this.scheduledSignature.cancel(false)) {
            this.writeSignature(writer);
        }
    }

    public void writeHeader(String ... header) throws IOException {
        this.writeHeader(this.csvWriter, header);
    }

    public void writeHeader(Writer writer, String ... header) throws IOException {
        String[] newHeader = this.addExtraColumns(header);
        writer.write(this.csvFormatter.formatHeader(newHeader));
        logger.trace("Header written to file");
        this.headerWritten = true;
    }

    @VisibleForTesting
    void writeSignature(Writer writer) throws IOException {
        this.signatureLock.lock();
        try {
            this.lastSignature = this.secureStorage.sign(CsvSecureUtils.dataToSign(this.lastSignature, this.lastHMAC));
            logger.trace("Calculated new Signature");
            Map<String, String> values = Collections.singletonMap("SIGNATURE", Base64.encode((byte[])this.lastSignature));
            this.writeEvent(writer, values);
            logger.trace("Signature written to file");
            this.secureStorage.writeCurrentSignatureKey((SecretKey)new SecretKeySpec(this.lastSignature, "SHA256withRSA"));
            logger.trace("Signature written to secureStorage");
        }
        catch (SecureStorageException ex) {
            logger.error(ex.getMessage(), (Throwable)ex);
            throw new IOException(ex);
        }
        finally {
            this.signatureLock.unlock();
            this.flush();
        }
    }

    @Override
    public boolean forceRotation() throws IOException {
        return this.rotatableWriter != null ? this.rotatableWriter.forceRotation() : false;
    }

    @Override
    public void writeEvent(Map<String, String> values) throws IOException {
        this.writeEvent(this.csvWriter, values);
    }

    public void writeEvent(Writer writer, Map<String, String> values) throws IOException {
        this.signatureLock.lock();
        try {
            if (!this.headerWritten) {
                this.writeHeader(this.headers);
            }
            String[] extendedHeaders = this.addExtraColumns(this.headers);
            HashMap<String, String> extendedValues = new HashMap<String, String>(values);
            if (!values.containsKey("SIGNATURE")) {
                this.insertHMACSignature(extendedValues, this.headers);
            }
            writer.write(this.csvFormatter.formatEvent(extendedValues, extendedHeaders));
            writer.flush();
            this.secureStorage.writeCurrentKey(this.hmacCalculator.getCurrentKey());
            if (!values.containsKey("SIGNATURE") && (this.scheduledSignature == null || this.scheduledSignature.isDone())) {
                logger.trace("Triggering a new signature task to be executed in {}", (Object)this.signatureInterval);
                try {
                    logger.trace("SHCHEDULED!!");
                    this.scheduledSignature = this.scheduler.schedule(this.signatureTask, this.signatureInterval.getValue(), this.signatureInterval.getUnit());
                }
                catch (RejectedExecutionException e) {
                    logger.error(e.getMessage(), (Throwable)e);
                }
            }
        }
        catch (SecureStorageException ex) {
            throw new IOException(ex);
        }
        finally {
            this.signatureLock.unlock();
        }
    }

    private void insertHMACSignature(Map<String, String> values, String[] nameMapping) throws IOException {
        try {
            this.lastHMAC = this.hmacCalculator.calculate(CsvSecureUtils.dataToSign(logger, values, nameMapping));
            values.put("HMAC", this.lastHMAC);
        }
        catch (SignatureException ex) {
            logger.error(ex.getMessage(), (Throwable)ex);
            throw new IOException(ex);
        }
    }

    private String[] addExtraColumns(String ... header) {
        String[] newHeader = new String[header.length + 2];
        System.arraycopy(header, 0, newHeader, 0, header.length);
        newHeader[header.length] = "HMAC";
        newHeader[header.length + 1] = "SIGNATURE";
        return newHeader;
    }

    private void setLastHMAC(String lastHMac) {
        this.lastHMAC = lastHMac;
    }

    private void setLastSignature(byte[] lastSignature) {
        this.lastSignature = lastSignature;
    }

    private void writeLastSignature(Writer writer) throws IOException {
        this.signatureLock.lock();
        try {
            Map<String, String> values = Collections.singletonMap("SIGNATURE", Base64.encode((byte[])this.lastSignature));
            this.writeEvent(writer, values);
            logger.trace("Signature from previous file written to new file");
        }
        catch (IOException ex) {
            logger.error(ex.getMessage(), (Throwable)ex);
            throw new IOException(ex);
        }
        finally {
            this.signatureLock.unlock();
        }
    }

    private class SecureCsvWriterRotationHooks
    implements RotationHooks {
        private SecureCsvWriterRotationHooks() {
        }

        public void preRotationAction(RotationContext context) throws IOException {
            SecureCsvWriter.this.forceWriteSignature(context.getWriter());
        }

        public void postRotationAction(RotationContext context) throws IOException {
            String currentName = SecureCsvWriter.this.keyStoreFile.getName();
            String nextName = currentName.replaceFirst(context.getInitialFile().getName(), context.getNextFile().getName());
            File nextFile = new File(SecureCsvWriter.this.keyStoreFile.getParent(), nextName);
            logger.trace("Renaming keystore file {} to {}", (Object)currentName, (Object)nextName);
            boolean renamed = SecureCsvWriter.this.keyStoreFile.renameTo(nextFile);
            if (!renamed) {
                logger.error("Unable to rename {} to {}", (Object)SecureCsvWriter.this.keyStoreFile.getAbsolutePath(), (Object)nextFile.getAbsolutePath());
            }
            try {
                SecureCsvWriter.this.secureStorage.setKeyStoreHandler((KeyStoreHandler)new JcaKeyStoreHandler("JCEKS", SecureCsvWriter.this.keyStoreFile.getPath(), SecureCsvWriter.this.keyStorePassword));
                logger.trace("Updated secureStorage to reference new keyStoreFile");
                SecureCsvWriter.this.initHmacCalculatorWithRandomData();
            }
            catch (Exception ex) {
                throw new IOException(ex);
            }
            Writer writer = context.getWriter();
            SecureCsvWriter.this.writeHeader(writer, SecureCsvWriter.this.headers);
            SecureCsvWriter.this.writeLastSignature(writer);
            writer.flush();
        }
    }
}

