/*
 * Decompiled with CFR 0.152.
 */
package org.gluu.oxauth.model.crypto;

import com.google.common.collect.Lists;
import com.nimbusds.jose.JWSAlgorithm;
import com.nimbusds.jose.crypto.impl.ECDSA;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.math.BigInteger;
import java.security.InvalidParameterException;
import java.security.Key;
import java.security.KeyPair;
import java.security.KeyPairGenerator;
import java.security.KeyStore;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.security.PrivateKey;
import java.security.PublicKey;
import java.security.SecureRandom;
import java.security.Signature;
import java.security.SignatureException;
import java.security.UnrecoverableKeyException;
import java.security.cert.Certificate;
import java.security.cert.CertificateException;
import java.security.cert.X509Certificate;
import java.security.interfaces.ECPublicKey;
import java.security.interfaces.RSAPublicKey;
import java.security.spec.ECGenParameterSpec;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.List;
import java.util.UUID;
import java.util.stream.Collectors;
import javax.crypto.Mac;
import javax.crypto.spec.SecretKeySpec;
import org.apache.commons.codec.binary.Base64;
import org.apache.commons.io.FilenameUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.log4j.Logger;
import org.bouncycastle.asn1.ASN1Encodable;
import org.bouncycastle.asn1.ASN1EncodableVector;
import org.bouncycastle.asn1.ASN1ObjectIdentifier;
import org.bouncycastle.asn1.DERSequence;
import org.bouncycastle.asn1.x500.X500Name;
import org.bouncycastle.asn1.x509.KeyPurposeId;
import org.bouncycastle.cert.CertIOException;
import org.bouncycastle.cert.X509CertificateHolder;
import org.bouncycastle.cert.jcajce.JcaX509CertificateConverter;
import org.bouncycastle.cert.jcajce.JcaX509v3CertificateBuilder;
import org.bouncycastle.operator.ContentSigner;
import org.bouncycastle.operator.OperatorCreationException;
import org.bouncycastle.operator.jcajce.JcaContentSignerBuilder;
import org.gluu.oxauth.model.configuration.AppConfiguration;
import org.gluu.oxauth.model.crypto.AbstractCryptoProvider;
import org.gluu.oxauth.model.crypto.signature.AlgorithmFamily;
import org.gluu.oxauth.model.crypto.signature.SignatureAlgorithm;
import org.gluu.oxauth.model.jwk.Algorithm;
import org.gluu.oxauth.model.jwk.JSONWebKey;
import org.gluu.oxauth.model.jwk.JSONWebKeySet;
import org.gluu.oxauth.model.jwk.KeySelectionStrategy;
import org.gluu.oxauth.model.jwk.Use;
import org.gluu.oxauth.model.util.Base64Util;
import org.gluu.oxauth.model.util.CertUtils;
import org.gluu.oxauth.model.util.Util;
import org.gluu.util.security.SecurityProviderUtility;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

public class OxAuthCryptoProvider
extends AbstractCryptoProvider {
    protected static final Logger LOG = Logger.getLogger(OxAuthCryptoProvider.class);
    private KeyStore keyStore;
    private String keyStoreFile;
    private String keyStoreSecret;
    private String dnName;
    private final boolean rejectNoneAlg;
    private final KeySelectionStrategy keySelectionStrategy;

    public OxAuthCryptoProvider() throws Exception {
        this(null, null, null);
    }

    public OxAuthCryptoProvider(String keyStoreFile, String keyStoreSecret, String dnName) throws Exception {
        this(keyStoreFile, keyStoreSecret, dnName, false);
    }

    public OxAuthCryptoProvider(String keyStoreFile, String keyStoreSecret, String dnName, boolean rejectNoneAlg) throws Exception {
        this(keyStoreFile, keyStoreSecret, dnName, rejectNoneAlg, AppConfiguration.DEFAULT_KEY_SELECTION_STRATEGY);
    }

    public OxAuthCryptoProvider(String keyStoreFile, String keyStoreSecret, String dnName, boolean rejectNoneAlg, KeySelectionStrategy keySelectionStrategy) throws Exception {
        this.rejectNoneAlg = rejectNoneAlg;
        KeySelectionStrategy keySelectionStrategy2 = this.keySelectionStrategy = keySelectionStrategy != null ? keySelectionStrategy : AppConfiguration.DEFAULT_KEY_SELECTION_STRATEGY;
        if (!Util.isNullOrEmpty(keyStoreFile) && !Util.isNullOrEmpty(keyStoreSecret)) {
            this.keyStoreFile = keyStoreFile;
            this.keyStoreSecret = keyStoreSecret;
            this.dnName = dnName;
            SecurityProviderUtility.KeyStorageType keyStorageType = this.solveKeyStorageType();
            switch (keyStorageType) {
                case JKS_KS: {
                    this.keyStore = KeyStore.getInstance("JKS");
                    break;
                }
                case PKCS12_KS: {
                    this.keyStore = KeyStore.getInstance("PKCS12", SecurityProviderUtility.getBCProvider());
                    break;
                }
                case BCFKS_KS: {
                    this.keyStore = KeyStore.getInstance("BCFKS", SecurityProviderUtility.getBCProvider());
                }
            }
            try {
                File f = new File(keyStoreFile);
                if (!f.exists()) {
                    this.keyStore.load(null, keyStoreSecret.toCharArray());
                    FileOutputStream fos = new FileOutputStream(keyStoreFile);
                    this.keyStore.store(fos, keyStoreSecret.toCharArray());
                    fos.close();
                }
                FileInputStream is = new FileInputStream(keyStoreFile);
                this.keyStore.load(is, keyStoreSecret.toCharArray());
                LOG.debug((Object)"Loaded keys from keystore.");
                LOG.debug((Object)("Security Mode: " + SecurityProviderUtility.getSecurityMode().toString()));
                LOG.debug((Object)("Keystore Type: " + keyStorageType.toString()));
                LOG.trace((Object)("Loaded keys:" + this.getKeys()));
            }
            catch (Exception e) {
                LOG.error((Object)e.getMessage(), (Throwable)e);
                LOG.error((Object)("Check type of keystorage. Expected keystorage type: '" + keyStorageType.toString() + "'"));
            }
        }
    }

    public void load(String keyStoreSecret) {
        this.keyStoreSecret = keyStoreSecret;
        SecurityProviderUtility.KeyStorageType keyStorageType = this.solveKeyStorageType();
        try (FileInputStream is = new FileInputStream(this.keyStoreFile);){
            switch (keyStorageType) {
                case JKS_KS: {
                    this.keyStore = KeyStore.getInstance("JKS");
                    break;
                }
                case PKCS12_KS: {
                    this.keyStore = KeyStore.getInstance("PKCS12", SecurityProviderUtility.getBCProvider());
                    break;
                }
                case BCFKS_KS: {
                    this.keyStore = KeyStore.getInstance("BCFKS", SecurityProviderUtility.getBCProvider());
                }
            }
            this.keyStore.load(is, keyStoreSecret.toCharArray());
            LOG.debug((Object)"Loaded keys from keystore.");
            LOG.debug((Object)("Security Mode: " + SecurityProviderUtility.getSecurityMode().toString()));
            LOG.debug((Object)("Keystore Type: " + keyStorageType.toString()));
            LOG.trace((Object)("Loaded keys:" + this.getKeys()));
        }
        catch (Exception e) {
            LOG.error((Object)e.getMessage(), (Throwable)e);
            LOG.error((Object)("Check type of keystorage. Expected keystorage type: '" + keyStorageType.toString() + "'"));
        }
    }

    public String getKeyStoreFile() {
        return this.keyStoreFile;
    }

    public String getKeyStoreSecret() {
        return this.keyStoreSecret;
    }

    public String getDnName() {
        return this.dnName;
    }

    @Override
    public JSONObject generateKey(Algorithm algorithm, Long expirationTime, Use use) throws Exception {
        return this.generateKey(algorithm, expirationTime, use, 2048);
    }

    @Override
    public JSONObject generateKey(Algorithm algorithm, Long expirationTime, Use use, int keyLength) throws Exception {
        KeyPairGenerator keyGen = null;
        SignatureAlgorithm signatureAlgorithm = SignatureAlgorithm.fromString(algorithm.getParamName());
        if (signatureAlgorithm == null) {
            signatureAlgorithm = SignatureAlgorithm.RS256;
        }
        if (algorithm == null) {
            throw new RuntimeException("The signature algorithm parameter cannot be null");
        }
        if (AlgorithmFamily.RSA.equals((Object)algorithm.getFamily())) {
            keyGen = KeyPairGenerator.getInstance(algorithm.getFamily().toString(), SecurityProviderUtility.getBCProvider());
            keyGen.initialize(keyLength, new SecureRandom());
        } else if (AlgorithmFamily.EC.equals((Object)algorithm.getFamily())) {
            ECGenParameterSpec eccgen = new ECGenParameterSpec(signatureAlgorithm.getCurve().getAlias());
            keyGen = KeyPairGenerator.getInstance(algorithm.getFamily().toString(), SecurityProviderUtility.getBCProvider());
            keyGen.initialize(eccgen, new SecureRandom());
        } else {
            throw new RuntimeException("The provided signature algorithm parameter is not supported");
        }
        KeyPair keyPair = keyGen.generateKeyPair();
        PrivateKey pk = keyPair.getPrivate();
        X509Certificate cert = this.generateV3Certificate(keyPair, this.dnName, signatureAlgorithm.getAlgorithm(), expirationTime);
        Certificate[] chain = new X509Certificate[]{cert};
        String alias = UUID.randomUUID().toString() + OxAuthCryptoProvider.getKidSuffix(use, algorithm);
        this.keyStore.setKeyEntry(alias, pk, this.keyStoreSecret.toCharArray(), chain);
        String oldAliasByAlgorithm = this.getAliasByAlgorithmForDeletion(algorithm, alias, use);
        if (StringUtils.isNotBlank((String)oldAliasByAlgorithm)) {
            this.keyStore.deleteEntry(oldAliasByAlgorithm);
            LOG.trace((Object)("New key: " + alias + ", deleted key: " + oldAliasByAlgorithm));
        }
        FileOutputStream stream = new FileOutputStream(this.keyStoreFile);
        this.keyStore.store(stream, this.keyStoreSecret.toCharArray());
        stream.close();
        PublicKey publicKey = keyPair.getPublic();
        JSONObject jsonObject = new JSONObject();
        jsonObject.put("kty", (Object)algorithm.getFamily());
        jsonObject.put("kid", (Object)alias);
        jsonObject.put("use", (Object)use.getParamName());
        jsonObject.put("alg", (Object)algorithm.getParamName());
        jsonObject.put("exp", (Object)expirationTime);
        if (publicKey instanceof RSAPublicKey) {
            RSAPublicKey rsaPublicKey = (RSAPublicKey)publicKey;
            jsonObject.put("n", (Object)Base64Util.base64urlencodeUnsignedBigInt(rsaPublicKey.getModulus()));
            jsonObject.put("e", (Object)Base64Util.base64urlencodeUnsignedBigInt(rsaPublicKey.getPublicExponent()));
        } else if (publicKey instanceof ECPublicKey) {
            ECPublicKey ecPublicKey = (ECPublicKey)publicKey;
            jsonObject.put("crv", (Object)signatureAlgorithm.getCurve().getName());
            jsonObject.put("x", (Object)Base64Util.base64urlencodeUnsignedBigInt(ecPublicKey.getW().getAffineX()));
            jsonObject.put("y", (Object)Base64Util.base64urlencodeUnsignedBigInt(ecPublicKey.getW().getAffineY()));
        }
        JSONArray x5c = new JSONArray();
        x5c.put((Object)Base64.encodeBase64String((byte[])cert.getEncoded()));
        jsonObject.put("x5c", (Object)x5c);
        return jsonObject;
    }

    private static String getKidSuffix(Use use, Algorithm algorithm) {
        return "_" + use.getParamName().toLowerCase() + "_" + algorithm.getParamName().toLowerCase();
    }

    public String getAliasByAlgorithmForDeletion(Algorithm algorithm, String newAlias, Use use) throws KeyStoreException {
        for (String alias : Collections.list(this.keyStore.aliases())) {
            if (newAlias.equals(alias) || !alias.endsWith(OxAuthCryptoProvider.getKidSuffix(use, algorithm))) continue;
            return alias;
        }
        return null;
    }

    @Override
    public boolean containsKey(String keyId) {
        try {
            if (StringUtils.isBlank((String)keyId)) {
                return false;
            }
            return this.keyStore.getKey(keyId, this.keyStoreSecret.toCharArray()) != null;
        }
        catch (Exception e) {
            LOG.error((Object)e.getMessage(), (Throwable)e);
            return false;
        }
    }

    @Override
    public String sign(String signingInput, String alias, String sharedSecret, SignatureAlgorithm signatureAlgorithm) throws Exception {
        if (signatureAlgorithm == SignatureAlgorithm.NONE) {
            return "";
        }
        if (AlgorithmFamily.HMAC.equals((Object)signatureAlgorithm.getFamily())) {
            SecretKeySpec secretKey = new SecretKeySpec(sharedSecret.getBytes("UTF-8"), signatureAlgorithm.getAlgorithm());
            Mac mac = Mac.getInstance(signatureAlgorithm.getAlgorithm());
            mac.init(secretKey);
            byte[] sig = mac.doFinal(signingInput.getBytes());
            return Base64Util.base64urlencode(sig);
        }
        PrivateKey privateKey = this.getPrivateKey(alias);
        if (privateKey == null) {
            String error = "Failed to find private key by kid: " + alias + ", signatureAlgorithm: " + (Object)((Object)signatureAlgorithm) + "(check whether web keys JSON in persistence corresponds to keystore file), keySelectionStrategy: " + (Object)((Object)this.keySelectionStrategy);
            LOG.error((Object)error);
            throw new RuntimeException(error);
        }
        Signature signer = Signature.getInstance(signatureAlgorithm.getAlgorithm(), SecurityProviderUtility.getBCProvider());
        signer.initSign(privateKey);
        signer.update(signingInput.getBytes());
        byte[] signature = signer.sign();
        if (AlgorithmFamily.EC.equals((Object)signatureAlgorithm.getFamily())) {
            int signatureLenght = ECDSA.getSignatureByteArrayLength((JWSAlgorithm)JWSAlgorithm.parse((String)signatureAlgorithm.getName()));
            signature = ECDSA.transcodeSignatureToConcat((byte[])signature, (int)signatureLenght);
        }
        return Base64Util.base64urlencode(signature);
    }

    @Override
    public boolean verifySignature(String signingInput, String encodedSignature, String alias, JSONObject jwks, String sharedSecret, SignatureAlgorithm signatureAlgorithm) throws Exception {
        if (this.rejectNoneAlg && signatureAlgorithm == SignatureAlgorithm.NONE) {
            LOG.trace((Object)"None algorithm is forbidden by `rejectJwtWithNoneAlg` property.");
            return false;
        }
        if (signatureAlgorithm == SignatureAlgorithm.NONE) {
            return Util.isNullOrEmpty(encodedSignature);
        }
        if (AlgorithmFamily.HMAC.equals((Object)signatureAlgorithm.getFamily())) {
            String expectedSignature = this.sign(signingInput, null, sharedSecret, signatureAlgorithm);
            return expectedSignature.equals(encodedSignature);
        }
        PublicKey publicKey = null;
        try {
            byte[] signature;
            publicKey = jwks == null ? this.getPublicKey(alias) : this.getPublicKey(alias, jwks, signatureAlgorithm.getAlg());
            if (publicKey == null) {
                return false;
            }
            byte[] signatureDer = signature = Base64Util.base64urldecode(encodedSignature);
            if (AlgorithmFamily.EC.equals((Object)signatureAlgorithm.getFamily())) {
                signatureDer = ECDSA.transcodeSignatureToDER((byte[])signatureDer);
            }
            Signature verifier = Signature.getInstance(signatureAlgorithm.getAlgorithm(), SecurityProviderUtility.getBCProvider());
            verifier.initVerify(publicKey);
            verifier.update(signingInput.getBytes());
            try {
                return verifier.verify(signatureDer);
            }
            catch (SignatureException e) {
                return verifier.verify(signature);
            }
        }
        catch (Exception e) {
            LOG.error((Object)e.getMessage(), (Throwable)e);
            return false;
        }
    }

    private String getJWKSValue(JSONObject jwks, String node) throws JSONException {
        try {
            return jwks.getString(node);
        }
        catch (Exception ex) {
            JSONObject publicKey = jwks.getJSONObject("publicKey");
            return publicKey.getString(node);
        }
    }

    @Override
    public boolean deleteKey(String alias) throws Exception {
        this.keyStore.deleteEntry(alias);
        FileOutputStream stream = new FileOutputStream(this.keyStoreFile);
        this.keyStore.store(stream, this.keyStoreSecret.toCharArray());
        stream.close();
        return true;
    }

    public PublicKey getPublicKey(String alias) {
        PublicKey publicKey = null;
        try {
            if (Util.isNullOrEmpty(alias)) {
                return null;
            }
            Certificate certificate = this.keyStore.getCertificate(alias);
            if (certificate == null) {
                return null;
            }
            publicKey = certificate.getPublicKey();
            this.checkKeyExpiration(alias);
        }
        catch (KeyStoreException e) {
            e.printStackTrace();
        }
        return publicKey;
    }

    @Override
    public String getKeyId(JSONWebKeySet jsonWebKeySet, Algorithm algorithm, Use use) throws Exception {
        JSONWebKey selectedKey;
        if (algorithm == null || AlgorithmFamily.HMAC.equals((Object)algorithm.getFamily())) {
            return null;
        }
        String kid = null;
        List<JSONWebKey> keys = jsonWebKeySet.getKeys();
        LOG.trace((Object)("WebKeys:" + keys.stream().map(JSONWebKey::getKid).collect(Collectors.toList())));
        LOG.trace((Object)("KeyStoreKeys:" + this.getKeys()));
        ArrayList<JSONWebKey> keysByAlgAndUse = new ArrayList<JSONWebKey>();
        for (JSONWebKey key : keys) {
            Key keyFromStore;
            if (algorithm != key.getAlg() || use != null && use != key.getUse() || (keyFromStore = this.keyStore.getKey(kid = key.getKid(), this.keyStoreSecret.toCharArray())) == null) continue;
            keysByAlgAndUse.add(key);
        }
        if (keysByAlgAndUse.isEmpty()) {
            LOG.trace((Object)("kid is not in keystore, algorithm: " + (Object)((Object)algorithm) + ", kid: " + kid + ", keyStorePath:" + this.keyStoreFile));
            return kid;
        }
        if (LOG.isTraceEnabled()) {
            LOG.trace((Object)("Select among keys (\u00e4lg: " + (Object)((Object)algorithm) + ", use: " + (Object)((Object)use) + "): " + this.traceWithExp(keysByAlgAndUse)));
        }
        String selectedKid = (selectedKey = this.keySelectionStrategy.select(keysByAlgAndUse)) != null ? selectedKey.getKid() : null;
        LOG.trace((Object)("Selected kid: " + selectedKid + ", keySelectionStrategy: " + (Object)((Object)this.keySelectionStrategy)));
        return selectedKid;
    }

    private String traceWithExp(List<JSONWebKey> list) {
        StringBuilder sb = new StringBuilder("[");
        for (JSONWebKey key : list) {
            sb.append("{\"kid\":").append(key.getKid()).append(",\"exp\":").append(key.getExp()).append("},");
        }
        sb.append("]");
        return sb.toString();
    }

    @Override
    public PrivateKey getPrivateKey(String alias) throws UnrecoverableKeyException, NoSuchAlgorithmException, KeyStoreException {
        if (Util.isNullOrEmpty(alias)) {
            return null;
        }
        Key key = this.keyStore.getKey(alias, this.keyStoreSecret.toCharArray());
        if (key == null) {
            return null;
        }
        PrivateKey privateKey = (PrivateKey)key;
        this.checkKeyExpiration(alias);
        return privateKey;
    }

    public X509Certificate generateV3Certificate(KeyPair keyPair, String issuer, String signatureAlgorithm, Long expirationTime) throws CertIOException, OperatorCreationException, CertificateException {
        PrivateKey privateKey = keyPair.getPrivate();
        PublicKey publicKey = keyPair.getPublic();
        X500Name issuerName = new X500Name(issuer);
        X500Name subjectName = new X500Name(issuer);
        BigInteger serial = new BigInteger(256, new SecureRandom());
        Date notBefore = new Date(System.currentTimeMillis() - 10000L);
        Date notAfter = new Date(expirationTime);
        JcaX509v3CertificateBuilder builder = new JcaX509v3CertificateBuilder(issuerName, serial, notBefore, notAfter, subjectName, publicKey);
        ASN1EncodableVector purposes = new ASN1EncodableVector();
        purposes.add((ASN1Encodable)KeyPurposeId.id_kp_serverAuth);
        purposes.add((ASN1Encodable)KeyPurposeId.id_kp_clientAuth);
        purposes.add((ASN1Encodable)KeyPurposeId.anyExtendedKeyUsage);
        ASN1ObjectIdentifier extendedKeyUsage = new ASN1ObjectIdentifier("2.5.29.37").intern();
        builder.addExtension(extendedKeyUsage, false, (ASN1Encodable)new DERSequence(purposes));
        ContentSigner signer = new JcaContentSignerBuilder(signatureAlgorithm).setProvider(SecurityProviderUtility.getBCProvider()).build(privateKey);
        X509CertificateHolder holder = builder.build(signer);
        X509Certificate cert = new JcaX509CertificateConverter().setProvider(SecurityProviderUtility.getBCProvider()).getCertificate(holder);
        return cert;
    }

    @Override
    public List<String> getKeys() {
        try {
            return Collections.list(this.keyStore.aliases());
        }
        catch (KeyStoreException e) {
            LOG.error((Object)e.getMessage(), (Throwable)e);
            return Lists.newArrayList();
        }
    }

    public SignatureAlgorithm getSignatureAlgorithm(String alias) throws KeyStoreException {
        Certificate[] chain = this.keyStore.getCertificateChain(alias);
        if (chain == null || chain.length == 0) {
            return null;
        }
        X509Certificate cert = (X509Certificate)chain[0];
        return CertUtils.getSignatureAlgorithm(cert);
    }

    private void checkKeyExpiration(String alias) {
        try {
            Date expirationDate = ((X509Certificate)this.keyStore.getCertificate(alias)).getNotAfter();
            this.checkKeyExpiration(alias, expirationDate.getTime());
        }
        catch (KeyStoreException e) {
            e.printStackTrace();
        }
    }

    public KeyStore getKeyStore() {
        return this.keyStore;
    }

    public static boolean checkExtension(String extension, SecurityProviderUtility.SecurityModeType securityMode) {
        boolean res = false;
        if (securityMode != null) {
            res = securityMode.toString().equals(extension);
        }
        return res;
    }

    private SecurityProviderUtility.KeyStorageType solveKeyStorageType() {
        SecurityProviderUtility.SecurityModeType securityMode = SecurityProviderUtility.getSecurityMode();
        if (securityMode == null) {
            throw new InvalidParameterException("Security Mode wasn't initialized. Call installBCProvider() before");
        }
        String keyStoreExt = FilenameUtils.getExtension((String)this.keyStoreFile);
        SecurityProviderUtility.KeyStorageType keyStorageType = SecurityProviderUtility.KeyStorageType.fromExtension((String)keyStoreExt);
        boolean ksTypeFound = false;
        for (SecurityProviderUtility.KeyStorageType ksType : securityMode.getKeystorageTypes()) {
            if (keyStorageType != ksType) continue;
            ksTypeFound = true;
            break;
        }
        if (!ksTypeFound) {
            switch (securityMode) {
                case BCFIPS_SECURITY_MODE: {
                    keyStorageType = SecurityProviderUtility.KeyStorageType.BCFKS_KS;
                    break;
                }
                case BCPROV_SECURITY_MODE: {
                    keyStorageType = SecurityProviderUtility.KeyStorageType.PKCS12_KS;
                }
            }
        }
        return keyStorageType;
    }
}

