/*
 * Decompiled with CFR 0.152.
 */
package org.gluu.idp.storage;

import java.io.IOException;
import java.time.Duration;
import java.util.Date;
import java.util.TimerTask;
import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import net.shibboleth.utilities.java.support.annotation.constraint.NonNegative;
import net.shibboleth.utilities.java.support.collection.Pair;
import net.shibboleth.utilities.java.support.component.ComponentSupport;
import net.shibboleth.utilities.java.support.component.InitializableComponent;
import net.shibboleth.utilities.java.support.logic.Constraint;
import net.shibboleth.utilities.java.support.primitive.StringSupport;
import org.cryptacular.util.ByteUtil;
import org.cryptacular.util.CodecUtil;
import org.cryptacular.util.HashUtil;
import org.gluu.idp.externalauth.openid.conf.IdpConfigurationFactory;
import org.gluu.idp.storage.VersionMutableStorageRecord;
import org.gluu.persist.PersistenceEntryManager;
import org.gluu.service.cache.CacheConfiguration;
import org.gluu.service.cache.CacheProvider;
import org.gluu.service.cache.StandaloneCacheProviderFactory;
import org.gluu.util.security.StringEncrypter;
import org.opensaml.storage.AbstractStorageService;
import org.opensaml.storage.StorageCapabilitiesEx;
import org.opensaml.storage.StorageRecord;
import org.opensaml.storage.VersionMismatchException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class GluuStorageService
extends AbstractStorageService
implements StorageCapabilitiesEx {
    private final Logger LOG = LoggerFactory.getLogger(GluuStorageService.class);
    @NonNegative
    private Duration contextExpiration;
    private static final int MAX_KEY_LENGTH = 250;
    private final IdpConfigurationFactory configurationFactory;
    private CacheProvider<?> cacheProvider;

    public GluuStorageService(IdpConfigurationFactory configurationFactory) {
        this.LOG.debug("GluuStorage: create");
        Constraint.isNotNull((Object)((Object)configurationFactory), (String)"Configuration factory cannot be null");
        this.configurationFactory = configurationFactory;
        this.cacheProvider = this.createCacheProvider();
        Constraint.isNotNull(this.cacheProvider, (String)"Cache Provider factory cannot be null");
        this.contextExpiration = Duration.ZERO;
        this.initCacheCapabilities();
    }

    private CacheProvider<?> createCacheProvider() {
        this.LOG.debug("GluuStorage: createCacheProvider");
        StringEncrypter stringEncrypter = this.configurationFactory.getStringEncrypter();
        PersistenceEntryManager persistenceEntryManager = this.configurationFactory.getPersistenceEntryManager();
        CacheConfiguration cacheConfiguration = this.configurationFactory.getCacheConfiguration();
        StandaloneCacheProviderFactory cacheProviderFactory = new StandaloneCacheProviderFactory(persistenceEntryManager, stringEncrypter);
        return cacheProviderFactory.getCacheProvider(cacheConfiguration);
    }

    private void initCacheCapabilities() {
        this.LOG.debug("GluuStorage: initCacheCapabilities");
        this.setContextSize(Integer.MAX_VALUE);
        this.setKeySize(Integer.MAX_VALUE);
        this.setValueSize(Integer.MAX_VALUE);
    }

    public boolean create(@Nonnull String context, @Nonnull String key, @Nonnull String value, @Nullable Long expiration) throws IOException {
        this.LOG.debug("GluuStorage: create");
        Constraint.isNotNull((Object)StringSupport.trimOrNull((String)context), (String)"Context cannot be null or empty");
        Constraint.isNotNull((Object)StringSupport.trimOrNull((String)key), (String)"Key cannot be null or empty");
        Constraint.isNotNull((Object)StringSupport.trimOrNull((String)value), (String)"Value cannot be null or empty");
        VersionMutableStorageRecord record = new VersionMutableStorageRecord(value, expiration, 1L);
        int expiry = record.getExpiry();
        Constraint.isGreaterThan((int)-1, (int)expiry, (String)"Expiration must be null or positive");
        String namespace = this.lookupNamespace(context);
        if (namespace == null) {
            namespace = this.createNamespace(context);
        }
        String cacheKey = this.memcachedKey(namespace, key);
        this.LOG.debug("Creating new entry at {} for context={}, key={}, exp={}", new Object[]{cacheKey, context, key, expiry});
        try {
            int ttl = this.getSystemExpiration(record.getExpiration());
            this.cacheProvider.put(ttl, cacheKey, (Object)record);
        }
        catch (Exception ex) {
            this.LOG.error("Failed to put object into cache, key: '{}'", (Object)cacheKey, (Object)ex);
            return false;
        }
        return true;
    }

    @Nullable
    public StorageRecord read(@Nonnull String context, @Nonnull String key) throws IOException {
        VersionMutableStorageRecord record;
        this.LOG.debug("GluuStorage: read (key)");
        Constraint.isNotNull((Object)StringSupport.trimOrNull((String)context), (String)"Context cannot be null or empty");
        Constraint.isNotNull((Object)StringSupport.trimOrNull((String)key), (String)"Key cannot be null or empty");
        String namespace = this.lookupNamespace(context);
        if (namespace == null) {
            this.LOG.debug("Namespace for context {} does not exist", (Object)context);
            return null;
        }
        String cacheKey = this.memcachedKey(namespace, key);
        this.LOG.debug("Reading entry at {} for context={}, key={}", new Object[]{cacheKey, context, key});
        try {
            record = (VersionMutableStorageRecord)this.cacheProvider.get(cacheKey);
        }
        catch (Exception ex) {
            this.LOG.error("Failed to get object from cache, key: '{}'", (Object)cacheKey, (Object)ex);
            throw new IOException("Cache Provider operation failed", ex);
        }
        this.LOG.debug("Read entry at {} for context={}, key={}, value={}", new Object[]{cacheKey, context, key, record});
        if (record == null) {
            return null;
        }
        return record;
    }

    @Nonnull
    public Pair<Long, StorageRecord> read(@Nonnull String context, @Nonnull String key, long version) throws IOException {
        this.LOG.debug("GluuStorage: read (key and version)");
        Constraint.isGreaterThan((long)0L, (long)version, (String)"Version must be positive");
        StorageRecord record = this.read(context, key);
        if (record == null) {
            return new Pair();
        }
        Pair result = new Pair((Object)record.getVersion(), null);
        if (version != record.getVersion()) {
            result.setSecond((Object)record);
        } else {
            this.LOG.debug("Record has invalid version context={}, key={}, version={}", new Object[]{context, key, version});
        }
        return result;
    }

    private Long doUpdate(Long version, String context, String key, String value, Long expiration) throws IOException {
        this.LOG.debug("GluuStorage: doUpdate");
        Constraint.isNotNull((Object)StringSupport.trimOrNull((String)context), (String)"Context cannot be null or empty");
        Constraint.isNotNull((Object)StringSupport.trimOrNull((String)key), (String)"Key cannot be null or empty");
        String namespace = this.lookupNamespace(context);
        if (namespace == null) {
            this.LOG.debug("Namespace for context {} does not exist", (Object)context);
            return null;
        }
        String cacheKey = this.memcachedKey(namespace, key);
        this.LOG.debug("Updating entry at {} for context={}, key={}, exp={}, version={}", new Object[]{cacheKey, context, key, expiration, version});
        VersionMutableStorageRecord record = (VersionMutableStorageRecord)this.read(context, key);
        if (record == null) {
            return null;
        }
        if (version != null && version.longValue() != record.getVersion()) {
            throw new VersionMismatchWrapperException((Throwable)new VersionMismatchException());
        }
        if (value != null) {
            record.setValue(value);
            record.incrementVersion();
        }
        record.setExpiration(expiration);
        int expiry = record.getExpiry();
        Constraint.isGreaterThan((int)-1, (int)expiry, (String)"Expiration must be null or positive");
        try {
            int ttl = this.getSystemExpiration(record.getExpiration());
            this.cacheProvider.put(ttl, cacheKey, (Object)record);
        }
        catch (Exception ex) {
            this.LOG.error("Failed to update object in cache, key: '{}'", (Object)cacheKey, (Object)ex);
            return null;
        }
        return record.getVersion();
    }

    public boolean update(@Nonnull String context, @Nonnull String key, @Nonnull String value, @Nullable Long expiration) throws IOException {
        this.LOG.debug("GluuStorage: update");
        Constraint.isNotNull((Object)StringSupport.trimOrNull((String)value), (String)"Value cannot be null or empty");
        return this.doUpdate(null, context, key, value, expiration) != null;
    }

    @Nullable
    public Long updateWithVersion(long version, @Nonnull String context, @Nonnull String key, @Nonnull String value, @Nullable Long expiration) throws IOException, VersionMismatchException {
        this.LOG.debug("GluuStorage: updateWithVersion");
        try {
            Constraint.isNotNull((Object)StringSupport.trimOrNull((String)value), (String)"Value cannot be null or empty");
            return this.doUpdate(version, context, key, value, expiration);
        }
        catch (VersionMismatchWrapperException ex) {
            throw (VersionMismatchException)ex.getCause();
        }
    }

    public boolean updateExpiration(@Nonnull String context, @Nonnull String key, @Nullable Long expiration) throws IOException {
        this.LOG.debug("GluuStorage: updateExpiration");
        Constraint.isGreaterThan((long)-1L, (long)expiration, (String)"Expiration must be null or positive");
        return this.doUpdate(null, context, key, null, expiration) != null;
    }

    private boolean doDelete(String context, String key, Long version) throws IOException {
        this.LOG.debug("GluuStorage: doDelete");
        Constraint.isNotNull((Object)StringSupport.trimOrNull((String)context), (String)"Context cannot be null or empty");
        Constraint.isNotNull((Object)StringSupport.trimOrNull((String)key), (String)"Key cannot be null or empty");
        String namespace = this.lookupNamespace(context);
        if (namespace == null) {
            this.LOG.debug("Namespace for context {} does not exist", (Object)context);
            return false;
        }
        String cacheKey = this.memcachedKey(namespace, key);
        this.LOG.debug("Deleting entry at {} for context={}, key={}, version={}", new Object[]{cacheKey, context, key, version});
        if (version != null) {
            VersionMutableStorageRecord record = (VersionMutableStorageRecord)this.read(context, key);
            if (record == null) {
                return false;
            }
            if (version != null && record.getVersion() != version.longValue()) {
                throw new VersionMismatchWrapperException((Throwable)new VersionMismatchException());
            }
        }
        try {
            this.cacheProvider.remove(cacheKey);
        }
        catch (Exception ex) {
            this.LOG.error("Failed to remove object from cache, key: '{}'", (Object)cacheKey, (Object)ex);
            return false;
        }
        return true;
    }

    public boolean delete(@Nonnull String context, @Nonnull String key) throws IOException {
        this.LOG.debug("GluuStorage: delete");
        return this.doDelete(context, key, null);
    }

    public boolean deleteWithVersion(long version, @Nonnull String context, @Nonnull String key) throws IOException, VersionMismatchException {
        this.LOG.debug("GluuStorage: deleteWithVersion");
        try {
            return this.doDelete(context, key, version);
        }
        catch (VersionMismatchWrapperException e) {
            throw (VersionMismatchException)e.getCause();
        }
    }

    public void reap(@Nonnull String context) throws IOException {
        this.LOG.debug("GluuStorage: reap");
    }

    public void updateContextExpiration(@Nonnull String context, @Nullable Long expiration) throws IOException {
        this.LOG.debug("GluuStorage: updateContextExpiration");
        Constraint.isNotNull((Object)StringSupport.trimOrNull((String)context), (String)"Context cannot be null or empty");
        int expiry = VersionMutableStorageRecord.expiry(expiration);
        Constraint.isGreaterThan((int)-1, (int)expiry, (String)"Expiration must be null or positive");
        throw new UnsupportedOperationException("updateContextExpiration is not supported");
    }

    public void deleteContext(@Nonnull String context) throws IOException {
        this.LOG.debug("GluuStorage: deleteContext");
        Constraint.isNotNull((Object)StringSupport.trimOrNull((String)context), (String)"Context cannot be null or empty");
        String namespace = this.lookupNamespace(context);
        if (namespace == null) {
            this.LOG.debug("Namespace for context {} does not exist. Context values effectively deleted.", (Object)context);
            return;
        }
        try {
            this.cacheProvider.remove(namespace);
            this.cacheProvider.remove(context);
        }
        catch (Exception ex) {
            this.LOG.error("Failed to delete context {} from cache ", (Object)context, (Object)ex);
        }
    }

    public boolean isServerSide() {
        this.LOG.debug("GluuStorage: isServerSide");
        return true;
    }

    public boolean isClustered() {
        this.LOG.debug("GluuStorage: isClustered");
        return true;
    }

    protected String lookupNamespace(String context) throws IOException {
        this.LOG.debug("GluuStorage: lookupNamespace");
        try {
            return (String)this.cacheProvider.get(this.memcachedKey(context));
        }
        catch (Exception ex) {
            throw new IOException("Memcached operation failed", ex);
        }
    }

    protected String createNamespace(String context) throws IOException {
        this.LOG.debug("GluuStorage: createNamespace");
        int maxIterations = 10;
        String namespace = null;
        boolean success = false;
        while (!success && maxIterations-- >= 0) {
            namespace = CodecUtil.hex((byte[])ByteUtil.toBytes((long)System.currentTimeMillis()));
            try {
                boolean foundNamespace = this.cacheProvider.hasKey(namespace);
                if (foundNamespace) continue;
                this.cacheProvider.put((int)this.contextExpiration.getSeconds(), namespace, (Object)context);
                success = true;
            }
            catch (Exception foundNamespace) {}
        }
        if (!success) {
            throw new IllegalStateException("Failed to create namespace for context " + context);
        }
        try {
            this.cacheProvider.put((int)this.contextExpiration.getSeconds(), this.memcachedKey(context), (Object)namespace);
        }
        catch (Exception ex) {
            throw new IllegalStateException(context + " already exists");
        }
        return namespace;
    }

    private String memcachedKey(String ... parts) {
        String key;
        if (parts.length > 0) {
            StringBuilder sb = new StringBuilder();
            int i = 0;
            for (String part : parts) {
                if (i++ > 0) {
                    sb.append(':');
                }
                sb.append(part);
            }
            key = sb.toString();
        } else {
            key = parts[0];
        }
        if (key.length() > 250) {
            return CodecUtil.hex((byte[])HashUtil.sha512((Object[])new Object[]{key}));
        }
        return key;
    }

    public TimerTask getCleanupTask() {
        return new TimerTask(){

            @Override
            public void run() {
                Long now = System.currentTimeMillis();
                GluuStorageService.this.LOG.debug("Running cleanup task at {}", (Object)now);
                try {
                    GluuStorageService.this.cacheProvider.cleanup(new Date(now));
                }
                catch (Exception e) {
                    GluuStorageService.this.LOG.error("Error running cleanup task for {}", (Object)now, (Object)e);
                }
                GluuStorageService.this.LOG.debug("Finished cleanup task for {}", (Object)now);
            }
        };
    }

    public void setContextExpiration(@NonNegative Duration contextExpiration) {
        this.LOG.debug("GluuStorage: setContextExpiration");
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException((InitializableComponent)this);
        Constraint.isNotNull((Object)contextExpiration, (String)"Context expiration interval cannot be null");
        Constraint.isFalse((boolean)contextExpiration.isNegative(), (String)"Context expiration interval must be greater than or equal to zero");
        this.contextExpiration = contextExpiration;
    }

    private int getSystemExpiration(Long expiration) {
        return (int)(expiration == null || expiration == 0L ? 0L : (expiration - System.currentTimeMillis()) / 1000L);
    }

    public static class VersionMismatchWrapperException
    extends RuntimeException {
        public VersionMismatchWrapperException(Throwable cause) {
            super(cause);
        }
    }
}

