package org.gluu.oxd.license.validator;

import net.nicholaswilliams.java.licensing.FeatureRestriction;
import net.nicholaswilliams.java.licensing.SignedLicense;
import net.nicholaswilliams.java.licensing.exception.InsecureEnvironmentException;

import java.io.FileDescriptor;
import java.lang.reflect.Member;
import java.net.InetAddress;
import java.security.Permission;

final class OxLicenseSecurityManager extends SecurityManager {
    private static OxLicenseSecurityManager instance;

    private static final String FEATURE_RESTRICTION = FeatureRestriction.class.getCanonicalName();

    private static final String SIGNED_LICENSE = SignedLicense.class.getCanonicalName();

    private static final RuntimePermission CHECK_MEMBER_ACCESS_PERMISSION =
            new RuntimePermission("accessDeclaredMembers");

    private static final RuntimePermission SET_SECURITY_MANAGER_PERMISSION =
            new RuntimePermission("setSecurityManager");

    static {
        SecurityManager manager = System.getSecurityManager();
        if (manager == null) {
            // install the security manager
            OxLicenseSecurityManager.installSecurityManagerWithParent(manager);
        } else if (!manager.getClass().equals(OxLicenseSecurityManager.class)) {
            if (!OxLicenseSecurityManager.securityManagerIsSuitableReplacement(manager)) {
                // if it's not a suitable replacement, reset the security manager
                OxLicenseSecurityManager.installSecurityManagerWithParent(manager);
            }
        }
    }

    protected static boolean securityManagerIsSuitableReplacement(SecurityManager securityManager) {
        if (securityManager == null)
            throw new IllegalArgumentException("Parameter securityManager cannot be null!");

        // Make sure we can't call java.lang.System#setSecurityManager()
        try {
            securityManager.checkPermission(OxLicenseSecurityManager.SET_SECURITY_MANAGER_PERMISSION);
            return false;
        } catch (SecurityException ignore) {
            // this is a good thing
        }

        return true;
    }

    private static void installSecurityManagerWithParent(SecurityManager parent) {
        try {
            // install the security manager
            OxLicenseSecurityManager.instance = new OxLicenseSecurityManager(parent);
            System.setSecurityManager(OxLicenseSecurityManager.instance);
        } catch (SecurityException e) {
            // since we can't install the security manager, indicate that the environment is insecure
            throw new InsecureEnvironmentException(e);
        }
    }

    private final SecurityManager next;

    private OxLicenseSecurityManager(SecurityManager next) {
        super();

        this.next = next;
    }

    //@Override
    @SuppressWarnings("deprecation")
    public void checkMemberAccess(Class<?> reflectionClass, int memberAccessType) {
        if (reflectionClass == null)
            throw new IllegalArgumentException("Parameter reflectionClass cannot be null.");

        //this.inCheck = true;

        try {
            if (memberAccessType != Member.PUBLIC) {
                /*
                 * We check class canonical name, not class object, for equivalency. This is because
                 * (SomeClass.class == SomeClass.class) evaluates to false when the classes are loaded by two different
                 * ClassLoaders and (SomeClass.class.equals(SomeClass.class)) evaluates to false when the classes are
                 * loaded by two different ClassLoaders. Only their class canonical names are guaranteed to be the same.
                 */
                Package packageObject = reflectionClass.getPackage();
                if (packageObject != null &&
                        packageObject.getName().startsWith("net.nicholaswilliams.java.licensing") &&
                        !reflectionClass.getCanonicalName().equals(OxLicenseSecurityManager.FEATURE_RESTRICTION) &&
                        !reflectionClass.getCanonicalName().equals(OxLicenseSecurityManager.SIGNED_LICENSE)
                ) {
                    throw new SecurityException("Reflection access to non-public members of LicenseManager class [" +
                            reflectionClass.getSimpleName() + "] prohibited.");
                }

                if (reflectionClass == java.lang.Class.class || reflectionClass == java.lang.System.class) {
                    Class stack[] = getClassContext();
                    if (stack.length < 4 || !stack[3].getPackage().getName().startsWith("java."))
                        throw new SecurityException("Reflection access to non-public members of java.lang.Class " +
                                "and java.lang.System prohibited.");
                }

                if (this.next != null) {
                    /*
                     * Per Java SE 6 documentation for java.lang.SecurityManager#checkMemberAccess: If this method is
                     * overridden, then a call to super.checkMemberAccess cannot be made, as the default implementation
                     * of checkMemberAccess relies on the code being checked being at a stack depth of 4. So, we
                     * copy-and-paste the implementation from Java SE 6.
                     *
                     * this.next.checkMemberAccess(reflectionClass, memberAccessType);
                     *
                     * Copyright 1994-2007 Sun Microsystems, Inc.  All Rights Reserved.
                     * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
                     *
                     * This code is free software; you can redistribute it and/or modify it
                     * under the terms of the GNU General Public License version 2 only, as
                     * published by the Free Software Foundation.  Sun designates this
                     * particular file as subject to the "Classpath" exception as provided
                     * by Sun in the LICENSE file that accompanied this code.
                     *
                     * This code is distributed in the hope that it will be useful, but WITHOUT
                     * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
                     * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
                     * version 2 for more details (a copy is included in the LICENSE file that
                     * accompanied this code).
                     *
                     * You should have received a copy of the GNU General Public License version
                     * 2 along with this work; if not, write to the Free Software Foundation,
                     * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
                     *
                     * Please contact Sun Microsystems, Inc., 4150 Network Circle, Santa Clara,
                     * CA 95054 USA or visit www.sun.com if you need additional information or
                     * have any questions.
                     */

                    Class stack[] = getClassContext();
                    /*
                     * stack depth of 4 should be the caller of one of the
                     * methods in java.lang.Class that invoke checkMember
                     * access. The stack should look like:
                     *
                     * someCaller                        [3]
                     * java.lang.Class.someReflectionAPI [2]
                     * java.lang.Class.checkMemberAccess [1]
                     * SecurityManager.checkMemberAccess [0]
                     *
                     */
                    if ((stack.length < 4) || (stack[3].getClassLoader() != reflectionClass.getClassLoader())) {
                        this.checkPermission(OxLicenseSecurityManager.CHECK_MEMBER_ACCESS_PERMISSION);
                    }
                }
            }
        } finally {
            //this.inCheck = false;
        }
    }

    @Override
    @SuppressWarnings("deprecation")
    public void checkPermission(Permission permission) {
        //this.inCheck = true;
        try {
            //if(permission.getName().equals("setSecurityManager"))
            //    throw new SecurityException("Setting a SecurityManager other than the LicenseSecurityManager is prohibited.");

            if (this.next != null)
                this.next.checkPermission(permission);
        } finally {
            //this.inCheck = false;
        }
    }

    @Override
    public void checkPackageAccess(String packageName) {
        if (this.next != null)
            this.next.checkPackageAccess(packageName);
    }

    @Override
    public void checkPermission(Permission permission, Object object) {
        if (this.next != null)
            this.next.checkPermission(permission, object);
    }

    @Override
    public void checkCreateClassLoader() {
        if (this.next != null)
            this.next.checkCreateClassLoader();
    }

    @Override
    public void checkAccess(Thread thread) {
        if (this.next != null)
            this.next.checkAccess(thread);
    }

    @Override
    public void checkAccess(ThreadGroup threadGroup) {
        if (this.next != null)
            this.next.checkAccess(threadGroup);
    }

    @Override
    public void checkExit(int i) {
        if (this.next != null)
            this.next.checkExit(i);
    }

    @Override
    public void checkExec(String s) {
        if (this.next != null)
            this.next.checkExec(s);
    }

    @Override
    public void checkLink(String s) {
        if (this.next != null)
            this.next.checkLink(s);
    }

    @Override
    public void checkRead(FileDescriptor fileDescriptor) {
        if (this.next != null)
            this.next.checkRead(fileDescriptor);
    }

    @Override
    public void checkRead(String s) {
        if (this.next != null)
            this.next.checkRead(s);
    }

    @Override
    public void checkRead(String s, Object o) {
        if (this.next != null)
            this.next.checkRead(s);
    }

    @Override
    public void checkWrite(FileDescriptor fileDescriptor) {
        if (this.next != null)
            this.next.checkWrite(fileDescriptor);
    }

    @Override
    public void checkWrite(String s) {
        if (this.next != null)
            this.next.checkWrite(s);
    }

    @Override
    public void checkDelete(String s) {
        if (this.next != null)
            this.next.checkDelete(s);
    }

    @Override
    public void checkConnect(String s, int i) {
        if (this.next != null)
            this.next.checkConnect(s, i);
    }

    @Override
    public void checkConnect(String s, int i, Object o) {
        if (this.next != null)
            this.next.checkConnect(s, i, o);
    }

    @Override
    public void checkListen(int i) {
        if (this.next != null)
            this.next.checkListen(i);
    }

    @Override
    public void checkAccept(String s, int i) {
        if (this.next != null)
            this.next.checkAccept(s, i);
    }

    @Override
    public void checkMulticast(InetAddress inetAddress) {
        if (this.next != null)
            this.next.checkMulticast(inetAddress);
    }

    @Override
    @SuppressWarnings("deprecation")
    public void checkMulticast(InetAddress inetAddress, byte b) {
        if (this.next != null)
            this.next.checkMulticast(inetAddress, b);
    }

    @Override
    public void checkPropertiesAccess() {
        if (this.next != null)
            this.next.checkPropertiesAccess();
    }

    @Override
    public void checkPropertyAccess(String s) {
        if (this.next != null)
            this.next.checkPropertyAccess(s);
    }

    @Override
    public void checkPrintJobAccess() {
        if (this.next != null)
            this.next.checkPrintJobAccess();
    }

    /*@Override
    public void checkSystemClipboardAccess()
    {
        if(this.next != null)
            this.next.checkSystemClipboardAccess();
    }

    @Override
    public void checkAwtEventQueueAccess()
    {
        if(this.next != null)
            this.next.checkAwtEventQueueAccess();
    }


    @Override
    public boolean checkTopLevelWindow(Object window)
    {
        return this.next == null || this.next.checkTopLevelWindow(window);
    }*/

    @Override
    public void checkPackageDefinition(String s) {
        if (this.next != null)
            this.next.checkPackageDefinition(s);
    }

    @Override
    public void checkSetFactory() {
        if (this.next != null)
            this.next.checkSetFactory();
    }

    @Override
    public void checkSecurityAccess(String s) {
        if (this.next != null)
            this.next.checkSecurityAccess(s);
    }

    @Override
    public ThreadGroup getThreadGroup() {
        return this.next != null ? this.next.getThreadGroup() : super.getThreadGroup();
    }

    @Override
    public Object getSecurityContext() {
        return this.next != null ? this.next.getSecurityContext() : super.getSecurityContext();
    }

    protected static OxLicenseSecurityManager getInstance() {
        return OxLicenseSecurityManager.instance;
    }
}
