package org.gluu.oxd.licenser.server.ws;

import com.google.inject.Inject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.gluu.oxd.license.client.js.StatisticUpdateRequest;
import org.gluu.oxd.licenser.server.service.ErrorService;
import org.gluu.oxd.licenser.server.service.LicenseGeneratedStatService;
import org.gluu.oxd.licenser.server.service.StatisticService;
import org.gluu.oxd.licenser.server.service.StatisticUpdator;
import org.gluu.oxd.licenser.server.service.ValidationService;

import javax.ws.rs.Consumes;
import javax.ws.rs.GET;
import javax.ws.rs.POST;
import javax.ws.rs.Path;
import javax.ws.rs.Produces;
import javax.ws.rs.QueryParam;
import javax.ws.rs.WebApplicationException;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.Response;

/**
 * @author Yuriy Zabrovarnyy
 * @version 0.9, 06/09/2016
 */

@Path("/rest/statistic")
public class StatisticWS {

    private static final Logger LOG = LoggerFactory.getLogger(StatisticWS.class);

    @Inject
    ValidationService validationService;
    @Inject
    ErrorService errorService;
    @Inject
    StatisticService statisticService;
    @Inject
    StatisticUpdator statisticUpdator;
    @Inject
    LicenseGeneratedStatService licenseGeneratedStatService;

    @POST
    @Path("/update")
    @Produces({MediaType.APPLICATION_JSON})
    @Consumes({MediaType.APPLICATION_JSON})
    public Response update(StatisticUpdateRequest request) {
        try {
            statisticUpdator.update(request);
            return Response.ok().build();
        } catch (Exception e) {
            LOG.error(e.getMessage(), e);
            throw new WebApplicationException(ErrorService.response(e.getMessage()));
        }
    }

    @GET
    @Path("/monthly")
    @Produces({MediaType.APPLICATION_JSON})
    public Response licenseGeneratedMonthly(@QueryParam("licenseId") String licenseId) {
        LOG.debug("License generated monthly request, licenseId: " + licenseId);

        return okWithCors(licenseGeneratedStatService.monthlyStatistic(licenseId));
    }

    @GET
    @Path("/lastHours")
    @Produces({MediaType.APPLICATION_JSON})
    public Response licenseGeneratedLastHours(@QueryParam("licenseId") String licenseId,
                                              @QueryParam("hours") int hours) {
        LOG.debug("License generated last hours request, licenseId: " + licenseId + ", hours: " + hours);
        return okWithCors(licenseGeneratedStatService.lastHours(licenseId, hours));
    }

    @GET
    @Path("/client/monthly")
    @Produces({MediaType.APPLICATION_JSON})
    public Response clientMonthly(@QueryParam("licenseId") String licenseId) {
        LOG.debug("Client monthly request, licenseId: " + licenseId);

        return okWithCors(statisticService.clientMonthlyStatistic(licenseId));
    }

    @GET
    @Path("/client/lastHours")
    @Produces({MediaType.APPLICATION_JSON})
    public Response clientEventLastHours(@QueryParam("licenseId") String licenseId,
                                         @QueryParam("hours") int hours) {
        LOG.debug("Client event last hours request, licenseId: " + licenseId + ", hours: " + hours);
        return okWithCors(statisticService.lastHoursLicenseIdStatistic(licenseId, hours));
    }

    private Response okWithCors(String entity) {
        Response.ResponseBuilder builder = Response.ok().entity(entity);
        builder.header("Access-Control-Allow-Origin", "*");
        builder.header("Access-Control-Allow-Methods", "GET");
        return builder.build();
    }
}
