package org.gluu.oxd.licenser.server.service;

import com.google.common.base.Preconditions;
import com.google.common.base.Predicate;
import com.google.common.base.Strings;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.inject.Inject;
import com.unboundid.util.StaticUtils;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.gluu.oxd.license.client.js.Configuration;
import org.gluu.oxd.license.client.js.HasCreationDate;
import org.gluu.oxd.license.client.js.event.LdapClientEvent;
import org.gluu.oxd.license.client.js.event.LdapLicenseGeneratedEvent;
import org.gluu.oxd.licenser.server.Utils;
import org.gluu.oxd.licenser.server.ldap.LdapStructure;
import org.gluu.oxd.licenser.server.stat.ClientIdKey;
import org.gluu.oxd.licenser.server.stat.MacKey;
import org.gluu.oxd.licenser.server.stat.Stat;
import org.gluu.persist.PersistenceEntryManager;
import org.gluu.search.filter.Filter;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.util.*;

/**
 * @author Yuriy Zabrovarnyy
 */

public class StatisticService {

    private static final Logger LOG = LoggerFactory.getLogger(StatisticService.class);

    @Inject
    PersistenceEntryManager ldapEntryManager;
    @Inject
    Configuration conf;
    @Inject
    LdapStructure ldapStructure;
    @Inject
    ValidationService validationService;

    public void merge(LdapClientEvent entity) {
        try {
            ldapEntryManager.merge(entity);
        } catch (Exception e) {
            LOG.error(e.getMessage(), e);
        }
    }

    public LdapLicenseGeneratedEvent saveGeneratedEvent(LdapLicenseGeneratedEvent entity, String licenseId) {
        try {
            if (Strings.isNullOrEmpty(entity.getId())) {
                entity.setId(UUID.randomUUID().toString());
            }

            if (Strings.isNullOrEmpty(entity.getDn())) {
                entity.setDn(dn(entity.getId(), licenseId));
            }

            ldapEntryManager.persist(entity);
            return entity;
        } catch (Exception e) {
            LOG.error(e.getMessage(), e);
            return null;
        }
    }

    public LdapClientEvent saveClientEvent(LdapClientEvent event, String licenseId) {
        return saveClientEvent(event, licenseId, false);
    }

    public LdapClientEvent saveClientEvent(LdapClientEvent event, String licenseId, boolean forcePersistence) {
        try {
            Preconditions.checkArgument(StringUtils.isNotBlank(licenseId), "License ID can't be blank.");
            Preconditions.checkArgument(StringUtils.isNotBlank(event.getOxdId()), "oxdid can't be blank.");

            // persist if no entries with given oxdId for last 24 hours
            final boolean shouldPersist = getByLicenseIdAndOxdIdInLastHours(licenseId, event.getOxdId(), 24).isEmpty();
            if (shouldPersist || forcePersistence) {
                if (event.getCreationDate() == null) {
                    event.setCreationDate(new Date());
                }

                setDnIfEmpty(event, licenseId);
                ldapEntryManager.persist(event);
                LOG.trace("Persisted event: " + event);
                return event;
            } else {
                LOG.trace("Skip persistence of client event, licenseId: " + licenseId + ", event: " + event);
            }
        } catch (Exception e) {
            LOG.error(e.getMessage(), e);
        }
        return null;
    }

    private void setDnIfEmpty(LdapClientEvent entity, String licenseId) {
        if (Strings.isNullOrEmpty(entity.getId())) {
            entity.setId(UUID.randomUUID().toString());
        }

        if (Strings.isNullOrEmpty(entity.getDn())) {
            entity.setDn(dn(entity.getId(), licenseId));
        }
    }

    private String dn(String id, String licenseId) {
        return "uniqueIdentifier=" + id + "," + ldapStructure.getStatisticBaseDn(licenseId);
    }

    public LdapClientEvent get(String dn) {
        return ldapEntryManager.find(LdapClientEvent.class, dn);
    }

    public List<LdapClientEvent> getAll(String licenseId) {
        try {
            Filter filter = Filter.createPresenceFilter("uniqueIdentifier");
            return ldapEntryManager.findEntries(ldapStructure.getStatisticBaseDn(licenseId), LdapClientEvent.class, filter);
        } catch (Exception e) {
            LOG.error(e.getMessage(), e);
        }
        return Collections.emptyList();
    }

    public List<LdapLicenseGeneratedEvent> getAllLicenseGenerated(String licenseId) {
            try {
                Filter filter = Filter.createPresenceFilter("uniqueIdentifier");
                return ldapEntryManager.findEntries(ldapStructure.getStatisticBaseDn(licenseId), LdapLicenseGeneratedEvent.class, filter);
            } catch (Exception e) {
                LOG.error(e.getMessage(), e);
            }
            return Collections.emptyList();
        }

    public List<LdapClientEvent> getByLicenseIdAndOxdId(String licenseId, String oxdId) {
        try {
            Filter filter = Filter.createEqualityFilter("oxdId", oxdId);
            final List<LdapClientEvent> entries = ldapEntryManager.findEntries(ldapStructure.getStatisticBaseDn(licenseId), LdapClientEvent.class, filter);
            return entries != null ? entries : Collections.<LdapClientEvent>emptyList();
        } catch (Exception e) {
            LOG.error(e.getMessage(), e);
        }
        return Collections.emptyList();
    }

    public List<LdapClientEvent> getByLicenseIdAndOxdIdInLastHours(String licenseId, String oxdId, int hours) {
        final List<LdapClientEvent> all = getByLicenseIdAndOxdId(licenseId, oxdId);
        return filter(all, StatisticService.<LdapClientEvent>timePredicate(hours));
    }

    public static <T> List<T> filter(List<T> all, Predicate<T> predicate) {
        List<T> filtered = Lists.newArrayList();
        if (all != null && !all.isEmpty()) {
            for (T item : all) {
                if (predicate.apply(item)) {
                    filtered.add(item);
                }
            }
        }
        LOG.trace("ALL size: " + (all != null ? all.size() : 0) + ", filtered size: " + filtered.size());
        return filtered;
    }

    public List<LdapClientEvent> getForPeriod(String licenseId, Date from, Date to) {
        try {
            String filter = String.format("&(uniqueIdentifier=*)(oxCreationDate>=%s)(oxCreationDate<=%s)",
                    StaticUtils.encodeGeneralizedTime(from),
                    StaticUtils.encodeGeneralizedTime(to));
            LOG.trace("period filter: " + filter);
            return ldapEntryManager.findEntries(ldapStructure.getStatisticBaseDn(licenseId), LdapClientEvent.class, Filter.create(filter));
        } catch (Exception e) {
            LOG.error(e.getMessage(), e);
        }
        return Collections.emptyList();
    }

    public static <T extends HasCreationDate> Predicate<T> timePredicate(int hours) {
        ValidationService.validateHours(hours);

        final Calendar inPast = Calendar.getInstance();
        inPast.add(Calendar.HOUR, -hours);

        return new Predicate<T>() {
            @Override
            public boolean apply(T input) {
                //LOG.trace("Result: " + input.getCreationDate().before(inPast.getTime()) + ", creation: " + input.getCreationDate() + ", inPast: " + inPast.getTime());
                return input.getCreationDate().after(inPast.getTime());
            }
        };
    }

    public String lastHoursLicenseIdStatistic(String licenseId, int hours) {
        validationService.getLicenseId(licenseId);

        try {
            List<LdapClientEvent> filtered = filter(getAll(licenseId), StatisticService.<LdapClientEvent>timePredicate(hours));

            Map<String, Integer> macAddressToCounter = new HashMap<>();
            for (LdapClientEvent item : filtered) {
                Integer counter = macAddressToCounter.get(item.getMacAddress());
                if (counter == null) {
                    counter = 1;
                } else {
                    counter++;
                }

                //LOG.trace("macAddressToCounter put, mac: " + item.getMacAddress() + ", counter: " + counter);
                macAddressToCounter.put(item.getMacAddress(), counter);
            }

            LOG.trace("macAddressToCounter size: " + macAddressToCounter.size());

            JSONObject lastHoursStatistic = new JSONObject();

            for (Map.Entry<String, Integer> entity : macAddressToCounter.entrySet()) {
//                JSONObject stat = new JSONObject();
//                stat.put("mac_address", entity.getKey());
//                stat.put("count", entity.getValue());

                lastHoursStatistic.put(entity.getKey(), entity.getValue());
            }

            JSONObject wrapper = new JSONObject();
            wrapper.put("statistic", lastHoursStatistic);
            wrapper.put("total_generated_licenses", filtered.size());
            return wrapper.toString(2);
        } catch (Exception e) {
            LOG.error("Failed to construct statistic for license_id: " + licenseId, e);
            return "{\"error\":\"Failed to construct statistic for license_id: " + licenseId + "\"}";
        }
    }

    public String clientMonthlyStatistic(String licenseId) {
        validationService.getLicenseId(licenseId);

        try {
            Map<String, Stat> map = constructMonthlyMap(licenseId);

            JSONObject monthlyStatistic = new JSONObject();

            for (Map.Entry<String, Stat> entity : map.entrySet()) {

                JSONObject stat = new JSONObject();
                //stat.put("mac_address", MacKey.countMap(entity.getValue().getMacKeys()));
                stat.put("local_clients", clientsJsonArray(entity.getValue().getLocalClientIdKeys()));
                stat.put("web_clients", clientsJsonArray(entity.getValue().getWebClientIdKeys()));

                monthlyStatistic.put(entity.getKey(), stat);
            }

            JSONObject wrapper = new JSONObject();
            wrapper.put("monthly_statistic", monthlyStatistic);
            return wrapper.toString(2);
        } catch (Exception e) {
            LOG.error("Failed to construct statistic for license_id: " + licenseId, e);
            return "{\"error\":\"Failed to construct statistic for license_id: " + licenseId + "\"}";
        }
    }

    private JSONArray clientsJsonArray(Set<ClientIdKey> clients) throws JSONException {
        JSONArray array = new JSONArray();

        final Map<String, Integer> countMap = ClientIdKey.countMap(clients);
        for (Map.Entry<String, Integer> entry : countMap.entrySet()) {
            final ClientIdKey clientIdKey = ClientIdKey.byClientId(clients, entry.getKey());
            Set<ClientIdKey> recordsOfClient = ClientIdKey.filterByClientId(clients, entry.getKey());

            JSONObject object = new JSONObject();
            object.put("client_id", entry.getKey());
            object.put("oxd_id", clientIdKey.getOxdId());
            object.put("client_name", clientIdKey.getClientName());
            object.put("count", entry.getValue());
            object.put("mac_address", clientIdKey.getMacAddress());
            object.put("programming_language", clientIdKey.getMetadata() != null ? clientIdKey.getMetadata().getProgrammingLanguage() : "");
            object.put("app_name", clientIdKey.getMetadata() != null ? clientIdKey.getMetadata().getAppName() : "");
            object.put("app_version", clientIdKey.getMetadata() != null ? clientIdKey.getMetadata().getAppVersion() : "");
            object.put("server_name", clientIdKey.getMetadata() != null ? clientIdKey.getMetadata().getData().get("server_name") : "");
            object.put("active_dates", activeDatesArray(recordsOfClient));
            object.put("first_used_at", firstUsedAt(recordsOfClient));

            array.put(object);
        }

        return array;
    }

    private static String firstUsedAt(Set<ClientIdKey> recordsOfClient) {
        List<Date> dates = new ArrayList<>();
        for (ClientIdKey record : recordsOfClient) {
            dates.add(record.getCreationDate());
        }
        if (dates.isEmpty()) {
            return "none";
        } else {
            return Utils.yearAndMonthAndDayOfMonth(Collections.min(dates));
        }
    }

    private static Set<String> activeDatesArray(Set<ClientIdKey> clients) {
        Set<String> result = new HashSet<>();
        if (clients != null && !clients.isEmpty()) {
            for (ClientIdKey key : clients) {
                result.add(key.getCreationDateAsYY_MM_DD());
            }
        }
        return result;
    }

    private Map<String, Stat> constructMonthlyMap(String licenseId) {
        List<LdapClientEvent> entities = getAll(licenseId);

        Map<String, Stat> map = Maps.newTreeMap();

        for (LdapClientEvent s : entities) {
            String key = Utils.yearAndMonth(s.getCreationDate());

            Stat stat = map.get(key);
            if (stat == null) {  // init
                stat = new Stat();
                map.put(key, stat);
            }

            stat.getMacKeys().add(new MacKey(s));
            if (s.getClientLocal()) {
                stat.getLocalClientIdKeys().add(new ClientIdKey(s));
            } else {
                stat.getWebClientIdKeys().add(new ClientIdKey(s));
            }
        }
        return map;
    }
}
