package org.gluu.oxd.licenser.server.stat;

import com.google.common.base.Strings;
import com.google.common.collect.Maps;
import org.gluu.oxd.license.client.Jackson;
import org.gluu.oxd.license.client.js.AppMetadata;
import org.gluu.oxd.license.client.js.event.LdapClientEvent;
import org.gluu.oxd.licenser.server.Utils;

import java.io.IOException;
import java.util.*;

/**
 * @author Yuriy Zabrovarnyy
 */

public class ClientIdKey {

    private final String creationDateAsYY_MM_DD;
    private final Date creationDate;
    private final String clientId;
    private final String oxdId;
    private final String clientName;
    private final String macAddress;
    private final AppMetadata metadata;

    public ClientIdKey(LdapClientEvent stat) {
        this.creationDate = stat.getCreationDate();
        this.creationDateAsYY_MM_DD = Utils.yearAndMonthAndDayOfMonth(stat.getCreationDate());
        this.clientId = stat.getClientId();
        this.clientName = stat.getClientName();
        this.oxdId = stat.getOxdId();
        this.macAddress = stat.getMacAddress();
        this.metadata = parseSilently(stat.getAppMetadata());
    }

    private static AppMetadata parseSilently(String json) {
        try {
            return Jackson.createJsonMapper().readValue(json, AppMetadata.class);
        } catch (IOException e) {
            return null;
        }
    }

    public AppMetadata getMetadata() {
        return metadata;
    }

    public Date getCreationDate() {
        return creationDate;
    }

    public String getCreationDateAsYY_MM_DD() {
        return creationDateAsYY_MM_DD;
    }

    public String getClientId() {
        return clientId;
    }

    public String getOxdId() {
        return oxdId;
    }

    public String getClientName() {
        return clientName;
    }

    public String getMacAddress() {
        return macAddress;
    }

    public static Map<String, Integer> countMap(Set<ClientIdKey> value) {
        Map<String, Integer> map = Maps.newHashMap();
        for (ClientIdKey v : value) {
            if (Strings.isNullOrEmpty(v.getClientId())) {
                continue;
            }

            Integer counter = map.get(v.getClientId());
            if (counter == null) {
                map.put(v.getClientId(), 1);
            } else {
                counter++;
                map.put(v.getClientId(), counter);
            }
        }
        return map;
    }

    public static ClientIdKey byClientId(Collection<ClientIdKey> collection, String clientId) {
        for (ClientIdKey k : collection) {
            if (k.getClientId().equals(clientId)) {
                return k;
            }
        }
        return null;
    }

    public static Set<ClientIdKey> filterByClientId(Collection<ClientIdKey> collection, String clientId) {
        Set<ClientIdKey> result = new HashSet<>();
        for (ClientIdKey k : collection) {
            if (k.getClientId().equals(clientId)) {
                result.add(k);
            }
        }
        return result;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;

        ClientIdKey that = (ClientIdKey) o;

        return !(clientId != null ? !clientId.equals(that.clientId) : that.clientId != null) && !(creationDateAsYY_MM_DD != null ? !creationDateAsYY_MM_DD.equals(that.creationDateAsYY_MM_DD) : that.creationDateAsYY_MM_DD != null) && !(oxdId != null ? !oxdId.equals(that.oxdId) : that.oxdId != null);
    }

    @Override
    public int hashCode() {
        int result = creationDateAsYY_MM_DD != null ? creationDateAsYY_MM_DD.hashCode() : 0;
        result = 31 * result + (clientId != null ? clientId.hashCode() : 0);
        result = 31 * result + (oxdId != null ? oxdId.hashCode() : 0);
        return result;
    }
}
